/***********************************************************************************
 * Sound Memory             (C) Ralf Stoffels 2023
 * 
 ***********************************************************************************/
//================================ Global Variables ================================
#define PITS 36           // Size of the board
#define XMAX 6            // Number of Columns of the board
#define YMAX 6            // Number or Rows of the board
#define THRESHOLD 200     // Minimum Sensor-Value-Change for Ball Detection
#define ABS_THRESHOLD 250 // To detect if a card is present or not (for Card Count)

#define DEBUG 0           // Debug-Level: 1 = All Sensor Values, 2 = Decisions

int senseArray[PITS];     // The up-to-date representation of the board as analog sensor values
int lastSenseArray[PITS]; // Stores the previous values of the sense Array
int soundArray[PITS];     // Contains the refernces to the light effects (random)
int cardNr;               // first or second card
int lastSoundNr;          // remembers the previous effect
int lastPitNr;            // previous pit;
int cardCount;            // number of cards left on the board
int soundSet = 0;         // allows differnt sound sets 0 --> sounds from 1..18; 1 --> sounds from 19..37; ... 

/*****************************************************************************************************
* Setup is run only after power up
* Each new game is initialized by the function resetGame()
******************************************************************************************************/ 
void setup() {
  pinMode(2, OUTPUT);    // Drivers for the LED of the refection switches
  pinMode(3, OUTPUT);
  pinMode(4, OUTPUT);
  pinMode(5, OUTPUT);
  pinMode(6, OUTPUT);
  pinMode(7, OUTPUT);

  pinMode(A0, INPUT);    // Receivers of the phototransistors of the reflecetion sensors
  pinMode(A1, INPUT);
  pinMode(A2, INPUT);
  pinMode(A3, INPUT);
  pinMode(A4, INPUT);
  pinMode(A5, INPUT);

  for (int i=2; i<8; i++) {
    digitalWrite(i,HIGH);      // Disable all reflection sensors
  }

  Serial.begin(9600);          // For debug
  delay(500);

  soundInit();

  int i = getChange();                // Updates the Card count
  soundSet = 0;                       // Default: Sounds 1..18
  if (cardCount == 35) soundSet = 1;   // Sounds 19..37
  if (cardCount == 34) soundSet = 2;   // Sounds 38..56

  announceSoundSet(soundSet+1);
  resetGame();
}

/****************************************************************************************************************
*                                    resetGame is executed to start a new Game
****************************************************************************************************************/
void resetGame() {  
  shuffle();                   // Distribute the effects randomly over the board#
  Serial.println("================ Effects =================");
  printAsMatrix(soundArray);  // Debug and Cheating ;-)

  //readSensors();
  //for (int i=0; i<PITS; i++) lastSenseArray[i] = senseArray[i];

  int i = getChange();        // Check if all cards on board -- otherwise announce instructions
  if (cardCount < 36) announceAllCardsOnBoard();

  do {
    int i = getChange();
  } while (cardCount < 36);

  playFanfare();              // Official game start
  
  lastSoundNr = -1;
  lastPitNr = -1;
}


/***************************************************************************************************************
 *                                                   Game Loop 
 ***************************************************************************************************************/
void loop() {
  int nr=0, soundNr;

  #if(DEBUG == 1)  
    readSensors();
    printAsMatrix(senseArray);
  #endif
  
  nr = getChange();    // Check if a ball has been added (return -1 if no change)
  if (cardCount < 1) {
    announceGameEnd();
    resetGame();
  }
  
  #if(DEBUG == 2) 
    if (nr>=0) {
      Serial.print("Changed: ");
      Serial.println(nr);
    }
  #endif

  if (nr >= 0) {
    soundNr = soundArray[nr];
    playSound(soundNr);
  }   
  delay(10);
}


/****************************************************************************************************************
 *                                        High Level Game Functions
 ****************************************************************************************************************/

//================= Fills the effectArray with random pairs ==================
void shuffle() {     
  int nr;
  
  for (int i=0; i<PITS; i++) soundArray[i]=0;   
  
  pinMode(A6,INPUT);
  randomSeed(analogRead(A6));  // random Seed by using the input noise of an analog pin
  
  for (int soundNr = 1; soundNr<=PITS/2; soundNr++) {      // assign half of the number of pits effects
    do {
       nr = random(0, PITS);
    } while (soundArray[nr]!=0);
    soundArray[nr] = soundNr;
    do {
       nr = random(0, PITS);
    } while (soundArray[nr]!=0);
    soundArray[nr] = soundNr;
  }
} 


//============ Returns the first pit that has changed ========================
int getChange() {
  int firstChange=-1;

  cardCount = 0;  // Reset CardCount
  
  readSensors();
  for (int i=PITS-1; i>=0; i--) {
    if (senseArray[i] < ABS_THRESHOLD) cardCount++;
    if (senseArray[i] > lastSenseArray[i]+THRESHOLD) {
      firstChange = i;                      
    } else {
      lastSenseArray[i] = senseArray[i];    // Update the lastSenseArray with the recent Values
    }
  }
  if (firstChange >= 0) lastSenseArray[firstChange] = senseArray[firstChange];  // only update the first field

  Serial.print("Card count = ");  
  Serial.println(cardCount);
  
  return(firstChange);
}


/****************************************************************************************************************
 *                                  Hardware access to read board status
 ****************************************************************************************************************/
void readSensors() {
  int row, col;
  int val;

  for (row = 0; row < YMAX; row++) {
    for (col = 0; col < XMAX; col++) {
       digitalWrite(row+2,LOW);
       delay(1);
       val = analogRead(col+A0);
       digitalWrite(row+2,HIGH);
       delay(1);
       senseArray[col*YMAX + row] = val;
    }
  } 
}


/**********************************************************************************************************
*                                       Debug Functions
***********************************************************************************************************/

//======== Print all detected boards in the Terminal Window ======================
void printAsArray(int a[]) {
  for (int i=0; i<PITS; i++) {
    Serial.print(a[i]);
    Serial.print(", ");
  }
  Serial.println(" ");
  Serial.println("==========================================");
}

//=========== Prints the Array with XMAX x YMAX Matrix for easier debug ===========
void printAsMatrix(int m[]) {
  int row, col;
  
  for (row = 0; row < YMAX; row++) {
    for (col = 0; col < XMAX; col++) {
       Serial.print(m[col + XMAX * row]); Serial.print("  ");
    }
    Serial.println(" ");
  }
  Serial.println("==========================================");
}
