// CNCHotwire-Fuselage
// Rumpfschneideprogramm
// Version 1.0.0
// Datum 15.Mai.14
// Versionskommentar: Erste Version zum Schneiden von Rümpfen. 
//                    Erfordert ArduinoCNC ab Version 1.4.0
// Benötigt Processing 2.x
// Benötigt ArduinoCNC ab 1.4.0
// ----------------------------------------------------------------------------------
// Interface für den PC/Linux
// Gekoppelt über Serielle Schnittstelle mit Arduino
//
// Befehle, die über die Serielle Schnittstelle gesendet werden:
//  '5' - Links Y-Achse nach oben
//  '6' - Links Y-Achse nach unten
//  '7' - Links X-Achse nach vorne
//  '8' - Links X-Achse nach hinten
//  '1' - Rechts Y-Achse nach oben
//  '2' - Rechts Y-Achse nach unten
//  '3' - Rechts X-Achse nach vorne
//  '4' - Rechts X-Achse nach hinten
//  '0' - Stoppt die Bewegung aller Achsen
//  'S' - Startet den Schneidevorgang. Ab dann holt sich der Arduino die Daten ab
//  "<DIR><hex>" - bewegt beide Seiten um <hex> Schritte
//                 <DIR> = 'U'  Aufwärts
//                 <DIR> = 'D'  Abwärts
//                 <DIR> = 'F'  Vorwärts
//                 <DIR> = 'B'  Rückwärts
//                 Beispiel: F00F3
//  "<X/Y><sign><StepHex><sign><XaspectHex><sign>YaspectHex>"
//                 - Übergibt einen Vector an den Arduino. Der Arduino weiß, ob für die rechte
//                   oder linke Seite, weil er den Vector vorher entsprechend angefragt hat. 
//                   Die drei Vorzeichen geben die Richtung an.
//                   Die drei Werte sind unsigned Hex (4 Characters).
//                   StepHex - Anzahl der Schritte in X-Richtung
//                   XAspectHex - Alle wieviel Clocks wird ein X-Motorschritt gemacht.
//                   YAspectHex - Alle wieviel Clocks wird ein Y-Motorschritt gemacht.
//                   Beispiel: +024E+0041-003D
// "##----#----#----" - Keine weiteren Vectoren mehr d.h. Arduino hört auf danach zu fragen.
//
// Befehle, die empfangen werden:
// 'L' - Arduino fordert nächsten Vector für linke Seite an
// 'R' - Arduino fordert nächsten Vector für rechte Seite an
// 'A' - Rechter Endschalter in X-Richtung erreicht
// 'B' - Linker Endschalter in X-Richtung erreicht
// 'a' - Rechter Endschalter in Y-Richtung erreicht
// 'b' - Linker Endschalter in Y-Richtung erreicht

import controlP5.*;
import processing.serial.*;

final int TURNS_PER_MM = 48;  // Anzahl der Schritte pro Umdrehung des Schrittmotors 
final float MOTOR_DIST = 850;  // Abstand der beiden x/y-Führungen voneinander in mm
final int X_STOP = 7;       // Abstand des Endschalters vom Nullpunkt des Styropors
final int Y_STOP = 5;       // Abstand des Endschalters von der Grundplatte
final float PLT_SCALE = 0.02;  // Anzahl mm pro PLT Einheit

String[] configLines;
String configPath, configSerial;

String FoilLeft, FoilRight;  
String[] profilLinks, profilRechts;
float[] plx, ply,prx,pry;   // Koordinaten für Arduino incl. Tiefe, Anstellwinkel etc.
float[] plxRaw, plyRaw,prxRaw,pryRaw;  // Koordinaten aus Profildatei
char[] plmRaw, prmRaw; // Marker, die aus der PLT Datei ausgelesen werden
int sizeLeft, sizeRight; // Anzahl der Koordinaten pro Seite

byte[] rawFile; // In dieses Array werden die Inputdateien byte-weise geladen
int cmdIndex; // Index des geparsten Kommandos

int idxLeft, idxRight;   // globale Command Counter (pointer auf die aktuelle Stützstelle)
int Heizleistung=70;

float lastXR=0, lastYR=0, lastXL=0, lastYL=0;
float contourLeft=0, contourRight=0, contourAverage=0;
int markerCountL, markerCountR;
int idxLastMarkerL;    // Speichert immer die letzte position eines Markers
int idxLastMarkerR;     


float tiefeL0, tiefeR0;   // wahre Tiefe an den Motorpositionen (= programmierte Tiefe nur bei Rechteckflächen)

ControlP5 controlP5;

Textfield SegBreite;
Textfield yStart;

// The serial port:
Serial myPort;  

void setup() {
  int i;
  size(600,560);
  frameRate(25);
  
  controlP5 = new ControlP5(this);

  controlP5.addButton("LoadLeft",0,10,25,100,19).setId(1);
  controlP5.addButton("LoadRight",0,300,25,100,19).setId(2);
  
  SegBreite = controlP5.addTextfield("Rumpflaenge",190,370,80,20);
  yStart = controlP5.addTextfield("y-Start",310,370,80,20);
  yStart.setText("20");
  controlP5.addSlider("Heizleistung",0,255,70,60,475,160,14).setId(7);
 
  controlP5.addButton("Home",0,30,520,100,19).setId(8);
  controlP5.addButton("Start",0,170,520,100,19).setId(9);
  controlP5.addButton("Heiz-Test",128,400,470,100,20).setId(12);
  controlP5.addButton("Cancel",255,310,520,100,19).setId(10);
  controlP5.addButton("Exit",128,450,520,100,19).setId(11);

  controlP5.addButton("UL",128,65,320,30,30).setId(21);
  controlP5.addButton("FL",128,30,360,30,30).setId(22);
  controlP5.addButton("BL",128,100,360,30,30).setId(23);
  controlP5.addButton("DL",128,65,400,30,30).setId(24);

  controlP5.addButton("UR",128,485,320,30,30).setId(31);
  controlP5.addButton("FR",128,450,360,30,30).setId(32);
  controlP5.addButton("BR",128,520,360,30,30).setId(33);
  controlP5.addButton("DR",128,485,400,30,30).setId(34);

  println("Serial Interfaces found!");
  println(Serial.list());

  configLines = loadStrings("PCHotwire.ini");
  configPath = configLines[0];
  configSerial = configLines[1];

  println("Config File found!");
  println("Serial Interface: "+configSerial);
  println("Profile Path: "+configPath);
  println("sketchPath="+sketchPath);
  
  myPort = new Serial(this, configSerial, 9600);

  background(100);
  fill(100);
  stroke(60);
  rect(10,50,280,240);
  rect(300,50,280,240);  
  rect(10,300,150,150);
  rect(170,300,250,150);
  rect(430,300,150,150);
  rect(10,460,570,40);
}


// Diese Funktion wird automatisch 30x pro Sekunde aufgerufen. Daher sollte das Polling des Ports 
// funktionieren.
// Hier fordert der Arduino jeweils rechts oder links einen neuen Vector an.
void draw() {
   char inByte='0';
   
   if (myPort.available() > 0) {    
      inByte = myPort.readChar();
      print(inByte);             // nur für debug
      
      if (inByte=='L') {  // Linke Seite braucht neuen Vector
         myPort.write(getNextVector('L'));  //Vector über serielle Schnittstelle senden
      }
      
      if (inByte=='R') {  // Rechte Seite braucht neuen Vector
         myPort.write(getNextVector('R'));  //Vector über serielle Schnittstelle senden
      }

   }   
}

//======================= Support functions ========================

String getNextVector(char side) {
  float x,y,dx,dy;
  float stepsX, stepsY;
  float aspectX, aspectY;
  float contour;
  float speedFactor;
  int aspX, aspY, stp;
  String outStr, aspXhex, aspYhex, stpHex;
  char aspXsign, aspYsign, stpSign;
  int ende=0;
  int linDist;
  char xySelect;   // Neu: Gibt an, ob relativ zur X- oder Y-Achse gerechnet wird 

  if (side=='L') {
    contour = contourLeft;
    x=plx[idxLeft];    
    y=ply[idxLeft];
    dx=(x-lastXL);
    dy=(y-lastYL);
    lastXL=x;
    lastYL=y;
    if (idxLeft < sizeLeft) idxLeft++; else ende=1;
    
  } else {
    contour = contourRight;
    x=prx[idxRight];  
    y=pry[idxRight];
    dx=(x-lastXR);
    dy=(y-lastYR);
    lastXR=x;
    lastYR=y;
    if (idxRight < sizeRight) idxRight++; else ende=1;
  }  

  stepsX=dx*TURNS_PER_MM;   
  stepsY=dy*TURNS_PER_MM;
  linDist = int(sqrt(stepsX*stepsX + stepsY*stepsY));     // Vektorlänge in steps (Pytagoras)
  
  // Je größer die Tiefe desto kleiner der Abstand zwischen zwei Schritten
  // Der SpeedFactor ist für x und y gleich aber unterscheidet sich bei Trapezflächen zwischen rechts und links 
  speedFactor = contourAverage / contour;    
  
  // Ursrünglich wurde hier mit 20 multipliziert. Nun wird mit 50 multipliziert, was die Genauigkeit um den
  // Faktor 2.5 verbessert. Auf der Arduino Seite muss daher der Delay verkelinert werden, damit die Fahr-
  // geschwindigkeit nicht auch um den Faktor 2.5 heruntergeht. 
  if (stepsX!=0) aspectX=speedFactor*linDist*50/stepsX; else aspectX=linDist*100;   // Berechnung der cycles pro Schritt 
  if (stepsY!=0) aspectY=speedFactor*linDist*50/stepsY; else aspectY=linDist*100;

  if (abs(aspectX)>32000) aspectX=32000;   // Sicherstellen, dass es keinen Überlauf gibt
  if (abs(aspectY)>32000) aspectY=32000; 
  
  aspX= int(aspectX);
  aspY= int(aspectY);
  
  //Abhängig davon, ob mehr X- oder Y-Schritte gebraucht werden, wird die Referenz für die Total-Steps ausgewählt
  if (abs(stepsX)>abs(stepsY)) {
    stp= int(stepsX);
    xySelect = 'X';
  }
  else {
    stp = int(stepsY);
    xySelect = 'Y';
  } 
  
  if (stp>=0) stpSign='+'; else stpSign='-';
  if (aspX>=0) aspXsign='+'; else aspXsign='-';
  if (aspY>=0) aspYsign='+'; else aspYsign='-';
  
  println(ende+" # "+idxLeft+"-"+sizeLeft+" | "+idxRight+"-"+sizeRight+" # "+y+" * "+stp+" - "+aspX+" * "+aspY+" * "+xySelect);
  
  stpHex = hex(abs(stp));
  aspXhex = hex(abs(aspX));
  aspYhex = hex(abs(aspY));

  // Neu seit 1.5.0: xySelect wird am Anfang ausgegeben, damit der Arduino weiß, auf welche Achse sich die Steps beziehen
  outStr= stpSign + stpHex.substring(4) +  aspXsign + aspXhex.substring(4) + aspYsign + aspYhex.substring(4);
  outStr = xySelect + outStr; 
  println("--> "+outStr);

  // Wenn an dieser Koordinate ein Marker war, und das bereits auf der anderen Seite auch schon passiert ist
  // dann werden hier neue Umfänge berechnet
  if (side=='L') {
    if (plmRaw[idxLeft-1] == '*') {
      markerCountL++;
      idxLastMarkerL = idxLeft-1;
      if (markerCountL == markerCountR) generateNextContour();  // Wenn die rechte Seite auch schon den Marker erreicht hat, dann wird
                                                               // der nächste Umfang berechnet
    }
  } else {
    if (prmRaw[idxRight-1] == '*') {
      markerCountR++;
      idxLastMarkerR = idxRight-1;
      if (markerCountL == markerCountR) generateNextContour();  // Wenn die linke Seite auch schon den Marker erreicht hat, dann wird
                                                               // der nächste Umfang berechnet
    }
  }
    
  if (ende==0) return(outStr); else return("##----#----#----");

}

// Verfährt beide Seiten um distance in mm
void drive(char axis, int distance) { 
  String outStr, distHex;
  char sign;
  
  if (distance>0) sign='+'; else sign='-';
  distHex = hex(abs(distance*TURNS_PER_MM));
  outStr = axis + sign + distHex.substring(4);
  println("Drive: "+outStr);
  myPort.write(outStr);
}

// Hier wird der tatsächliche Unfang bis zum nächsten Marker berechnet bzw. bis zum Ende
void generateNextContour() {
  int i;
  float lastX, lastY;
  boolean markerFound;
  
  markerFound=false;
  contourLeft=0;
  if (idxLastMarkerL>=0) {
    lastX=plx[idxLastMarkerL];
    lastY=ply[idxLastMarkerL];
    i = idxLastMarkerL+1;
  } else {
    lastX=0;
    lastY=0;
    i=0;
  }
  while ((i<sizeLeft-1)&&(!markerFound)) {    // Umfang bis Ende oder nächstem Marker berechnen
                                                // -1 deshalb weil die Rückfahrt zum Startpunkt nicht mehr dazu gehört
    contourLeft += sqrt(sq(plx[i]-lastX) + sq(ply[i]-lastY));  // Pythagoras für Streckenlänge
    lastX=plx[i];
    lastY=ply[i];
    if (plmRaw[i]=='*') markerFound=true;
    i++;
  }
  

  markerFound=false;
  contourRight=0;
  if (idxLastMarkerR>=0) {
    lastX=prx[idxLastMarkerR];
    lastY=pry[idxLastMarkerR];
    i = idxLastMarkerR+1;
  } else {
    lastX=0;
    lastY=0;
    i=0;
  }
  while ((i<sizeRight-1)&&(!markerFound)) {    // Umfang bis Ende oder nächstem Marker berechnen
                                                // -1 deshalb weil die Rückfahrt zum Startpunkt nicht mehr dazu gehört
    contourRight += sqrt(sq(prx[i]-lastX) + sq(pry[i]-lastY));  // Pythagoras für Streckenlänge
    lastX=prx[i];
    lastY=pry[i];
    if (prmRaw[i]=='*') markerFound=true;
    i++;
  }
  
  contourAverage = (contourRight + contourLeft) / 2;
  println("ContourRight = ",contourRight);
  println("ContourLeftt = ",contourLeft);
  println("ContourAverage = ",contourAverage);
}

void applySettings()  // Berechne Koordinaten aus Raw-Koordinaten
{
   int i;
   float diameterR, diameterL;
   float alphaOrg, alphaNew, deltaAlpha, len;
   float alphaSweep;
   float segmentLength, scaleL, scaleR;
   float d, dir, x1, x2, y1, y2;
   float min, max;
   
   int lastx=0, lasty=0; 
   
   plx = new float [plxRaw.length];
   ply = new float [plyRaw.length];
   prx = new float [prxRaw.length];
   pry = new float [pryRaw.length];
   
   min=plxRaw[0]; max=min;      // Breite des linken Teils bestimmen
   for (i=1; i<sizeLeft; i++) {
     if (plxRaw[i]>max) max= plxRaw[i];
     if (plxRaw[i]<min) min= plxRaw[i];
   }
   diameterL = max - min;

   min=prxRaw[0]; max=min;      // Breite des rechten Teils bestimmen
   for (i=1; i<sizeRight; i++) {
     if (prxRaw[i]>max) max= prxRaw[i];
     if (prxRaw[i]<min) min= prxRaw[i];
   }
   diameterR = max - min;
   
   println("DiameterL = ", diameterL);
   println("DiameterR = ", diameterR);
   
   
   //========= Ab hier wird die Segmentlänge verrechnet um die tatsächliche X/Y-Bewegung zu erhalten =========
   segmentLength = float(SegBreite.getText());
   tiefeL0 = diameterL + (MOTOR_DIST-segmentLength) * (diameterL - diameterR) / 2 / segmentLength;
   tiefeR0 = diameterR + + (MOTOR_DIST-segmentLength) * (diameterR - diameterL) / 2 / segmentLength;
   scaleL = tiefeL0/diameterL;
   scaleR = tiefeR0/diameterR;
   
   for (i=0; i<plxRaw.length; i++) {
     plx[i] = plxRaw[i] * scaleL;  
     ply[i] = plyRaw[i] * scaleL; 
   }   
   for (i=0; i<prxRaw.length; i++) {
     prx[i] = prxRaw[i] * scaleR;
     pry[i] = pryRaw[i] * scaleR;
   }   
}

public void homeWire() {
int xr, xl, yr, yl;
int xDist, yDist;
char inByte;
String hexNumber;

   xr=0; xl=0;
   myPort.write('8');    // Die beiden x-Achsen werden rückwärts bewegt
   myPort.write('4');
   do {
      if (myPort.available() > 0) { 
      inByte=myPort.readChar();
        if (inByte=='A') xr=1;
        if (inByte=='B') xl=1;
      }  
   } while (xr*xl==0);   // Ende, wenn beide Seiten am Anschlag sind
   yr=0; yl=0;
   myPort.write('6');    // Die beiden y-Achsen werden nach unten bewegt
   myPort.write('2');
   do {
      if (myPort.available() > 0) { 
        inByte=myPort.readChar();
        if (inByte=='a') yr=1;
        if (inByte=='b') yl=1;
      }  
   } while (yr*yl==0);    // Ende, wenn beide Seiten am Anschlag sind
       
   yDist = int(yStart.getText());      // Draht auf die y-Startposition fahren
   hexNumber=hex((yDist-Y_STOP)*TURNS_PER_MM);   
   println("Schritte fahren: "+'U'+hexNumber.substring(4));
   myPort.write('U'+hexNumber.substring(4));

   hexNumber=hex(X_STOP*TURNS_PER_MM);  // Draht an den Anfang des Styroblocks fahren 
   println("Schritte fahren: "+'F'+hexNumber.substring(4));
   myPort.write('F'+hexNumber.substring(4));
}

// Hier werden die GUI Events verarbeitet

public void controlEvent(ControlEvent theEvent) {
  int i;
  String profil;
  String hexNumber;

  println("got a control event from controller with id "+theEvent.controller().id());
  switch(theEvent.controller().id()) {
  case(1): // LoadLeft
    println("Load Left");
    selectInput("Datei für linke Kontur auswählen", "leftFile");
  break;
  case(2): //Loadright
    println("load Right");
    selectInput("Datei für rechte Kontur auswählen", "rightFile");
  break;  
  case(8): // Home
    println("Home");
    homeWire();
  break;
  case(9): // Start
    println("Start");
    applySettings();
    idxLeft=0;      // Vectorzähler zurücksetzen
    idxRight=0;
    markerCountL=0;  // Zähler für Marker zurücksetzen
    markerCountR=0;
    idxLastMarkerL=-1;  // Marker positionen initialisieren
    idxLastMarkerR=-1;
    generateNextContour();  // Zunächst mal die Startwerte generieren
    lastXL=0; lastXR=0; lastYL=0; lastYR=0;
    hexNumber = hex(Heizleistung); 
    myPort.write('H'+hexNumber.substring(6));   // Heizdraht einschalten
    delay(500);  // Warten bis aufgeheizt
    myPort.write('S');   // Der Arduino wird gestartet und holt sich dann selbst immer 
                           // wieder neue Vektoren ab (siehe Funktion "draw")  
    break;
  case(10): // Cancel
    println("Cancel");
    myPort.write('0');
  break;
  case(11): // Exit
    println("Exit");
    exit();
  break;
  case(12): // Start Heizung
    hexNumber = hex(Heizleistung); 
    println("Heizung an:"+'H'+hexNumber.substring(6));
    myPort.write('H'+hexNumber.substring(6));
  break;
  case(21):
    println("UL");
    myPort.write('5');
  break;
  case(22):
    println("FL");
    myPort.write('7');
  break;
  case(23):
    println("BL");
    myPort.write('8');
  break;
  case(24):
    println("DL");
    myPort.write('6');
  break;
  case(31):
    println("UR");
    myPort.write('1');
  break;
  case(32):
    println("FR");
    myPort.write('3');
  break;
  case(33):
    println("BR");
    myPort.write('4');
  break;
  case(34):
    println("DR");
    myPort.write('2');
  break;
  }
}

void leftFile(File selection) {
  if (selection!=null) 
     loadPLT('L',selection.getAbsolutePath());
}

void rightFile(File selection) {
   println(selection.getAbsolutePath());

   if (selection!=null) 
     loadPLT('R',selection.getAbsolutePath());
}


public void loadPLT(char id, String datei) {
  int i,j;
  float x,y,lastx=0,lasty=0;
  StringList commands;
  String cmd,opCode;
  String[] pieces;
  boolean ende=false;
  
  cmdIndex = 0;   // Auf den Anfang der Datei setzen;
  rawFile=loadBytes(datei);
  commands = new StringList();   // Enthält die Liste der PLT Commandos
  
  while ((!ende)&&(cmdIndex<rawFile.length)){   // Liste wird aus dem byte-Array herausgelesen
    ende = true;
    cmd = getNextPLTCommand(rawFile);
    if (cmd != "-") {
      commands.append(cmd);
      ende = false; 
    }
  }
  commands.append("PU0,0");   // Zum Schluss noch nach Hause fahren
  
  if (id=='L') {
    plxRaw = new float [commands.size()];    // Linkes Koordinaten-File anlegen;
    plyRaw = new float [commands.size()];
    plmRaw = new char [commands.size()];
  } else {
    prxRaw = new float [commands.size()];   // Rechtes Koordinaten-File anlegen
    pryRaw = new float [commands.size()];
    prmRaw = new char [commands.size()];
  }
  
  j=0;
  for (i=0; i<commands.size(); i++) {  // PLT Coomands nacheinander in mm Koordinaten umrechnen     
    cmd=commands.get(i);
    opCode = cmd.substring(0,2);
    if (opCode.equals("PD")||opCode.equals("PU")) {
      pieces = splitTokens(cmd.substring(2),",");
      if (pieces.length>1) {
        x = float(pieces[0])*PLT_SCALE;
        y = float(pieces[1])*PLT_SCALE;
        println(x," , ",y);
        if (id=='L') {
          plxRaw[j] = x;
          plyRaw[j] = y;
          j++;
        } else {
          prxRaw[j] = x;
          pryRaw[j] = y;
          j++;
        }
      }
      if (pieces.length == 3) {     // Kommando enthält einen Marker
        if (id=='L') plmRaw[j-1] = '*';
        else prmRaw[j-1] = '*';
      } else {
        if (id=='L') plmRaw[j-1] = ' ';
        else prmRaw[j-1] = ' ';
      }
    }
  }
  
  if (id=='L')  {                                   // Hier werden die tatsächlichen Koordinaten gezählt
    sizeLeft=j;                                     // Das können weniger sein als die Anzahl der Commands, da
    println("Linke Seite: ", sizeLeft, " Punkte");  // das PLT File auch Commands ohne Koordinaten enthalten kann.
  } else {
    sizeRight=j;  
    println("Rechte Seite: ", sizeRight, " Punkte");
  }

  // Die erste Strecke wird mit einem Marker belegt, weil dann der Darht auf beiden Seiten gleichzeitig an der Kontur ankommt
  if (id=='L' )plmRaw[0]='*'; else prmRaw[0]='*';
  displayShape(id);  
}  

String getNextPLTCommand(byte b[]) {
    String out = "";

    if (cmdIndex >= b.length) return("-");
    while((cmdIndex<b.length)&&(b[cmdIndex]!=';')&&(b[cmdIndex]!='*')) {
      out += char(b[cmdIndex]);
      cmdIndex++;
    }
    if (b[cmdIndex] == '*') out+=",*";  // Marker
    cmdIndex++;
    return(out);
}

void displayShape(char id) {
  float max=0;
  float lastX=0, lastY=0;
  int i;
  
  if (id=='L') {
    fill(100);  // Linken Anzeigebereich löschen
    stroke(60);
    rect(10,50,280,240);
    fill(255,0,0);
    stroke(0);
    for (i=0; i<plxRaw.length; i++) if (plxRaw[i]>max) max=plxRaw[i];    // Größten Wert finden (für Skalierung)
    for (i=0; i<plxRaw.length; i++) {
      line(int(20+lastX/max*250), int(270-lastY/max*250), int(20+plxRaw[i]/max*250), int(270-plyRaw[i]/max*250));
      if (plmRaw[i]=='*') ellipse(int(20+plxRaw[i]/max*250),int(270-plyRaw[i]/max*250),5,5);
      lastX=plxRaw[i];
      lastY=plyRaw[i];
    }
    fill(0,0,180); 
    text("Breite = "+str(max)+" mm", 150,280);
  } else {
    fill(100);  // Rechten Anzeigebereich löschen
    stroke(60);
    rect(300,50,280,240);
    fill(255,0,0);
    stroke(0); 
    for (i=0; i<prxRaw.length; i++) if (prxRaw[i]>max) max=prxRaw[i];    // Größten Wert finden (für Skalierung)
    for (i=0; i<prxRaw.length; i++) {
      line(int(310+lastX/max*250), int(270-lastY/max*250), int(310+prxRaw[i]/max*250), int(270-pryRaw[i]/max*250));
      if (prmRaw[i]=='*') ellipse(int(310+prxRaw[i]/max*250),int(270-pryRaw[i]/max*250),5,5);
      lastX=prxRaw[i];
      lastY=pryRaw[i];
    }
    fill(0,0,180); 
    text("Breite = "+str(max)+" mm", 450,280);
  }
}
