/* ----------------------------------------------------------------------
 * Copyright (C) 2010 ARM Limited. All rights reserved.
 *
 * $Date:        15. February 2012
 * $Revision: 	V1.1.0
 *
 * Project: 	    CMSIS DSP Library
 * Title:	    arm_common_tables.c
 *
 * Description:	This file has common tables like fft twiddle factors,
 * Bitreverse, reciprocal etc which are used across different functions
 *
 * Target Processor: Cortex-M4/Cortex-M3/Cortex-M0
 *
 * Version 1.1.0 2012/02/15
 *    Updated with more optimizations, bug fixes and minor API changes.
 *
 * Version 1.0.10 2011/7/15
 *    Big Endian support added and Merged M0 and M3/M4 Source code.
 *
 * Version 1.0.3 2010/11/29
 *    Re-organized the CMSIS folders and updated documentation.
 *
 * Version 1.0.2 2010/11/11
 *    Documentation updated.
 *
 * Version 1.0.1 2010/10/05
 *    Production release and review comments incorporated.
 *
 * Version 1.0.0 2010/09/20
 *    Production release and review comments incorporated.
 * -------------------------------------------------------------------- */

#include "arm_common_tables.h"

/**
 * @ingroup groupTransforms
 */

/**
 * @addtogroup CFFT_CIFFT Complex FFT Tables
 * @{
 */

/**
 * \par
 * Pseudo code for Generation of Bit reversal Table is
 * \par
 * <pre>for(l=1;l <= N/4;l++)
 * {
 *   for(i=0;i<logN2;i++)
 *   {
 *     a[i]=l&(1<<i);
 *   }
 *   for(j=0; j<logN2; j++)
 *   {
 *     if (a[j]!=0)
 *     y[l]+=(1<<((logN2-1)-j));
 *   }
 *   y[l] = y[l] >> 1;
 *  } </pre>
 * \par
 * where N = 4096	logN2 = 12
 * \par
 * N is the maximum FFT Size supported
 */

/*
 * @brief  Table for bit reversal process
 */
const uint16_t armBitRevTable[1024] = {
    0x400, 0x200, 0x600, 0x100, 0x500, 0x300, 0x700, 0x80,  0x480, 0x280, 0x680,
    0x180, 0x580, 0x380, 0x780, 0x40,  0x440, 0x240, 0x640, 0x140, 0x540, 0x340,
    0x740, 0xc0,  0x4c0, 0x2c0, 0x6c0, 0x1c0, 0x5c0, 0x3c0, 0x7c0, 0x20,  0x420,
    0x220, 0x620, 0x120, 0x520, 0x320, 0x720, 0xa0,  0x4a0, 0x2a0, 0x6a0, 0x1a0,
    0x5a0, 0x3a0, 0x7a0, 0x60,  0x460, 0x260, 0x660, 0x160, 0x560, 0x360, 0x760,
    0xe0,  0x4e0, 0x2e0, 0x6e0, 0x1e0, 0x5e0, 0x3e0, 0x7e0, 0x10,  0x410, 0x210,
    0x610, 0x110, 0x510, 0x310, 0x710, 0x90,  0x490, 0x290, 0x690, 0x190, 0x590,
    0x390, 0x790, 0x50,  0x450, 0x250, 0x650, 0x150, 0x550, 0x350, 0x750, 0xd0,
    0x4d0, 0x2d0, 0x6d0, 0x1d0, 0x5d0, 0x3d0, 0x7d0, 0x30,  0x430, 0x230, 0x630,
    0x130, 0x530, 0x330, 0x730, 0xb0,  0x4b0, 0x2b0, 0x6b0, 0x1b0, 0x5b0, 0x3b0,
    0x7b0, 0x70,  0x470, 0x270, 0x670, 0x170, 0x570, 0x370, 0x770, 0xf0,  0x4f0,
    0x2f0, 0x6f0, 0x1f0, 0x5f0, 0x3f0, 0x7f0, 0x8,   0x408, 0x208, 0x608, 0x108,
    0x508, 0x308, 0x708, 0x88,  0x488, 0x288, 0x688, 0x188, 0x588, 0x388, 0x788,
    0x48,  0x448, 0x248, 0x648, 0x148, 0x548, 0x348, 0x748, 0xc8,  0x4c8, 0x2c8,
    0x6c8, 0x1c8, 0x5c8, 0x3c8, 0x7c8, 0x28,  0x428, 0x228, 0x628, 0x128, 0x528,
    0x328, 0x728, 0xa8,  0x4a8, 0x2a8, 0x6a8, 0x1a8, 0x5a8, 0x3a8, 0x7a8, 0x68,
    0x468, 0x268, 0x668, 0x168, 0x568, 0x368, 0x768, 0xe8,  0x4e8, 0x2e8, 0x6e8,
    0x1e8, 0x5e8, 0x3e8, 0x7e8, 0x18,  0x418, 0x218, 0x618, 0x118, 0x518, 0x318,
    0x718, 0x98,  0x498, 0x298, 0x698, 0x198, 0x598, 0x398, 0x798, 0x58,  0x458,
    0x258, 0x658, 0x158, 0x558, 0x358, 0x758, 0xd8,  0x4d8, 0x2d8, 0x6d8, 0x1d8,
    0x5d8, 0x3d8, 0x7d8, 0x38,  0x438, 0x238, 0x638, 0x138, 0x538, 0x338, 0x738,
    0xb8,  0x4b8, 0x2b8, 0x6b8, 0x1b8, 0x5b8, 0x3b8, 0x7b8, 0x78,  0x478, 0x278,
    0x678, 0x178, 0x578, 0x378, 0x778, 0xf8,  0x4f8, 0x2f8, 0x6f8, 0x1f8, 0x5f8,
    0x3f8, 0x7f8, 0x4,   0x404, 0x204, 0x604, 0x104, 0x504, 0x304, 0x704, 0x84,
    0x484, 0x284, 0x684, 0x184, 0x584, 0x384, 0x784, 0x44,  0x444, 0x244, 0x644,
    0x144, 0x544, 0x344, 0x744, 0xc4,  0x4c4, 0x2c4, 0x6c4, 0x1c4, 0x5c4, 0x3c4,
    0x7c4, 0x24,  0x424, 0x224, 0x624, 0x124, 0x524, 0x324, 0x724, 0xa4,  0x4a4,
    0x2a4, 0x6a4, 0x1a4, 0x5a4, 0x3a4, 0x7a4, 0x64,  0x464, 0x264, 0x664, 0x164,
    0x564, 0x364, 0x764, 0xe4,  0x4e4, 0x2e4, 0x6e4, 0x1e4, 0x5e4, 0x3e4, 0x7e4,
    0x14,  0x414, 0x214, 0x614, 0x114, 0x514, 0x314, 0x714, 0x94,  0x494, 0x294,
    0x694, 0x194, 0x594, 0x394, 0x794, 0x54,  0x454, 0x254, 0x654, 0x154, 0x554,
    0x354, 0x754, 0xd4,  0x4d4, 0x2d4, 0x6d4, 0x1d4, 0x5d4, 0x3d4, 0x7d4, 0x34,
    0x434, 0x234, 0x634, 0x134, 0x534, 0x334, 0x734, 0xb4,  0x4b4, 0x2b4, 0x6b4,
    0x1b4, 0x5b4, 0x3b4, 0x7b4, 0x74,  0x474, 0x274, 0x674, 0x174, 0x574, 0x374,
    0x774, 0xf4,  0x4f4, 0x2f4, 0x6f4, 0x1f4, 0x5f4, 0x3f4, 0x7f4, 0xc,   0x40c,
    0x20c, 0x60c, 0x10c, 0x50c, 0x30c, 0x70c, 0x8c,  0x48c, 0x28c, 0x68c, 0x18c,
    0x58c, 0x38c, 0x78c, 0x4c,  0x44c, 0x24c, 0x64c, 0x14c, 0x54c, 0x34c, 0x74c,
    0xcc,  0x4cc, 0x2cc, 0x6cc, 0x1cc, 0x5cc, 0x3cc, 0x7cc, 0x2c,  0x42c, 0x22c,
    0x62c, 0x12c, 0x52c, 0x32c, 0x72c, 0xac,  0x4ac, 0x2ac, 0x6ac, 0x1ac, 0x5ac,
    0x3ac, 0x7ac, 0x6c,  0x46c, 0x26c, 0x66c, 0x16c, 0x56c, 0x36c, 0x76c, 0xec,
    0x4ec, 0x2ec, 0x6ec, 0x1ec, 0x5ec, 0x3ec, 0x7ec, 0x1c,  0x41c, 0x21c, 0x61c,
    0x11c, 0x51c, 0x31c, 0x71c, 0x9c,  0x49c, 0x29c, 0x69c, 0x19c, 0x59c, 0x39c,
    0x79c, 0x5c,  0x45c, 0x25c, 0x65c, 0x15c, 0x55c, 0x35c, 0x75c, 0xdc,  0x4dc,
    0x2dc, 0x6dc, 0x1dc, 0x5dc, 0x3dc, 0x7dc, 0x3c,  0x43c, 0x23c, 0x63c, 0x13c,
    0x53c, 0x33c, 0x73c, 0xbc,  0x4bc, 0x2bc, 0x6bc, 0x1bc, 0x5bc, 0x3bc, 0x7bc,
    0x7c,  0x47c, 0x27c, 0x67c, 0x17c, 0x57c, 0x37c, 0x77c, 0xfc,  0x4fc, 0x2fc,
    0x6fc, 0x1fc, 0x5fc, 0x3fc, 0x7fc, 0x2,   0x402, 0x202, 0x602, 0x102, 0x502,
    0x302, 0x702, 0x82,  0x482, 0x282, 0x682, 0x182, 0x582, 0x382, 0x782, 0x42,
    0x442, 0x242, 0x642, 0x142, 0x542, 0x342, 0x742, 0xc2,  0x4c2, 0x2c2, 0x6c2,
    0x1c2, 0x5c2, 0x3c2, 0x7c2, 0x22,  0x422, 0x222, 0x622, 0x122, 0x522, 0x322,
    0x722, 0xa2,  0x4a2, 0x2a2, 0x6a2, 0x1a2, 0x5a2, 0x3a2, 0x7a2, 0x62,  0x462,
    0x262, 0x662, 0x162, 0x562, 0x362, 0x762, 0xe2,  0x4e2, 0x2e2, 0x6e2, 0x1e2,
    0x5e2, 0x3e2, 0x7e2, 0x12,  0x412, 0x212, 0x612, 0x112, 0x512, 0x312, 0x712,
    0x92,  0x492, 0x292, 0x692, 0x192, 0x592, 0x392, 0x792, 0x52,  0x452, 0x252,
    0x652, 0x152, 0x552, 0x352, 0x752, 0xd2,  0x4d2, 0x2d2, 0x6d2, 0x1d2, 0x5d2,
    0x3d2, 0x7d2, 0x32,  0x432, 0x232, 0x632, 0x132, 0x532, 0x332, 0x732, 0xb2,
    0x4b2, 0x2b2, 0x6b2, 0x1b2, 0x5b2, 0x3b2, 0x7b2, 0x72,  0x472, 0x272, 0x672,
    0x172, 0x572, 0x372, 0x772, 0xf2,  0x4f2, 0x2f2, 0x6f2, 0x1f2, 0x5f2, 0x3f2,
    0x7f2, 0xa,   0x40a, 0x20a, 0x60a, 0x10a, 0x50a, 0x30a, 0x70a, 0x8a,  0x48a,
    0x28a, 0x68a, 0x18a, 0x58a, 0x38a, 0x78a, 0x4a,  0x44a, 0x24a, 0x64a, 0x14a,
    0x54a, 0x34a, 0x74a, 0xca,  0x4ca, 0x2ca, 0x6ca, 0x1ca, 0x5ca, 0x3ca, 0x7ca,
    0x2a,  0x42a, 0x22a, 0x62a, 0x12a, 0x52a, 0x32a, 0x72a, 0xaa,  0x4aa, 0x2aa,
    0x6aa, 0x1aa, 0x5aa, 0x3aa, 0x7aa, 0x6a,  0x46a, 0x26a, 0x66a, 0x16a, 0x56a,
    0x36a, 0x76a, 0xea,  0x4ea, 0x2ea, 0x6ea, 0x1ea, 0x5ea, 0x3ea, 0x7ea, 0x1a,
    0x41a, 0x21a, 0x61a, 0x11a, 0x51a, 0x31a, 0x71a, 0x9a,  0x49a, 0x29a, 0x69a,
    0x19a, 0x59a, 0x39a, 0x79a, 0x5a,  0x45a, 0x25a, 0x65a, 0x15a, 0x55a, 0x35a,
    0x75a, 0xda,  0x4da, 0x2da, 0x6da, 0x1da, 0x5da, 0x3da, 0x7da, 0x3a,  0x43a,
    0x23a, 0x63a, 0x13a, 0x53a, 0x33a, 0x73a, 0xba,  0x4ba, 0x2ba, 0x6ba, 0x1ba,
    0x5ba, 0x3ba, 0x7ba, 0x7a,  0x47a, 0x27a, 0x67a, 0x17a, 0x57a, 0x37a, 0x77a,
    0xfa,  0x4fa, 0x2fa, 0x6fa, 0x1fa, 0x5fa, 0x3fa, 0x7fa, 0x6,   0x406, 0x206,
    0x606, 0x106, 0x506, 0x306, 0x706, 0x86,  0x486, 0x286, 0x686, 0x186, 0x586,
    0x386, 0x786, 0x46,  0x446, 0x246, 0x646, 0x146, 0x546, 0x346, 0x746, 0xc6,
    0x4c6, 0x2c6, 0x6c6, 0x1c6, 0x5c6, 0x3c6, 0x7c6, 0x26,  0x426, 0x226, 0x626,
    0x126, 0x526, 0x326, 0x726, 0xa6,  0x4a6, 0x2a6, 0x6a6, 0x1a6, 0x5a6, 0x3a6,
    0x7a6, 0x66,  0x466, 0x266, 0x666, 0x166, 0x566, 0x366, 0x766, 0xe6,  0x4e6,
    0x2e6, 0x6e6, 0x1e6, 0x5e6, 0x3e6, 0x7e6, 0x16,  0x416, 0x216, 0x616, 0x116,
    0x516, 0x316, 0x716, 0x96,  0x496, 0x296, 0x696, 0x196, 0x596, 0x396, 0x796,
    0x56,  0x456, 0x256, 0x656, 0x156, 0x556, 0x356, 0x756, 0xd6,  0x4d6, 0x2d6,
    0x6d6, 0x1d6, 0x5d6, 0x3d6, 0x7d6, 0x36,  0x436, 0x236, 0x636, 0x136, 0x536,
    0x336, 0x736, 0xb6,  0x4b6, 0x2b6, 0x6b6, 0x1b6, 0x5b6, 0x3b6, 0x7b6, 0x76,
    0x476, 0x276, 0x676, 0x176, 0x576, 0x376, 0x776, 0xf6,  0x4f6, 0x2f6, 0x6f6,
    0x1f6, 0x5f6, 0x3f6, 0x7f6, 0xe,   0x40e, 0x20e, 0x60e, 0x10e, 0x50e, 0x30e,
    0x70e, 0x8e,  0x48e, 0x28e, 0x68e, 0x18e, 0x58e, 0x38e, 0x78e, 0x4e,  0x44e,
    0x24e, 0x64e, 0x14e, 0x54e, 0x34e, 0x74e, 0xce,  0x4ce, 0x2ce, 0x6ce, 0x1ce,
    0x5ce, 0x3ce, 0x7ce, 0x2e,  0x42e, 0x22e, 0x62e, 0x12e, 0x52e, 0x32e, 0x72e,
    0xae,  0x4ae, 0x2ae, 0x6ae, 0x1ae, 0x5ae, 0x3ae, 0x7ae, 0x6e,  0x46e, 0x26e,
    0x66e, 0x16e, 0x56e, 0x36e, 0x76e, 0xee,  0x4ee, 0x2ee, 0x6ee, 0x1ee, 0x5ee,
    0x3ee, 0x7ee, 0x1e,  0x41e, 0x21e, 0x61e, 0x11e, 0x51e, 0x31e, 0x71e, 0x9e,
    0x49e, 0x29e, 0x69e, 0x19e, 0x59e, 0x39e, 0x79e, 0x5e,  0x45e, 0x25e, 0x65e,
    0x15e, 0x55e, 0x35e, 0x75e, 0xde,  0x4de, 0x2de, 0x6de, 0x1de, 0x5de, 0x3de,
    0x7de, 0x3e,  0x43e, 0x23e, 0x63e, 0x13e, 0x53e, 0x33e, 0x73e, 0xbe,  0x4be,
    0x2be, 0x6be, 0x1be, 0x5be, 0x3be, 0x7be, 0x7e,  0x47e, 0x27e, 0x67e, 0x17e,
    0x57e, 0x37e, 0x77e, 0xfe,  0x4fe, 0x2fe, 0x6fe, 0x1fe, 0x5fe, 0x3fe, 0x7fe,
    0x1};

/*
 * @brief  Floating-point Twiddle factors Table Generation
 */

/**
 * \par
 * Example code for Floating-point Twiddle factors Generation:
 * \par
 * <pre>for(i = 0; i< 3N/4; i++)
 * {
 *	twiddleCoef[2*i]= cos(i * 2*PI/(float)N);
 *	twiddleCoef[2*i+1]= sin(i * 2*PI/(float)N);
 * } </pre>
 * \par
 * where N = 4096	and PI = 3.14159265358979
 * \par
 * Cos and Sin values are in interleaved fashion
 *
 */
const float32_t twiddleCoef[6144] = {
    1.000000000000000000f,  0.000000000000000000f,  0.999998823451701880f,
    0.001533980186284766f,  0.999995293809576190f,  0.003067956762965976f,
    0.999989411081928400f,  0.004601926120448571f,  0.999981175282601110f,
    0.006135884649154475f,  0.999970586430974140f,  0.007669828739531097f,
    0.999957644551963900f,  0.009203754782059819f,  0.999942349676023910f,
    0.010737659167264491f,  0.999924701839144500f,  0.012271538285719925f,
    0.999904701082852900f,  0.013805388528060391f,  0.999882347454212560f,
    0.015339206284988100f,  0.999857641005823860f,  0.016872987947281710f,
    0.999830581795823400f,  0.018406729905804820f,  0.999801169887884260f,
    0.019940428551514441f,  0.999769405351215280f,  0.021474080275469508f,
    0.999735288260561680f,  0.023007681468839369f,  0.999698818696204250f,
    0.024541228522912288f,  0.999659996743959220f,  0.026074717829103901f,
    0.999618822495178640f,  0.027608145778965740f,  0.999575296046749220f,
    0.029141508764193722f,  0.999529417501093140f,  0.030674803176636626f,
    0.999481186966166950f,  0.032208025408304586f,  0.999430604555461730f,
    0.033741171851377580f,  0.999377670388002850f,  0.035274238898213947f,
    0.999322384588349540f,  0.036807222941358832f,  0.999264747286594420f,
    0.038340120373552694f,  0.999204758618363890f,  0.039872927587739811f,
    0.999142418724816910f,  0.041405640977076739f,  0.999077727752645360f,
    0.042938256934940820f,  0.999010685854073380f,  0.044470771854938668f,
    0.998941293186856870f,  0.046003182130914623f,  0.998869549914283560f,
    0.047535484156959303f,  0.998795456205172410f,  0.049067674327418015f,
    0.998719012233872940f,  0.050599749036899282f,  0.998640218180265270f,
    0.052131704680283324f,  0.998559074229759310f,  0.053663537652730520f,
    0.998475580573294770f,  0.055195244349689934f,  0.998389737407340160f,
    0.056726821166907748f,  0.998301544933892890f,  0.058258264500435752f,
    0.998211003360478190f,  0.059789570746639868f,  0.998118112900149180f,
    0.061320736302208578f,  0.998022873771486240f,  0.062851757564161406f,
    0.997925286198596000f,  0.064382630929857465f,  0.997825350411111640f,
    0.065913352797003805f,  0.997723066644191640f,  0.067443919563664051f,
    0.997618435138519550f,  0.068974327628266746f,  0.997511456140303450f,
    0.070504573389613856f,  0.997402129901275300f,  0.072034653246889332f,
    0.997290456678690210f,  0.073564563599667426f,  0.997176436735326190f,
    0.075094300847921305f,  0.997060070339482960f,  0.076623861392031492f,
    0.996941357764982160f,  0.078153241632794232f,  0.996820299291165670f,
    0.079682437971430126f,  0.996696895202896060f,  0.081211446809592441f,
    0.996571145790554840f,  0.082740264549375692f,  0.996443051350042630f,
    0.084268887593324071f,  0.996312612182778000f,  0.085797312344439894f,
    0.996179828595696980f,  0.087325535206192059f,  0.996044700901251970f,
    0.088853552582524600f,  0.995907229417411720f,  0.090381360877864983f,
    0.995767414467659820f,  0.091908956497132724f,  0.995625256380994310f,
    0.093436335845747787f,  0.995480755491926940f,  0.094963495329638992f,
    0.995333912140482280f,  0.096490431355252593f,  0.995184726672196930f,
    0.098017140329560604f,  0.995033199438118630f,  0.099543618660069319f,
    0.994879330794805620f,  0.101069862754827820f,  0.994723121104325700f,
    0.102595869022436280f,  0.994564570734255420f,  0.104121633872054590f,
    0.994403680057679100f,  0.105647153713410620f,  0.994240449453187900f,
    0.107172424956808840f,  0.994074879304879370f,  0.108697444013138720f,
    0.993906970002356060f,  0.110222207293883060f,  0.993736721940724600f,
    0.111746711211126590f,  0.993564135520595300f,  0.113270952177564350f,
    0.993389211148080650f,  0.114794926606510080f,  0.993211949234794500f,
    0.116318630911904750f,  0.993032350197851410f,  0.117842061508324980f,
    0.992850414459865100f,  0.119365214810991350f,  0.992666142448948020f,
    0.120888087235777080f,  0.992479534598709970f,  0.122410675199216200f,
    0.992290591348257370f,  0.123932975118512160f,  0.992099313142191800f,
    0.125454983411546230f,  0.991905700430609330f,  0.126976696496885870f,
    0.991709753669099530f,  0.128498110793793170f,  0.991511473318743900f,
    0.130019222722233350f,  0.991310859846115440f,  0.131540028702883120f,
    0.991107913723276890f,  0.133060525157139060f,  0.990902635427780010f,
    0.134580708507126170f,  0.990695025442664630f,  0.136100575175706200f,
    0.990485084256457090f,  0.137620121586486040f,  0.990272812363169110f,
    0.139139344163826200f,  0.990058210262297120f,  0.140658239332849210f,
    0.989841278458820530f,  0.142176803519448030f,  0.989622017463200890f,
    0.143695033150294470f,  0.989400427791380380f,  0.145212924652847460f,
    0.989176509964781010f,  0.146730474455361750f,  0.988950264510302990f,
    0.148247678986896030f,  0.988721691960323780f,  0.149764534677321510f,
    0.988490792852696590f,  0.151281037957330220f,  0.988257567730749460f,
    0.152797185258443440f,  0.988022017143283530f,  0.154312973013020100f,
    0.987784141644572180f,  0.155828397654265230f,  0.987543941794359230f,
    0.157343455616238250f,  0.987301418157858430f,  0.158858143333861450f,
    0.987056571305750970f,  0.160372457242928280f,  0.986809401814185530f,
    0.161886393780111830f,  0.986559910264775410f,  0.163399949382973230f,
    0.986308097244598670f,  0.164913120489969890f,  0.986053963346195440f,
    0.166425903540464100f,  0.985797509167567480f,  0.167938294974731170f,
    0.985538735312176060f,  0.169450291233967960f,  0.985277642388941220f,
    0.170961888760301220f,  0.985014231012239840f,  0.172473083996795950f,
    0.984748501801904210f,  0.173983873387463820f,  0.984480455383220930f,
    0.175494253377271430f,  0.984210092386929030f,  0.177004220412148750f,
    0.983937413449218920f,  0.178513770938997510f,  0.983662419211730250f,
    0.180022901405699510f,  0.983385110321551180f,  0.181531608261124970f,
    0.983105487431216290f,  0.183039887955140950f,  0.982823551198705240f,
    0.184547736938619620f,  0.982539302287441240f,  0.186055151663446630f,
    0.982252741366289370f,  0.187562128582529600f,  0.981963869109555240f,
    0.189068664149806190f,  0.981672686196983110f,  0.190574754820252740f,
    0.981379193313754560f,  0.192080397049892440f,  0.981083391150486710f,
    0.193585587295803610f,  0.980785280403230430f,  0.195090322016128250f,
    0.980484861773469380f,  0.196594597670080220f,  0.980182135968117430f,
    0.198098410717953560f,  0.979877103699517640f,  0.199601757621130970f,
    0.979569765685440520f,  0.201104634842091900f,  0.979260122649082020f,
    0.202607038844421130f,  0.978948175319062200f,  0.204108966092816870f,
    0.978633924429423210f,  0.205610413053099240f,  0.978317370719627650f,
    0.207111376192218560f,  0.977998514934557140f,  0.208611851978263490f,
    0.977677357824509930f,  0.210111836880469610f,  0.977353900145199960f,
    0.211611327369227550f,  0.977028142657754390f,  0.213110319916091360f,
    0.976700086128711840f,  0.214608810993786760f,  0.976369731330021140f,
    0.216106797076219520f,  0.976037079039039020f,  0.217604274638483640f,
    0.975702130038528570f,  0.219101240156869800f,  0.975364885116656980f,
    0.220597690108873510f,  0.975025345066994120f,  0.222093620973203510f,
    0.974683510688510670f,  0.223589029229789990f,  0.974339382785575860f,
    0.225083911359792830f,  0.973992962167955830f,  0.226578263845610000f,
    0.973644249650811980f,  0.228072083170885730f,  0.973293246054698250f,
    0.229565365820518870f,  0.972939952205560180f,  0.231058108280671110f,
    0.972584368934732210f,  0.232550307038775240f,  0.972226497078936270f,
    0.234041958583543430f,  0.971866337480279400f,  0.235533059404975490f,
    0.971503890986251780f,  0.237023605994367200f,  0.971139158449725090f,
    0.238513594844318420f,  0.970772140728950350f,  0.240003022448741500f,
    0.970402838687555500f,  0.241491885302869330f,  0.970031253194543970f,
    0.242980179903263870f,  0.969657385124292450f,  0.244467902747824150f,
    0.969281235356548530f,  0.245955050335794590f,  0.968902804776428870f,
    0.247441619167773270f,  0.968522094274417380f,  0.248927605745720150f,
    0.968139104746362440f,  0.250413006572965220f,  0.967753837093475510f,
    0.251897818154216970f,  0.967366292222328510f,  0.253382036995570160f,
    0.966976471044852070f,  0.254865659604514570f,  0.966584374478333120f,
    0.256348682489942910f,  0.966190003445412500f,  0.257831102162158990f,
    0.965793358874083680f,  0.259312915132886230f,  0.965394441697689400f,
    0.260794117915275510f,  0.964993252854920320f,  0.262274707023913590f,
    0.964589793289812760f,  0.263754678974831350f,  0.964184063951745830f,
    0.265234030285511790f,  0.963776065795439840f,  0.266712757474898370f,
    0.963365799780954050f,  0.268190857063403180f,  0.962953266873683880f,
    0.269668325572915090f,  0.962538468044359160f,  0.271145159526808010f,
    0.962121404269041580f,  0.272621355449948980f,  0.961702076529122540f,
    0.274096909868706380f,  0.961280485811320640f,  0.275571819310958140f,
    0.960856633107679660f,  0.277046080306099900f,  0.960430519415565790f,
    0.278519689385053060f,  0.960002145737665960f,  0.279992643080273220f,
    0.959571513081984520f,  0.281464937925757940f,  0.959138622461841890f,
    0.282936570457055390f,  0.958703474895871600f,  0.284407537211271880f,
    0.958266071408017670f,  0.285877834727080620f,  0.957826413027532910f,
    0.287347459544729510f,  0.957384500788975860f,  0.288816408206049480f,
    0.956940335732208820f,  0.290284677254462330f,  0.956493918902395100f,
    0.291752263234989260f,  0.956045251349996410f,  0.293219162694258630f,
    0.955594334130771110f,  0.294685372180514330f,  0.955141168305770780f,
    0.296150888243623790f,  0.954685754941338340f,  0.297615707435086200f,
    0.954228095109105670f,  0.299079826308040480f,  0.953768189885990330f,
    0.300543241417273450f,  0.953306040354193860f,  0.302005949319228080f,
    0.952841647601198720f,  0.303467946572011320f,  0.952375012719765880f,
    0.304929229735402370f,  0.951906136807932350f,  0.306389795370860920f,
    0.951435020969008340f,  0.307849640041534870f,  0.950961666311575080f,
    0.309308760312268730f,  0.950486073949481700f,  0.310767152749611470f,
    0.950008245001843000f,  0.312224813921824880f,  0.949528180593036670f,
    0.313681740398891520f,  0.949045881852700560f,  0.315137928752522440f,
    0.948561349915730270f,  0.316593375556165850f,  0.948074585922276230f,
    0.318048077385014950f,  0.947585591017741090f,  0.319502030816015690f,
    0.947094366352777220f,  0.320955232427875210f,  0.946600913083283530f,
    0.322407678801069850f,  0.946105232370403450f,  0.323859366517852850f,
    0.945607325380521280f,  0.325310292162262930f,  0.945107193285260610f,
    0.326760452320131730f,  0.944604837261480260f,  0.328209843579092500f,
    0.944100258491272660f,  0.329658462528587490f,  0.943593458161960390f,
    0.331106305759876430f,  0.943084437466093490f,  0.332553369866044220f,
    0.942573197601446870f,  0.333999651442009380f,  0.942059739771017310f,
    0.335445147084531600f,  0.941544065183020810f,  0.336889853392220050f,
    0.941026175050889260f,  0.338333766965541130f,  0.940506070593268300f,
    0.339776884406826850f,  0.939983753034014050f,  0.341219202320282360f,
    0.939459223602189920f,  0.342660717311994380f,  0.938932483532064600f,
    0.344101425989938810f,  0.938403534063108060f,  0.345541324963989090f,
    0.937872376439989890f,  0.346980410845923680f,  0.937339011912574960f,
    0.348418680249434560f,  0.936803441735921560f,  0.349856129790134920f,
    0.936265667170278260f,  0.351292756085567090f,  0.935725689481080370f,
    0.352728555755210730f,  0.935183509938947610f,  0.354163525420490340f,
    0.934639129819680780f,  0.355597661704783850f,  0.934092550404258980f,
    0.357030961233429980f,  0.933543772978836170f,  0.358463420633736540f,
    0.932992798834738960f,  0.359895036534988110f,  0.932439629268462360f,
    0.361325805568454280f,  0.931884265581668150f,  0.362755724367397230f,
    0.931326709081180430f,  0.364184789567079890f,  0.930766961078983710f,
    0.365612997804773850f,  0.930205022892219070f,  0.367040345719767180f,
    0.929640895843181330f,  0.368466829953372320f,  0.929074581259315860f,
    0.369892447148934100f,  0.928506080473215590f,  0.371317193951837540f,
    0.927935394822617890f,  0.372741067009515760f,  0.927362525650401110f,
    0.374164062971457930f,  0.926787474304581750f,  0.375586178489217220f,
    0.926210242138311380f,  0.377007410216418260f,  0.925630830509872720f,
    0.378427754808765560f,  0.925049240782677580f,  0.379847208924051160f,
    0.924465474325262600f,  0.381265769222162380f,  0.923879532511286740f,
    0.382683432365089780f,  0.923291416719527640f,  0.384100195016935040f,
    0.922701128333878630f,  0.385516053843918850f,  0.922108668743345180f,
    0.386931005514388580f,  0.921514039342042010f,  0.388345046698826250f,
    0.920917241529189520f,  0.389758174069856410f,  0.920318276709110590f,
    0.391170384302253870f,  0.919717146291227360f,  0.392581674072951470f,
    0.919113851690057770f,  0.393992040061048100f,  0.918508394325212250f,
    0.395401478947816350f,  0.917900775621390500f,  0.396809987416710310f,
    0.917290997008377910f,  0.398217562153373560f,  0.916679059921042700f,
    0.399624199845646790f,  0.916064965799331720f,  0.401029897183575620f,
    0.915448716088267830f,  0.402434650859418430f,  0.914830312237946200f,
    0.403838457567654070f,  0.914209755703530690f,  0.405241314004989860f,
    0.913587047945250810f,  0.406643216870369030f,  0.912962190428398210f,
    0.408044162864978690f,  0.912335184623322750f,  0.409444148692257590f,
    0.911706032005429880f,  0.410843171057903910f,  0.911074734055176360f,
    0.412241226669882890f,  0.910441292258067250f,  0.413638312238434500f,
    0.909805708104652220f,  0.415034424476081630f,  0.909167983090522380f,
    0.416429560097637150f,  0.908528118716306120f,  0.417823715820212270f,
    0.907886116487666260f,  0.419216888363223910f,  0.907241977915295820f,
    0.420609074448402510f,  0.906595704514915330f,  0.422000270799799680f,
    0.905947297807268460f,  0.423390474143796050f,  0.905296759318118820f,
    0.424779681209108810f,  0.904644090578246240f,  0.426167888726799620f,
    0.903989293123443340f,  0.427555093430282080f,  0.903332368494511820f,
    0.428941292055329490f,  0.902673318237258830f,  0.430326481340082610f,
    0.902012143902493180f,  0.431710658025057260f,  0.901348847046022030f,
    0.433093818853151960f,  0.900683429228646970f,  0.434475960569655650f,
    0.900015892016160280f,  0.435857079922255470f,  0.899346236979341570f,
    0.437237173661044090f,  0.898674465693953820f,  0.438616238538527660f,
    0.898000579740739880f,  0.439994271309633260f,  0.897324580705418320f,
    0.441371268731716670f,  0.896646470178680150f,  0.442747227564570020f,
    0.895966249756185220f,  0.444122144570429200f,  0.895283921038557580f,
    0.445496016513981740f,  0.894599485631382700f,  0.446868840162374160f,
    0.893912945145203250f,  0.448240612285219890f,  0.893224301195515320f,
    0.449611329654606540f,  0.892533555402764580f,  0.450980989045103860f,
    0.891840709392342720f,  0.452349587233770890f,  0.891145764794583180f,
    0.453717121000163870f,  0.890448723244757880f,  0.455083587126343840f,
    0.889749586383072780f,  0.456448982396883920f,  0.889048355854664570f,
    0.457813303598877170f,  0.888345033309596350f,  0.459176547521944090f,
    0.887639620402853930f,  0.460538710958240010f,  0.886932118794342190f,
    0.461899790702462730f,  0.886222530148880640f,  0.463259783551860150f,
    0.885510856136199950f,  0.464618686306237820f,  0.884797098430937790f,
    0.465976495767966180f,  0.884081258712634990f,  0.467333208741988420f,
    0.883363338665731580f,  0.468688822035827900f,  0.882643339979562790f,
    0.470043332459595620f,  0.881921264348355050f,  0.471396736825997640f,
    0.881197113471222090f,  0.472749031950342790f,  0.880470889052160750f,
    0.474100214650549970f,  0.879742592800047410f,  0.475450281747155870f,
    0.879012226428633530f,  0.476799230063322090f,  0.878279791656541580f,
    0.478147056424843010f,  0.877545290207261350f,  0.479493757660153010f,
    0.876808723809145650f,  0.480839330600333960f,  0.876070094195406600f,
    0.482183772079122720f,  0.875329403104110890f,  0.483527078932918740f,
    0.874586652278176110f,  0.484869248000791060f,  0.873841843465366860f,
    0.486210276124486420f,  0.873094978418290090f,  0.487550160148436000f,
    0.872346058894391540f,  0.488888896919763170f,  0.871595086655950980f,
    0.490226483288291160f,  0.870842063470078980f,  0.491562916106549900f,
    0.870086991108711460f,  0.492898192229784040f,  0.869329871348606840f,
    0.494232308515959670f,  0.868570705971340900f,  0.495565261825772540f,
    0.867809496763303320f,  0.496897049022654470f,  0.867046245515692650f,
    0.498227666972781870f,  0.866280954024512990f,  0.499557112545081840f,
    0.865513624090569090f,  0.500885382611240710f,  0.864744257519462380f,
    0.502212474045710790f,  0.863972856121586810f,  0.503538383725717580f,
    0.863199421712124160f,  0.504863108531267590f,  0.862423956111040610f,
    0.506186645345155230f,  0.861646461143081300f,  0.507508991052970870f,
    0.860866938637767310f,  0.508830142543106990f,  0.860085390429390140f,
    0.510150096706766810f,  0.859301818357008470f,  0.511468850437970300f,
    0.858516224264442740f,  0.512786400633562960f,  0.857728610000272120f,
    0.514102744193221660f,  0.856938977417828760f,  0.515417878019462930f,
    0.856147328375194470f,  0.516731799017649870f,  0.855353664735196030f,
    0.518044504095999340f,  0.854557988365400530f,  0.519355990165589640f,
    0.853760301138111410f,  0.520666254140367160f,  0.852960604930363630f,
    0.521975292937154390f,  0.852158901623919830f,  0.523283103475656430f,
    0.851355193105265200f,  0.524589682678468950f,  0.850549481265603480f,
    0.525895027471084630f,  0.849741768000852550f,  0.527199134781901280f,
    0.848932055211639610f,  0.528502001542228480f,  0.848120344803297230f,
    0.529803624686294610f,  0.847306638685858320f,  0.531104001151255000f,
    0.846490938774052130f,  0.532403127877197900f,  0.845673246987299070f,
    0.533701001807152960f,  0.844853565249707120f,  0.534997619887097150f,
    0.844031895490066410f,  0.536292979065963180f,  0.843208239641845440f,
    0.537587076295645390f,  0.842382599643185850f,  0.538879908531008420f,
    0.841554977436898440f,  0.540171472729892850f,  0.840725374970458070f,
    0.541461765853123440f,  0.839893794195999520f,  0.542750784864515890f,
    0.839060237070312740f,  0.544038526730883820f,  0.838224705554838080f,
    0.545324988422046460f,  0.837387201615661940f,  0.546610166910834860f,
    0.836547727223512010f,  0.547894059173100190f,  0.835706284353752600f,
    0.549176662187719660f,  0.834862874986380010f,  0.550457972936604810f,
    0.834017501106018130f,  0.551737988404707340f,  0.833170164701913190f,
    0.553016705580027470f,  0.832320867767929680f,  0.554294121453620000f,
    0.831469612302545240f,  0.555570233019602180f,  0.830616400308846310f,
    0.556845037275160100f,  0.829761233794523050f,  0.558118531220556100f,
    0.828904114771864870f,  0.559390711859136140f,  0.828045045257755800f,
    0.560661576197336030f,  0.827184027273669130f,  0.561931121244689470f,
    0.826321062845663530f,  0.563199344013834090f,  0.825456154004377550f,
    0.564466241520519500f,  0.824589302785025290f,  0.565731810783613120f,
    0.823720511227391430f,  0.566996048825108680f,  0.822849781375826430f,
    0.568258952670131490f,  0.821977115279241550f,  0.569520519346947140f,
    0.821102514991104650f,  0.570780745886967260f,  0.820225982569434690f,
    0.572039629324757050f,  0.819347520076796900f,  0.573297166698042200f,
    0.818467129580298660f,  0.574553355047715760f,  0.817584813151583710f,
    0.575808191417845340f,  0.816700572866827850f,  0.577061672855679440f,
    0.815814410806733780f,  0.578313796411655590f,  0.814926329056526620f,
    0.579564559139405630f,  0.814036329705948410f,  0.580813958095764530f,
    0.813144414849253590f,  0.582061990340775440f,  0.812250586585203880f,
    0.583308652937698290f,  0.811354847017063730f,  0.584553942953015330f,
    0.810457198252594770f,  0.585797857456438860f,  0.809557642404051260f,
    0.587040393520917970f,  0.808656181588174980f,  0.588281548222645220f,
    0.807752817926190360f,  0.589521318641063940f,  0.806847553543799330f,
    0.590759701858874160f,  0.805940390571176280f,  0.591996694962040990f,
    0.805031331142963660f,  0.593232295039799800f,  0.804120377398265810f,
    0.594466499184664430f,  0.803207531480644940f,  0.595699304492433360f,
    0.802292795538115720f,  0.596930708062196500f,  0.801376171723140240f,
    0.598160706996342270f,  0.800457662192622820f,  0.599389298400564540f,
    0.799537269107905010f,  0.600616479383868970f,  0.798614994634760820f,
    0.601842247058580030f,  0.797690840943391160f,  0.603066598540348160f,
    0.796764810208418830f,  0.604289530948155960f,  0.795836904608883570f,
    0.605511041404325550f,  0.794907126328237010f,  0.606731127034524480f,
    0.793975477554337170f,  0.607949784967773630f,  0.793041960479443640f,
    0.609167012336453210f,  0.792106577300212390f,  0.610382806276309480f,
    0.791169330217690200f,  0.611597163926461910f,  0.790230221437310030f,
    0.612810082429409710f,  0.789289253168885650f,  0.614021558931038380f,
    0.788346427626606340f,  0.615231590580626820f,  0.787401747029031430f,
    0.616440174530853650f,  0.786455213599085770f,  0.617647307937803870f,
    0.785506829564053930f,  0.618852987960976320f,  0.784556597155575240f,
    0.620057211763289100f,  0.783604518609638200f,  0.621259976511087550f,
    0.782650596166575730f,  0.622461279374149970f,  0.781694832071059390f,
    0.623661117525694530f,  0.780737228572094490f,  0.624859488142386340f,
    0.779777787923014550f,  0.626056388404343520f,  0.778816512381475980f,
    0.627251815495144080f,  0.777853404209453150f,  0.628445766601832710f,
    0.776888465673232440f,  0.629638238914926980f,  0.775921699043407690f,
    0.630829229628424470f,  0.774953106594873930f,  0.632018735939809060f,
    0.773982690606822900f,  0.633206755050057190f,  0.773010453362736990f,
    0.634393284163645490f,  0.772036397150384520f,  0.635578320488556110f,
    0.771060524261813820f,  0.636761861236284200f,  0.770082836993347900f,
    0.637943903621844060f,  0.769103337645579700f,  0.639124444863775730f,
    0.768122028523365420f,  0.640303482184151670f,  0.767138911935820400f,
    0.641481012808583160f,  0.766153990196312920f,  0.642657033966226860f,
    0.765167265622458960f,  0.643831542889791390f,  0.764178740536116670f,
    0.645004536815543930f,  0.763188417263381270f,  0.646176012983316280f,
    0.762196298134578900f,  0.647345968636512060f,  0.761202385484261780f,
    0.648514401022112440f,  0.760206681651202420f,  0.649681307390683190f,
    0.759209188978388070f,  0.650846684996380880f,  0.758209909813015280f,
    0.652010531096959500f,  0.757208846506484570f,  0.653172842953776760f,
    0.756206001414394540f,  0.654333617831800440f,  0.755201376896536550f,
    0.655492852999615350f,  0.754194975316889170f,  0.656650545729428940f,
    0.753186799043612520f,  0.657806693297078640f,  0.752176850449042810f,
    0.658961292982037320f,  0.751165131909686480f,  0.660114342067420480f,
    0.750151645806215070f,  0.661265837839992270f,  0.749136394523459370f,
    0.662415777590171780f,  0.748119380450403600f,  0.663564158612039770f,
    0.747100605980180130f,  0.664710978203344790f,  0.746080073510063780f,
    0.665856233665509720f,  0.745057785441466060f,  0.666999922303637470f,
    0.744033744179929290f,  0.668142041426518450f,  0.743007952135121720f,
    0.669282588346636010f,  0.741980411720831070f,  0.670421560380173090f,
    0.740951125354959110f,  0.671558954847018330f,  0.739920095459516200f,
    0.672694769070772860f,  0.738887324460615110f,  0.673829000378756040f,
    0.737852814788465980f,  0.674961646102011930f,  0.736816568877369900f,
    0.676092703575315920f,  0.735778589165713590f,  0.677222170137180330f,
    0.734738878095963500f,  0.678350043129861470f,  0.733697438114660370f,
    0.679476319899364970f,  0.732654271672412820f,  0.680600997795453020f,
    0.731609381223892630f,  0.681724074171649710f,  0.730562769227827590f,
    0.682845546385248080f,  0.729514438146997010f,  0.683965411797315400f,
    0.728464390448225200f,  0.685083667772700360f,  0.727412628602375770f,
    0.686200311680038590f,  0.726359155084346010f,  0.687315340891759050f,
    0.725303972373060770f,  0.688428752784090440f,  0.724247082951467000f,
    0.689540544737066830f,  0.723188489306527460f,  0.690650714134534600f,
    0.722128193929215350f,  0.691759258364157750f,  0.721066199314508110f,
    0.692866174817424630f,  0.720002507961381650f,  0.693971460889654000f,
    0.718937122372804490f,  0.695075113980000880f,  0.717870045055731710f,
    0.696177131491462990f,  0.716801278521099540f,  0.697277510830886520f,
    0.715730825283818590f,  0.698376249408972920f,  0.714658687862769090f,
    0.699473344640283770f,  0.713584868780793640f,  0.700568793943248340f,
    0.712509370564692320f,  0.701662594740168450f,  0.711432195745216430f,
    0.702754744457225300f,  0.710353346857062420f,  0.703845240524484940f,
    0.709272826438865690f,  0.704934080375904880f,  0.708190637033195400f,
    0.706021261449339740f,  0.707106781186547570f,  0.707106781186547460f,
    0.706021261449339740f,  0.708190637033195290f,  0.704934080375904990f,
    0.709272826438865580f,  0.703845240524484940f,  0.710353346857062310f,
    0.702754744457225300f,  0.711432195745216430f,  0.701662594740168570f,
    0.712509370564692320f,  0.700568793943248450f,  0.713584868780793520f,
    0.699473344640283770f,  0.714658687862768980f,  0.698376249408972920f,
    0.715730825283818590f,  0.697277510830886630f,  0.716801278521099540f,
    0.696177131491462990f,  0.717870045055731710f,  0.695075113980000880f,
    0.718937122372804380f,  0.693971460889654000f,  0.720002507961381650f,
    0.692866174817424740f,  0.721066199314508110f,  0.691759258364157750f,
    0.722128193929215350f,  0.690650714134534720f,  0.723188489306527350f,
    0.689540544737066940f,  0.724247082951466890f,  0.688428752784090550f,
    0.725303972373060660f,  0.687315340891759160f,  0.726359155084346010f,
    0.686200311680038700f,  0.727412628602375770f,  0.685083667772700360f,
    0.728464390448225200f,  0.683965411797315510f,  0.729514438146996900f,
    0.682845546385248080f,  0.730562769227827590f,  0.681724074171649820f,
    0.731609381223892520f,  0.680600997795453130f,  0.732654271672412820f,
    0.679476319899365080f,  0.733697438114660260f,  0.678350043129861580f,
    0.734738878095963390f,  0.677222170137180450f,  0.735778589165713480f,
    0.676092703575316030f,  0.736816568877369790f,  0.674961646102012040f,
    0.737852814788465980f,  0.673829000378756150f,  0.738887324460615110f,
    0.672694769070772970f,  0.739920095459516090f,  0.671558954847018330f,
    0.740951125354959110f,  0.670421560380173090f,  0.741980411720830960f,
    0.669282588346636010f,  0.743007952135121720f,  0.668142041426518560f,
    0.744033744179929180f,  0.666999922303637470f,  0.745057785441465950f,
    0.665856233665509720f,  0.746080073510063780f,  0.664710978203344900f,
    0.747100605980180130f,  0.663564158612039880f,  0.748119380450403490f,
    0.662415777590171780f,  0.749136394523459260f,  0.661265837839992270f,
    0.750151645806214960f,  0.660114342067420480f,  0.751165131909686370f,
    0.658961292982037320f,  0.752176850449042700f,  0.657806693297078640f,
    0.753186799043612410f,  0.656650545729429050f,  0.754194975316889170f,
    0.655492852999615460f,  0.755201376896536550f,  0.654333617831800550f,
    0.756206001414394540f,  0.653172842953776760f,  0.757208846506484460f,
    0.652010531096959500f,  0.758209909813015280f,  0.650846684996380990f,
    0.759209188978387960f,  0.649681307390683190f,  0.760206681651202420f,
    0.648514401022112550f,  0.761202385484261780f,  0.647345968636512060f,
    0.762196298134578900f,  0.646176012983316390f,  0.763188417263381270f,
    0.645004536815544040f,  0.764178740536116670f,  0.643831542889791500f,
    0.765167265622458960f,  0.642657033966226860f,  0.766153990196312810f,
    0.641481012808583160f,  0.767138911935820400f,  0.640303482184151670f,
    0.768122028523365310f,  0.639124444863775730f,  0.769103337645579590f,
    0.637943903621844170f,  0.770082836993347900f,  0.636761861236284200f,
    0.771060524261813710f,  0.635578320488556230f,  0.772036397150384410f,
    0.634393284163645490f,  0.773010453362736990f,  0.633206755050057190f,
    0.773982690606822790f,  0.632018735939809060f,  0.774953106594873820f,
    0.630829229628424470f,  0.775921699043407580f,  0.629638238914927100f,
    0.776888465673232440f,  0.628445766601832710f,  0.777853404209453040f,
    0.627251815495144190f,  0.778816512381475870f,  0.626056388404343520f,
    0.779777787923014440f,  0.624859488142386450f,  0.780737228572094380f,
    0.623661117525694640f,  0.781694832071059390f,  0.622461279374150080f,
    0.782650596166575730f,  0.621259976511087660f,  0.783604518609638200f,
    0.620057211763289210f,  0.784556597155575240f,  0.618852987960976320f,
    0.785506829564053930f,  0.617647307937803980f,  0.786455213599085770f,
    0.616440174530853650f,  0.787401747029031320f,  0.615231590580626820f,
    0.788346427626606230f,  0.614021558931038490f,  0.789289253168885650f,
    0.612810082429409710f,  0.790230221437310030f,  0.611597163926462020f,
    0.791169330217690090f,  0.610382806276309480f,  0.792106577300212390f,
    0.609167012336453210f,  0.793041960479443640f,  0.607949784967773740f,
    0.793975477554337170f,  0.606731127034524480f,  0.794907126328237010f,
    0.605511041404325550f,  0.795836904608883460f,  0.604289530948156070f,
    0.796764810208418720f,  0.603066598540348280f,  0.797690840943391040f,
    0.601842247058580030f,  0.798614994634760820f,  0.600616479383868970f,
    0.799537269107905010f,  0.599389298400564540f,  0.800457662192622710f,
    0.598160706996342380f,  0.801376171723140130f,  0.596930708062196500f,
    0.802292795538115720f,  0.595699304492433470f,  0.803207531480644830f,
    0.594466499184664540f,  0.804120377398265700f,  0.593232295039799800f,
    0.805031331142963660f,  0.591996694962040990f,  0.805940390571176280f,
    0.590759701858874280f,  0.806847553543799220f,  0.589521318641063940f,
    0.807752817926190360f,  0.588281548222645330f,  0.808656181588174980f,
    0.587040393520918080f,  0.809557642404051260f,  0.585797857456438860f,
    0.810457198252594770f,  0.584553942953015330f,  0.811354847017063730f,
    0.583308652937698290f,  0.812250586585203880f,  0.582061990340775550f,
    0.813144414849253590f,  0.580813958095764530f,  0.814036329705948300f,
    0.579564559139405740f,  0.814926329056526620f,  0.578313796411655590f,
    0.815814410806733780f,  0.577061672855679550f,  0.816700572866827850f,
    0.575808191417845340f,  0.817584813151583710f,  0.574553355047715760f,
    0.818467129580298660f,  0.573297166698042320f,  0.819347520076796900f,
    0.572039629324757050f,  0.820225982569434690f,  0.570780745886967370f,
    0.821102514991104650f,  0.569520519346947250f,  0.821977115279241550f,
    0.568258952670131490f,  0.822849781375826320f,  0.566996048825108680f,
    0.823720511227391320f,  0.565731810783613230f,  0.824589302785025290f,
    0.564466241520519500f,  0.825456154004377440f,  0.563199344013834090f,
    0.826321062845663420f,  0.561931121244689470f,  0.827184027273669020f,
    0.560661576197336030f,  0.828045045257755800f,  0.559390711859136140f,
    0.828904114771864870f,  0.558118531220556100f,  0.829761233794523050f,
    0.556845037275160100f,  0.830616400308846200f,  0.555570233019602290f,
    0.831469612302545240f,  0.554294121453620110f,  0.832320867767929680f,
    0.553016705580027580f,  0.833170164701913190f,  0.551737988404707450f,
    0.834017501106018130f,  0.550457972936604810f,  0.834862874986380010f,
    0.549176662187719770f,  0.835706284353752600f,  0.547894059173100190f,
    0.836547727223511890f,  0.546610166910834860f,  0.837387201615661940f,
    0.545324988422046460f,  0.838224705554837970f,  0.544038526730883930f,
    0.839060237070312630f,  0.542750784864516000f,  0.839893794195999410f,
    0.541461765853123560f,  0.840725374970458070f,  0.540171472729892970f,
    0.841554977436898330f,  0.538879908531008420f,  0.842382599643185960f,
    0.537587076295645510f,  0.843208239641845440f,  0.536292979065963180f,
    0.844031895490066410f,  0.534997619887097260f,  0.844853565249707010f,
    0.533701001807152960f,  0.845673246987299070f,  0.532403127877198010f,
    0.846490938774052020f,  0.531104001151255000f,  0.847306638685858320f,
    0.529803624686294830f,  0.848120344803297120f,  0.528502001542228480f,
    0.848932055211639610f,  0.527199134781901390f,  0.849741768000852440f,
    0.525895027471084740f,  0.850549481265603370f,  0.524589682678468840f,
    0.851355193105265200f,  0.523283103475656430f,  0.852158901623919830f,
    0.521975292937154390f,  0.852960604930363630f,  0.520666254140367270f,
    0.853760301138111300f,  0.519355990165589530f,  0.854557988365400530f,
    0.518044504095999340f,  0.855353664735196030f,  0.516731799017649980f,
    0.856147328375194470f,  0.515417878019463150f,  0.856938977417828650f,
    0.514102744193221660f,  0.857728610000272120f,  0.512786400633563070f,
    0.858516224264442740f,  0.511468850437970520f,  0.859301818357008360f,
    0.510150096706766700f,  0.860085390429390140f,  0.508830142543106990f,
    0.860866938637767310f,  0.507508991052970870f,  0.861646461143081300f,
    0.506186645345155450f,  0.862423956111040500f,  0.504863108531267480f,
    0.863199421712124160f,  0.503538383725717580f,  0.863972856121586700f,
    0.502212474045710900f,  0.864744257519462380f,  0.500885382611240940f,
    0.865513624090568980f,  0.499557112545081890f,  0.866280954024512990f,
    0.498227666972781870f,  0.867046245515692650f,  0.496897049022654640f,
    0.867809496763303210f,  0.495565261825772490f,  0.868570705971340900f,
    0.494232308515959730f,  0.869329871348606730f,  0.492898192229784090f,
    0.870086991108711350f,  0.491562916106550060f,  0.870842063470078860f,
    0.490226483288291100f,  0.871595086655951090f,  0.488888896919763230f,
    0.872346058894391540f,  0.487550160148436050f,  0.873094978418290090f,
    0.486210276124486530f,  0.873841843465366750f,  0.484869248000791120f,
    0.874586652278176110f,  0.483527078932918740f,  0.875329403104110780f,
    0.482183772079122830f,  0.876070094195406600f,  0.480839330600333900f,
    0.876808723809145760f,  0.479493757660153010f,  0.877545290207261240f,
    0.478147056424843120f,  0.878279791656541460f,  0.476799230063322250f,
    0.879012226428633410f,  0.475450281747155870f,  0.879742592800047410f,
    0.474100214650550020f,  0.880470889052160750f,  0.472749031950342900f,
    0.881197113471221980f,  0.471396736825997810f,  0.881921264348354940f,
    0.470043332459595620f,  0.882643339979562790f,  0.468688822035827960f,
    0.883363338665731580f,  0.467333208741988530f,  0.884081258712634990f,
    0.465976495767966130f,  0.884797098430937790f,  0.464618686306237820f,
    0.885510856136199950f,  0.463259783551860260f,  0.886222530148880640f,
    0.461899790702462840f,  0.886932118794342080f,  0.460538710958240010f,
    0.887639620402853930f,  0.459176547521944150f,  0.888345033309596240f,
    0.457813303598877290f,  0.889048355854664570f,  0.456448982396883860f,
    0.889749586383072890f,  0.455083587126343840f,  0.890448723244757880f,
    0.453717121000163930f,  0.891145764794583180f,  0.452349587233771000f,
    0.891840709392342720f,  0.450980989045103810f,  0.892533555402764690f,
    0.449611329654606600f,  0.893224301195515320f,  0.448240612285220000f,
    0.893912945145203250f,  0.446868840162374330f,  0.894599485631382580f,
    0.445496016513981740f,  0.895283921038557580f,  0.444122144570429260f,
    0.895966249756185110f,  0.442747227564570130f,  0.896646470178680150f,
    0.441371268731716620f,  0.897324580705418320f,  0.439994271309633260f,
    0.898000579740739880f,  0.438616238538527710f,  0.898674465693953820f,
    0.437237173661044200f,  0.899346236979341460f,  0.435857079922255470f,
    0.900015892016160280f,  0.434475960569655710f,  0.900683429228646860f,
    0.433093818853152010f,  0.901348847046022030f,  0.431710658025057370f,
    0.902012143902493070f,  0.430326481340082610f,  0.902673318237258830f,
    0.428941292055329550f,  0.903332368494511820f,  0.427555093430282200f,
    0.903989293123443340f,  0.426167888726799620f,  0.904644090578246240f,
    0.424779681209108810f,  0.905296759318118820f,  0.423390474143796100f,
    0.905947297807268460f,  0.422000270799799790f,  0.906595704514915330f,
    0.420609074448402510f,  0.907241977915295930f,  0.419216888363223960f,
    0.907886116487666150f,  0.417823715820212380f,  0.908528118716306120f,
    0.416429560097637320f,  0.909167983090522270f,  0.415034424476081630f,
    0.909805708104652220f,  0.413638312238434560f,  0.910441292258067140f,
    0.412241226669883000f,  0.911074734055176250f,  0.410843171057903910f,
    0.911706032005429880f,  0.409444148692257590f,  0.912335184623322750f,
    0.408044162864978740f,  0.912962190428398100f,  0.406643216870369140f,
    0.913587047945250810f,  0.405241314004989860f,  0.914209755703530690f,
    0.403838457567654130f,  0.914830312237946090f,  0.402434650859418540f,
    0.915448716088267830f,  0.401029897183575790f,  0.916064965799331610f,
    0.399624199845646790f,  0.916679059921042700f,  0.398217562153373620f,
    0.917290997008377910f,  0.396809987416710420f,  0.917900775621390390f,
    0.395401478947816300f,  0.918508394325212250f,  0.393992040061048100f,
    0.919113851690057770f,  0.392581674072951530f,  0.919717146291227360f,
    0.391170384302253980f,  0.920318276709110480f,  0.389758174069856410f,
    0.920917241529189520f,  0.388345046698826300f,  0.921514039342041900f,
    0.386931005514388690f,  0.922108668743345070f,  0.385516053843919020f,
    0.922701128333878520f,  0.384100195016935040f,  0.923291416719527640f,
    0.382683432365089840f,  0.923879532511286740f,  0.381265769222162490f,
    0.924465474325262600f,  0.379847208924051110f,  0.925049240782677580f,
    0.378427754808765620f,  0.925630830509872720f,  0.377007410216418310f,
    0.926210242138311270f,  0.375586178489217330f,  0.926787474304581750f,
    0.374164062971457990f,  0.927362525650401110f,  0.372741067009515810f,
    0.927935394822617890f,  0.371317193951837600f,  0.928506080473215480f,
    0.369892447148934270f,  0.929074581259315750f,  0.368466829953372320f,
    0.929640895843181330f,  0.367040345719767240f,  0.930205022892219070f,
    0.365612997804773960f,  0.930766961078983710f,  0.364184789567079840f,
    0.931326709081180430f,  0.362755724367397230f,  0.931884265581668150f,
    0.361325805568454340f,  0.932439629268462360f,  0.359895036534988280f,
    0.932992798834738850f,  0.358463420633736540f,  0.933543772978836170f,
    0.357030961233430030f,  0.934092550404258870f,  0.355597661704783960f,
    0.934639129819680780f,  0.354163525420490510f,  0.935183509938947500f,
    0.352728555755210730f,  0.935725689481080370f,  0.351292756085567150f,
    0.936265667170278260f,  0.349856129790135030f,  0.936803441735921560f,
    0.348418680249434510f,  0.937339011912574960f,  0.346980410845923680f,
    0.937872376439989890f,  0.345541324963989150f,  0.938403534063108060f,
    0.344101425989938980f,  0.938932483532064490f,  0.342660717311994380f,
    0.939459223602189920f,  0.341219202320282410f,  0.939983753034013940f,
    0.339776884406826960f,  0.940506070593268300f,  0.338333766965541290f,
    0.941026175050889260f,  0.336889853392220050f,  0.941544065183020810f,
    0.335445147084531660f,  0.942059739771017310f,  0.333999651442009490f,
    0.942573197601446870f,  0.332553369866044220f,  0.943084437466093490f,
    0.331106305759876430f,  0.943593458161960390f,  0.329658462528587550f,
    0.944100258491272660f,  0.328209843579092660f,  0.944604837261480260f,
    0.326760452320131790f,  0.945107193285260610f,  0.325310292162262980f,
    0.945607325380521280f,  0.323859366517852960f,  0.946105232370403340f,
    0.322407678801070020f,  0.946600913083283530f,  0.320955232427875210f,
    0.947094366352777220f,  0.319502030816015750f,  0.947585591017741090f,
    0.318048077385015060f,  0.948074585922276230f,  0.316593375556165850f,
    0.948561349915730270f,  0.315137928752522440f,  0.949045881852700560f,
    0.313681740398891570f,  0.949528180593036670f,  0.312224813921825050f,
    0.950008245001843000f,  0.310767152749611470f,  0.950486073949481700f,
    0.309308760312268780f,  0.950961666311575080f,  0.307849640041534980f,
    0.951435020969008340f,  0.306389795370861080f,  0.951906136807932230f,
    0.304929229735402430f,  0.952375012719765880f,  0.303467946572011370f,
    0.952841647601198720f,  0.302005949319228200f,  0.953306040354193750f,
    0.300543241417273400f,  0.953768189885990330f,  0.299079826308040480f,
    0.954228095109105670f,  0.297615707435086310f,  0.954685754941338340f,
    0.296150888243623960f,  0.955141168305770670f,  0.294685372180514330f,
    0.955594334130771110f,  0.293219162694258680f,  0.956045251349996410f,
    0.291752263234989370f,  0.956493918902394990f,  0.290284677254462330f,
    0.956940335732208940f,  0.288816408206049480f,  0.957384500788975860f,
    0.287347459544729570f,  0.957826413027532910f,  0.285877834727080730f,
    0.958266071408017670f,  0.284407537211271820f,  0.958703474895871600f,
    0.282936570457055390f,  0.959138622461841890f,  0.281464937925758050f,
    0.959571513081984520f,  0.279992643080273380f,  0.960002145737665850f,
    0.278519689385053060f,  0.960430519415565790f,  0.277046080306099950f,
    0.960856633107679660f,  0.275571819310958250f,  0.961280485811320640f,
    0.274096909868706330f,  0.961702076529122540f,  0.272621355449948980f,
    0.962121404269041580f,  0.271145159526808070f,  0.962538468044359160f,
    0.269668325572915200f,  0.962953266873683880f,  0.268190857063403180f,
    0.963365799780954050f,  0.266712757474898420f,  0.963776065795439840f,
    0.265234030285511900f,  0.964184063951745720f,  0.263754678974831510f,
    0.964589793289812650f,  0.262274707023913590f,  0.964993252854920320f,
    0.260794117915275570f,  0.965394441697689400f,  0.259312915132886350f,
    0.965793358874083570f,  0.257831102162158930f,  0.966190003445412620f,
    0.256348682489942910f,  0.966584374478333120f,  0.254865659604514630f,
    0.966976471044852070f,  0.253382036995570270f,  0.967366292222328510f,
    0.251897818154216910f,  0.967753837093475510f,  0.250413006572965280f,
    0.968139104746362330f,  0.248927605745720260f,  0.968522094274417270f,
    0.247441619167773440f,  0.968902804776428870f,  0.245955050335794590f,
    0.969281235356548530f,  0.244467902747824210f,  0.969657385124292450f,
    0.242980179903263980f,  0.970031253194543970f,  0.241491885302869300f,
    0.970402838687555500f,  0.240003022448741500f,  0.970772140728950350f,
    0.238513594844318500f,  0.971139158449725090f,  0.237023605994367340f,
    0.971503890986251780f,  0.235533059404975460f,  0.971866337480279400f,
    0.234041958583543460f,  0.972226497078936270f,  0.232550307038775330f,
    0.972584368934732210f,  0.231058108280671280f,  0.972939952205560070f,
    0.229565365820518870f,  0.973293246054698250f,  0.228072083170885790f,
    0.973644249650811870f,  0.226578263845610110f,  0.973992962167955830f,
    0.225083911359792780f,  0.974339382785575860f,  0.223589029229790020f,
    0.974683510688510670f,  0.222093620973203590f,  0.975025345066994120f,
    0.220597690108873650f,  0.975364885116656870f,  0.219101240156869770f,
    0.975702130038528570f,  0.217604274638483670f,  0.976037079039039020f,
    0.216106797076219600f,  0.976369731330021140f,  0.214608810993786920f,
    0.976700086128711840f,  0.213110319916091360f,  0.977028142657754390f,
    0.211611327369227610f,  0.977353900145199960f,  0.210111836880469720f,
    0.977677357824509930f,  0.208611851978263460f,  0.977998514934557140f,
    0.207111376192218560f,  0.978317370719627650f,  0.205610413053099320f,
    0.978633924429423100f,  0.204108966092817010f,  0.978948175319062200f,
    0.202607038844421110f,  0.979260122649082020f,  0.201104634842091960f,
    0.979569765685440520f,  0.199601757621131050f,  0.979877103699517640f,
    0.198098410717953730f,  0.980182135968117320f,  0.196594597670080220f,
    0.980484861773469380f,  0.195090322016128330f,  0.980785280403230430f,
    0.193585587295803750f,  0.981083391150486590f,  0.192080397049892380f,
    0.981379193313754560f,  0.190574754820252800f,  0.981672686196983110f,
    0.189068664149806280f,  0.981963869109555240f,  0.187562128582529740f,
    0.982252741366289370f,  0.186055151663446630f,  0.982539302287441240f,
    0.184547736938619640f,  0.982823551198705240f,  0.183039887955141060f,
    0.983105487431216290f,  0.181531608261125130f,  0.983385110321551180f,
    0.180022901405699510f,  0.983662419211730250f,  0.178513770938997590f,
    0.983937413449218920f,  0.177004220412148860f,  0.984210092386929030f,
    0.175494253377271400f,  0.984480455383220930f,  0.173983873387463850f,
    0.984748501801904210f,  0.172473083996796030f,  0.985014231012239840f,
    0.170961888760301360f,  0.985277642388941220f,  0.169450291233967930f,
    0.985538735312176060f,  0.167938294974731230f,  0.985797509167567370f,
    0.166425903540464220f,  0.986053963346195440f,  0.164913120489970090f,
    0.986308097244598670f,  0.163399949382973230f,  0.986559910264775410f,
    0.161886393780111910f,  0.986809401814185420f,  0.160372457242928400f,
    0.987056571305750970f,  0.158858143333861390f,  0.987301418157858430f,
    0.157343455616238280f,  0.987543941794359230f,  0.155828397654265320f,
    0.987784141644572180f,  0.154312973013020240f,  0.988022017143283530f,
    0.152797185258443410f,  0.988257567730749460f,  0.151281037957330250f,
    0.988490792852696590f,  0.149764534677321620f,  0.988721691960323780f,
    0.148247678986896200f,  0.988950264510302990f,  0.146730474455361750f,
    0.989176509964781010f,  0.145212924652847520f,  0.989400427791380380f,
    0.143695033150294580f,  0.989622017463200780f,  0.142176803519448000f,
    0.989841278458820530f,  0.140658239332849240f,  0.990058210262297120f,
    0.139139344163826280f,  0.990272812363169110f,  0.137620121586486180f,
    0.990485084256456980f,  0.136100575175706200f,  0.990695025442664630f,
    0.134580708507126220f,  0.990902635427780010f,  0.133060525157139180f,
    0.991107913723276780f,  0.131540028702883280f,  0.991310859846115440f,
    0.130019222722233350f,  0.991511473318743900f,  0.128498110793793220f,
    0.991709753669099530f,  0.126976696496885980f,  0.991905700430609330f,
    0.125454983411546210f,  0.992099313142191800f,  0.123932975118512200f,
    0.992290591348257370f,  0.122410675199216280f,  0.992479534598709970f,
    0.120888087235777220f,  0.992666142448948020f,  0.119365214810991350f,
    0.992850414459865100f,  0.117842061508325020f,  0.993032350197851410f,
    0.116318630911904880f,  0.993211949234794500f,  0.114794926606510250f,
    0.993389211148080650f,  0.113270952177564360f,  0.993564135520595300f,
    0.111746711211126660f,  0.993736721940724600f,  0.110222207293883180f,
    0.993906970002356060f,  0.108697444013138670f,  0.994074879304879370f,
    0.107172424956808870f,  0.994240449453187900f,  0.105647153713410700f,
    0.994403680057679100f,  0.104121633872054730f,  0.994564570734255420f,
    0.102595869022436280f,  0.994723121104325700f,  0.101069862754827880f,
    0.994879330794805620f,  0.099543618660069444f,  0.995033199438118630f,
    0.098017140329560770f,  0.995184726672196820f,  0.096490431355252607f,
    0.995333912140482280f,  0.094963495329639061f,  0.995480755491926940f,
    0.093436335845747912f,  0.995625256380994310f,  0.091908956497132696f,
    0.995767414467659820f,  0.090381360877865011f,  0.995907229417411720f,
    0.088853552582524684f,  0.996044700901251970f,  0.087325535206192226f,
    0.996179828595696870f,  0.085797312344439880f,  0.996312612182778000f,
    0.084268887593324127f,  0.996443051350042630f,  0.082740264549375803f,
    0.996571145790554840f,  0.081211446809592386f,  0.996696895202896060f,
    0.079682437971430126f,  0.996820299291165670f,  0.078153241632794315f,
    0.996941357764982160f,  0.076623861392031617f,  0.997060070339482960f,
    0.075094300847921291f,  0.997176436735326190f,  0.073564563599667454f,
    0.997290456678690210f,  0.072034653246889416f,  0.997402129901275300f,
    0.070504573389614009f,  0.997511456140303450f,  0.068974327628266732f,
    0.997618435138519550f,  0.067443919563664106f,  0.997723066644191640f,
    0.065913352797003930f,  0.997825350411111640f,  0.064382630929857410f,
    0.997925286198596000f,  0.062851757564161420f,  0.998022873771486240f,
    0.061320736302208648f,  0.998118112900149180f,  0.059789570746640007f,
    0.998211003360478190f,  0.058258264500435732f,  0.998301544933892890f,
    0.056726821166907783f,  0.998389737407340160f,  0.055195244349690031f,
    0.998475580573294770f,  0.053663537652730679f,  0.998559074229759310f,
    0.052131704680283317f,  0.998640218180265270f,  0.050599749036899337f,
    0.998719012233872940f,  0.049067674327418126f,  0.998795456205172410f,
    0.047535484156959261f,  0.998869549914283560f,  0.046003182130914644f,
    0.998941293186856870f,  0.044470771854938744f,  0.999010685854073380f,
    0.042938256934940959f,  0.999077727752645360f,  0.041405640977076712f,
    0.999142418724816910f,  0.039872927587739845f,  0.999204758618363890f,
    0.038340120373552791f,  0.999264747286594420f,  0.036807222941358991f,
    0.999322384588349540f,  0.035274238898213947f,  0.999377670388002850f,
    0.033741171851377642f,  0.999430604555461730f,  0.032208025408304704f,
    0.999481186966166950f,  0.030674803176636581f,  0.999529417501093140f,
    0.029141508764193740f,  0.999575296046749220f,  0.027608145778965820f,
    0.999618822495178640f,  0.026074717829104040f,  0.999659996743959220f,
    0.024541228522912264f,  0.999698818696204250f,  0.023007681468839410f,
    0.999735288260561680f,  0.021474080275469605f,  0.999769405351215280f,
    0.019940428551514598f,  0.999801169887884260f,  0.018406729905804820f,
    0.999830581795823400f,  0.016872987947281773f,  0.999857641005823860f,
    0.015339206284988220f,  0.999882347454212560f,  0.013805388528060349f,
    0.999904701082852900f,  0.012271538285719944f,  0.999924701839144500f,
    0.010737659167264572f,  0.999942349676023910f,  0.009203754782059960f,
    0.999957644551963900f,  0.007669828739531077f,  0.999970586430974140f,
    0.006135884649154515f,  0.999981175282601110f,  0.004601926120448672f,
    0.999989411081928400f,  0.003067956762966138f,  0.999995293809576190f,
    0.001533980186284766f,  0.999998823451701880f,  0.000000000000000061f,
    1.000000000000000000f,  -0.001533980186284644f, 0.999998823451701880f,
    -0.003067956762966016f, 0.999995293809576190f,  -0.004601926120448550f,
    0.999989411081928400f,  -0.006135884649154393f, 0.999981175282601110f,
    -0.007669828739530955f, 0.999970586430974140f,  -0.009203754782059837f,
    0.999957644551963900f,  -0.010737659167264449f, 0.999942349676023910f,
    -0.012271538285719823f, 0.999924701839144500f,  -0.013805388528060226f,
    0.999904701082852900f,  -0.015339206284988098f, 0.999882347454212560f,
    -0.016872987947281651f, 0.999857641005823860f,  -0.018406729905804695f,
    0.999830581795823400f,  -0.019940428551514476f, 0.999801169887884260f,
    -0.021474080275469484f, 0.999769405351215280f,  -0.023007681468839289f,
    0.999735288260561680f,  -0.024541228522912142f, 0.999698818696204250f,
    -0.026074717829103915f, 0.999659996743959220f,  -0.027608145778965698f,
    0.999618822495178640f,  -0.029141508764193618f, 0.999575296046749220f,
    -0.030674803176636459f, 0.999529417501093140f,  -0.032208025408304579f,
    0.999481186966166950f,  -0.033741171851377517f, 0.999430604555461730f,
    -0.035274238898213822f, 0.999377670388002850f,  -0.036807222941358866f,
    0.999322384588349540f,  -0.038340120373552666f, 0.999264747286594420f,
    -0.039872927587739727f, 0.999204758618363890f,  -0.041405640977076594f,
    0.999142418724816910f,  -0.042938256934940834f, 0.999077727752645360f,
    -0.044470771854938619f, 0.999010685854073380f,  -0.046003182130914519f,
    0.998941293186856870f,  -0.047535484156959136f, 0.998869549914283560f,
    -0.049067674327418008f, 0.998795456205172410f,  -0.050599749036899212f,
    0.998719012233872940f,  -0.052131704680283192f, 0.998640218180265270f,
    -0.053663537652730554f, 0.998559074229759310f,  -0.055195244349689913f,
    0.998475580573294770f,  -0.056726821166907658f, 0.998389737407340160f,
    -0.058258264500435607f, 0.998301544933892890f,  -0.059789570746639882f,
    0.998211003360478190f,  -0.061320736302208530f, 0.998118112900149180f,
    -0.062851757564161309f, 0.998022873771486240f,  -0.064382630929857285f,
    0.997925286198596000f,  -0.065913352797003805f, 0.997825350411111640f,
    -0.067443919563663982f, 0.997723066644191640f,  -0.068974327628266607f,
    0.997618435138519550f,  -0.070504573389613898f, 0.997511456140303450f,
    -0.072034653246889291f, 0.997402129901275300f,  -0.073564563599667329f,
    0.997290456678690210f,  -0.075094300847921167f, 0.997176436735326190f,
    -0.076623861392031506f, 0.997060070339482960f,  -0.078153241632794190f,
    0.996941357764982160f,  -0.079682437971430015f, 0.996820299291165780f,
    -0.081211446809592261f, 0.996696895202896060f,  -0.082740264549375678f,
    0.996571145790554840f,  -0.084268887593324002f, 0.996443051350042630f,
    -0.085797312344439755f, 0.996312612182778000f,  -0.087325535206192101f,
    0.996179828595696870f,  -0.088853552582524559f, 0.996044700901251970f,
    -0.090381360877864886f, 0.995907229417411720f,  -0.091908956497132571f,
    0.995767414467659820f,  -0.093436335845747787f, 0.995625256380994310f,
    -0.094963495329638950f, 0.995480755491926940f,  -0.096490431355252482f,
    0.995333912140482280f,  -0.098017140329560645f, 0.995184726672196930f,
    -0.099543618660069319f, 0.995033199438118630f,  -0.101069862754827750f,
    0.994879330794805620f,  -0.102595869022436160f, 0.994723121104325700f,
    -0.104121633872054600f, 0.994564570734255420f,  -0.105647153713410570f,
    0.994403680057679100f,  -0.107172424956808760f, 0.994240449453187900f,
    -0.108697444013138560f, 0.994074879304879480f,  -0.110222207293883060f,
    0.993906970002356060f,  -0.111746711211126550f, 0.993736721940724600f,
    -0.113270952177564240f, 0.993564135520595300f,  -0.114794926606510130f,
    0.993389211148080650f,  -0.116318630911904750f, 0.993211949234794500f,
    -0.117842061508324890f, 0.993032350197851410f,  -0.119365214810991230f,
    0.992850414459865100f,  -0.120888087235777100f, 0.992666142448948020f,
    -0.122410675199216150f, 0.992479534598709970f,  -0.123932975118512080f,
    0.992290591348257370f,  -0.125454983411546070f, 0.992099313142191800f,
    -0.126976696496885870f, 0.991905700430609330f,  -0.128498110793793110f,
    0.991709753669099530f,  -0.130019222722233240f, 0.991511473318744010f,
    -0.131540028702883140f, 0.991310859846115440f,  -0.133060525157139040f,
    0.991107913723276890f,  -0.134580708507126110f, 0.990902635427780010f,
    -0.136100575175706060f, 0.990695025442664630f,  -0.137620121586486070f,
    0.990485084256456980f,  -0.139139344163826170f, 0.990272812363169110f,
    -0.140658239332849130f, 0.990058210262297120f,  -0.142176803519447890f,
    0.989841278458820530f,  -0.143695033150294440f, 0.989622017463200890f,
    -0.145212924652847410f, 0.989400427791380380f,  -0.146730474455361640f,
    0.989176509964781010f,  -0.148247678986896090f, 0.988950264510302990f,
    -0.149764534677321510f, 0.988721691960323780f,  -0.151281037957330140f,
    0.988490792852696700f,  -0.152797185258443300f, 0.988257567730749460f,
    -0.154312973013020130f, 0.988022017143283530f,  -0.155828397654265200f,
    0.987784141644572180f,  -0.157343455616238160f, 0.987543941794359340f,
    -0.158858143333861280f, 0.987301418157858430f,  -0.160372457242928260f,
    0.987056571305750970f,  -0.161886393780111770f, 0.986809401814185530f,
    -0.163399949382973110f, 0.986559910264775520f,  -0.164913120489969950f,
    0.986308097244598670f,  -0.166425903540464100f, 0.986053963346195440f,
    -0.167938294974731090f, 0.985797509167567480f,  -0.169450291233967820f,
    0.985538735312176060f,  -0.170961888760301240f, 0.985277642388941220f,
    -0.172473083996795920f, 0.985014231012239840f,  -0.173983873387463710f,
    0.984748501801904210f,  -0.175494253377271260f, 0.984480455383220930f,
    -0.177004220412148750f, 0.984210092386929030f,  -0.178513770938997450f,
    0.983937413449218920f,  -0.180022901405699400f, 0.983662419211730250f,
    -0.181531608261125020f, 0.983385110321551180f,  -0.183039887955140920f,
    0.983105487431216290f,  -0.184547736938619530f, 0.982823551198705350f,
    -0.186055151663446490f, 0.982539302287441240f,  -0.187562128582529600f,
    0.982252741366289370f,  -0.189068664149806160f, 0.981963869109555240f,
    -0.190574754820252660f, 0.981672686196983110f,  -0.192080397049892270f,
    0.981379193313754560f,  -0.193585587295803610f, 0.981083391150486710f,
    -0.195090322016128190f, 0.980785280403230430f,  -0.196594597670080110f,
    0.980484861773469380f,  -0.198098410717953620f, 0.980182135968117430f,
    -0.199601757621130940f, 0.979877103699517640f,  -0.201104634842091820f,
    0.979569765685440520f,  -0.202607038844420970f, 0.979260122649082130f,
    -0.204108966092816900f, 0.978948175319062200f,  -0.205610413053099210f,
    0.978633924429423210f,  -0.207111376192218450f, 0.978317370719627650f,
    -0.208611851978263320f, 0.977998514934557140f,  -0.210111836880469610f,
    0.977677357824509930f,  -0.211611327369227500f, 0.977353900145200070f,
    -0.213110319916091250f, 0.977028142657754390f,  -0.214608810993786810f,
    0.976700086128711840f,  -0.216106797076219490f, 0.976369731330021140f,
    -0.217604274638483560f, 0.976037079039039130f,  -0.219101240156869660f,
    0.975702130038528570f,  -0.220597690108873530f, 0.975364885116656980f,
    -0.222093620973203480f, 0.975025345066994120f,  -0.223589029229789880f,
    0.974683510688510670f,  -0.225083911359792670f, 0.974339382785575860f,
    -0.226578263845610000f, 0.973992962167955830f,  -0.228072083170885680f,
    0.973644249650811980f,  -0.229565365820518760f, 0.973293246054698250f,
    -0.231058108280671140f, 0.972939952205560180f,  -0.232550307038775220f,
    0.972584368934732210f,  -0.234041958583543320f, 0.972226497078936380f,
    -0.235533059404975350f, 0.971866337480279400f,  -0.237023605994367230f,
    0.971503890986251780f,  -0.238513594844318390f, 0.971139158449725090f,
    -0.240003022448741390f, 0.970772140728950350f,  -0.241491885302869160f,
    0.970402838687555500f,  -0.242980179903263870f, 0.970031253194543970f,
    -0.244467902747824100f, 0.969657385124292450f,  -0.245955050335794480f,
    0.969281235356548530f,  -0.247441619167773320f, 0.968902804776428870f,
    -0.248927605745720120f, 0.968522094274417380f,  -0.250413006572965170f,
    0.968139104746362440f,  -0.251897818154216800f, 0.967753837093475510f,
    -0.253382036995570160f, 0.967366292222328510f,  -0.254865659604514520f,
    0.966976471044852070f,  -0.256348682489942800f, 0.966584374478333120f,
    -0.257831102162158820f, 0.966190003445412620f,  -0.259312915132886230f,
    0.965793358874083680f,  -0.260794117915275460f, 0.965394441697689400f,
    -0.262274707023913480f, 0.964993252854920440f,  -0.263754678974831400f,
    0.964589793289812760f,  -0.265234030285511790f, 0.964184063951745830f,
    -0.266712757474898310f, 0.963776065795439840f,  -0.268190857063403010f,
    0.963365799780954050f,  -0.269668325572915090f, 0.962953266873683880f,
    -0.271145159526807960f, 0.962538468044359160f,  -0.272621355449948870f,
    0.962121404269041580f,  -0.274096909868706220f, 0.961702076529122540f,
    -0.275571819310958140f, 0.961280485811320640f,  -0.277046080306099840f,
    0.960856633107679660f,  -0.278519689385052950f, 0.960430519415565900f,
    -0.279992643080273270f, 0.960002145737665850f,  -0.281464937925757940f,
    0.959571513081984520f,  -0.282936570457055280f, 0.959138622461842010f,
    -0.284407537211271710f, 0.958703474895871600f,  -0.285877834727080620f,
    0.958266071408017670f,  -0.287347459544729460f, 0.957826413027532910f,
    -0.288816408206049370f, 0.957384500788975970f,  -0.290284677254462160f,
    0.956940335732208940f,  -0.291752263234989260f, 0.956493918902395100f,
    -0.293219162694258570f, 0.956045251349996520f,  -0.294685372180514220f,
    0.955594334130771110f,  -0.296150888243623840f, 0.955141168305770670f,
    -0.297615707435086200f, 0.954685754941338340f,  -0.299079826308040360f,
    0.954228095109105670f,  -0.300543241417273290f, 0.953768189885990330f,
    -0.302005949319228080f, 0.953306040354193860f,  -0.303467946572011260f,
    0.952841647601198720f,  -0.304929229735402260f, 0.952375012719765880f,
    -0.306389795370860970f, 0.951906136807932350f,  -0.307849640041534870f,
    0.951435020969008340f,  -0.309308760312268620f, 0.950961666311575080f,
    -0.310767152749611360f, 0.950486073949481810f,  -0.312224813921824940f,
    0.950008245001843000f,  -0.313681740398891410f, 0.949528180593036670f,
    -0.315137928752522330f, 0.949045881852700670f,  -0.316593375556165730f,
    0.948561349915730380f,  -0.318048077385014950f, 0.948074585922276230f,
    -0.319502030816015640f, 0.947585591017741200f,  -0.320955232427875100f,
    0.947094366352777220f,  -0.322407678801069850f, 0.946600913083283530f,
    -0.323859366517852850f, 0.946105232370403450f,  -0.325310292162262870f,
    0.945607325380521390f,  -0.326760452320131620f, 0.945107193285260610f,
    -0.328209843579092550f, 0.944604837261480260f,  -0.329658462528587440f,
    0.944100258491272660f,  -0.331106305759876320f, 0.943593458161960390f,
    -0.332553369866044060f, 0.943084437466093490f,  -0.333999651442009380f,
    0.942573197601446870f,  -0.335445147084531550f, 0.942059739771017420f,
    -0.336889853392219940f, 0.941544065183020810f,  -0.338333766965541180f,
    0.941026175050889260f,  -0.339776884406826850f, 0.940506070593268300f,
    -0.341219202320282300f, 0.939983753034014050f,  -0.342660717311994270f,
    0.939459223602189920f,  -0.344101425989938870f, 0.938932483532064490f,
    -0.345541324963989040f, 0.938403534063108170f,  -0.346980410845923570f,
    0.937872376439989890f,  -0.348418680249434400f, 0.937339011912574960f,
    -0.349856129790134920f, 0.936803441735921560f,  -0.351292756085567040f,
    0.936265667170278260f,  -0.352728555755210620f, 0.935725689481080370f,
    -0.354163525420490400f, 0.935183509938947610f,  -0.355597661704783850f,
    0.934639129819680780f,  -0.357030961233429920f, 0.934092550404258980f,
    -0.358463420633736430f, 0.933543772978836280f,  -0.359895036534988170f,
    0.932992798834738850f,  -0.361325805568454230f, 0.932439629268462360f,
    -0.362755724367397110f, 0.931884265581668150f,  -0.364184789567079730f,
    0.931326709081180540f,  -0.365612997804773850f, 0.930766961078983710f,
    -0.367040345719767120f, 0.930205022892219070f,  -0.368466829953372210f,
    0.929640895843181330f,  -0.369892447148934160f, 0.929074581259315750f,
    -0.371317193951837490f, 0.928506080473215590f,  -0.372741067009515700f,
    0.927935394822617890f,  -0.374164062971457880f, 0.927362525650401110f,
    -0.375586178489217220f, 0.926787474304581750f,  -0.377007410216418200f,
    0.926210242138311380f,  -0.378427754808765450f, 0.925630830509872830f,
    -0.379847208924050990f, 0.925049240782677700f,  -0.381265769222162380f,
    0.924465474325262600f,  -0.382683432365089730f, 0.923879532511286740f,
    -0.384100195016934930f, 0.923291416719527750f,  -0.385516053843918900f,
    0.922701128333878520f,  -0.386931005514388580f, 0.922108668743345180f,
    -0.388345046698826190f, 0.921514039342042010f,  -0.389758174069856300f,
    0.920917241529189520f,  -0.391170384302253870f, 0.920318276709110590f,
    -0.392581674072951410f, 0.919717146291227360f,  -0.393992040061047990f,
    0.919113851690057770f,  -0.395401478947816190f, 0.918508394325212250f,
    -0.396809987416710310f, 0.917900775621390500f,  -0.398217562153373510f,
    0.917290997008378020f,  -0.399624199845646680f, 0.916679059921042700f,
    -0.401029897183575680f, 0.916064965799331720f,  -0.402434650859418430f,
    0.915448716088267830f,  -0.403838457567654020f, 0.914830312237946200f,
    -0.405241314004989750f, 0.914209755703530690f,  -0.406643216870369030f,
    0.913587047945250810f,  -0.408044162864978630f, 0.912962190428398210f,
    -0.409444148692257480f, 0.912335184623322860f,  -0.410843171057903800f,
    0.911706032005429880f,  -0.412241226669882890f, 0.911074734055176360f,
    -0.413638312238434450f, 0.910441292258067250f,  -0.415034424476081520f,
    0.909805708104652330f,  -0.416429560097636990f, 0.909167983090522490f,
    -0.417823715820212270f, 0.908528118716306120f,  -0.419216888363224070f,
    0.907886116487666150f,  -0.420609074448402400f, 0.907241977915295930f,
    -0.422000270799799680f, 0.906595704514915330f,  -0.423390474143795770f,
    0.905947297807268570f,  -0.424779681209108690f, 0.905296759318118820f,
    -0.426167888726799670f, 0.904644090578246130f,  -0.427555093430281860f,
    0.903989293123443450f,  -0.428941292055329440f, 0.903332368494511820f,
    -0.430326481340082720f, 0.902673318237258830f,  -0.431710658025057090f,
    0.902012143902493290f,  -0.433093818853151900f, 0.901348847046022030f,
    -0.434475960569655820f, 0.900683429228646860f,  -0.435857079922255360f,
    0.900015892016160280f,  -0.437237173661044090f, 0.899346236979341570f,
    -0.438616238538527380f, 0.898674465693953930f,  -0.439994271309633140f,
    0.898000579740739880f,  -0.441371268731716730f, 0.897324580705418320f,
    -0.442747227564569800f, 0.896646470178680270f,  -0.444122144570429140f,
    0.895966249756185220f,  -0.445496016513981800f, 0.895283921038557470f,
    -0.446868840162373990f, 0.894599485631382810f,  -0.448240612285219890f,
    0.893912945145203250f,  -0.449611329654606710f, 0.893224301195515210f,
    -0.450980989045103700f, 0.892533555402764690f,  -0.452349587233770890f,
    0.891840709392342720f,  -0.453717121000163590f, 0.891145764794583410f,
    -0.455083587126343720f, 0.890448723244757990f,  -0.456448982396883970f,
    0.889749586383072780f,  -0.457813303598877010f, 0.889048355854664680f,
    -0.459176547521944030f, 0.888345033309596350f,  -0.460538710958240060f,
    0.887639620402853930f,  -0.461899790702462560f, 0.886932118794342310f,
    -0.463259783551860150f, 0.886222530148880640f,  -0.464618686306237930f,
    0.885510856136199840f,  -0.465976495767966010f, 0.884797098430937900f,
    -0.467333208741988420f, 0.884081258712634990f,  -0.468688822035827680f,
    0.883363338665731690f,  -0.470043332459595510f, 0.882643339979562900f,
    -0.471396736825997700f, 0.881921264348355050f,  -0.472749031950342570f,
    0.881197113471222200f,  -0.474100214650549910f, 0.880470889052160870f,
    -0.475450281747155980f, 0.879742592800047410f,  -0.476799230063321920f,
    0.879012226428633530f,  -0.478147056424843010f, 0.878279791656541580f,
    -0.479493757660153120f, 0.877545290207261240f,  -0.480839330600333790f,
    0.876808723809145760f,  -0.482183772079122720f, 0.876070094195406600f,
    -0.483527078932918460f, 0.875329403104111000f,  -0.484869248000791010f,
    0.874586652278176220f,  -0.486210276124486420f, 0.873841843465366860f,
    -0.487550160148435720f, 0.873094978418290200f,  -0.488888896919763120f,
    0.872346058894391540f,  -0.490226483288291210f, 0.871595086655950980f,
    -0.491562916106549790f, 0.870842063470078980f,  -0.492898192229783980f,
    0.870086991108711460f,  -0.494232308515959840f, 0.869329871348606730f,
    -0.495565261825772370f, 0.868570705971341010f,  -0.496897049022654520f,
    0.867809496763303210f,  -0.498227666972781590f, 0.867046245515692760f,
    -0.499557112545081780f, 0.866280954024513110f,  -0.500885382611240830f,
    0.865513624090569090f,  -0.502212474045710570f, 0.864744257519462490f,
    -0.503538383725717460f, 0.863972856121586810f,  -0.504863108531267590f,
    0.863199421712124160f,  -0.506186645345155120f, 0.862423956111040610f,
    -0.507508991052970760f, 0.861646461143081300f,  -0.508830142543107100f,
    0.860866938637767200f,  -0.510150096706766590f, 0.860085390429390250f,
    -0.511468850437970410f, 0.859301818357008360f,  -0.512786400633562730f,
    0.858516224264442960f,  -0.514102744193221660f, 0.857728610000272120f,
    -0.515417878019463040f, 0.856938977417828760f,  -0.516731799017649650f,
    0.856147328375194580f,  -0.518044504095999230f, 0.855353664735196030f,
    -0.519355990165589640f, 0.854557988365400530f,  -0.520666254140366940f,
    0.853760301138111520f,  -0.521975292937154280f, 0.852960604930363740f,
    -0.523283103475656540f, 0.852158901623919720f,  -0.524589682678468730f,
    0.851355193105265200f,  -0.525895027471084630f, 0.850549481265603480f,
    -0.527199134781901060f, 0.849741768000852660f,  -0.528502001542228370f,
    0.848932055211639720f,  -0.529803624686294720f, 0.848120344803297230f,
    -0.531104001151254780f, 0.847306638685858540f,  -0.532403127877197900f,
    0.846490938774052130f,  -0.533701001807152960f, 0.845673246987299070f,
    -0.534997619887097040f, 0.844853565249707230f,  -0.536292979065963070f,
    0.844031895490066410f,  -0.537587076295645620f, 0.843208239641845440f,
    -0.538879908531008310f, 0.842382599643185960f,  -0.540171472729892850f,
    0.841554977436898440f,  -0.541461765853123220f, 0.840725374970458180f,
    -0.542750784864515780f, 0.839893794195999630f,  -0.544038526730883930f,
    0.839060237070312630f,  -0.545324988422046240f, 0.838224705554838190f,
    -0.546610166910834860f, 0.837387201615661940f,  -0.547894059173100190f,
    0.836547727223512010f,  -0.549176662187719540f, 0.835706284353752720f,
    -0.550457972936604700f, 0.834862874986380120f,  -0.551737988404707450f,
    0.834017501106018020f,  -0.553016705580027360f, 0.833170164701913300f,
    -0.554294121453620110f, 0.832320867767929680f,  -0.555570233019601960f,
    0.831469612302545460f,  -0.556845037275159990f, 0.830616400308846310f,
    -0.558118531220556100f, 0.829761233794523050f,  -0.559390711859135800f,
    0.828904114771865100f,  -0.560661576197335920f, 0.828045045257755800f,
    -0.561931121244689470f, 0.827184027273669130f,  -0.563199344013833980f,
    0.826321062845663650f,  -0.564466241520519390f, 0.825456154004377550f,
    -0.565731810783613230f, 0.824589302785025180f,  -0.566996048825108460f,
    0.823720511227391540f,  -0.568258952670131490f, 0.822849781375826320f,
    -0.569520519346947250f, 0.821977115279241440f,  -0.570780745886967140f,
    0.821102514991104760f,  -0.572039629324757050f, 0.820225982569434690f,
    -0.573297166698041980f, 0.819347520076797120f,  -0.574553355047715760f,
    0.818467129580298770f,  -0.575808191417845340f, 0.817584813151583710f,
    -0.577061672855679330f, 0.816700572866827960f,  -0.578313796411655480f,
    0.815814410806733780f,  -0.579564559139405850f, 0.814926329056526510f,
    -0.580813958095764420f, 0.814036329705948520f,  -0.582061990340775550f,
    0.813144414849253590f,  -0.583308652937698400f, 0.812250586585203880f,
    -0.584553942953015220f, 0.811354847017063840f,  -0.585797857456438860f,
    0.810457198252594770f,  -0.587040393520917750f, 0.809557642404051480f,
    -0.588281548222645220f, 0.808656181588175090f,  -0.589521318641063940f,
    0.807752817926190360f,  -0.590759701858874050f, 0.806847553543799450f,
    -0.591996694962040880f, 0.805940390571176390f,  -0.593232295039799910f,
    0.805031331142963550f,  -0.594466499184664320f, 0.804120377398265810f,
    -0.595699304492433360f, 0.803207531480644940f,  -0.596930708062196610f,
    0.802292795538115610f,  -0.598160706996342160f, 0.801376171723140350f,
    -0.599389298400564540f, 0.800457662192622820f,  -0.600616479383868750f,
    0.799537269107905240f,  -0.601842247058579920f, 0.798614994634760930f,
    -0.603066598540348280f, 0.797690840943391040f,  -0.604289530948155850f,
    0.796764810208418940f,  -0.605511041404325430f, 0.795836904608883570f,
    -0.606731127034524590f, 0.794907126328236900f,  -0.607949784967773520f,
    0.793975477554337280f,  -0.609167012336453210f, 0.793041960479443640f,
    -0.610382806276309590f, 0.792106577300212280f,  -0.611597163926461800f,
    0.791169330217690310f,  -0.612810082429409710f, 0.790230221437310030f,
    -0.614021558931038160f, 0.789289253168885870f,  -0.615231590580626710f,
    0.788346427626606340f,  -0.616440174530853650f, 0.787401747029031320f,
    -0.617647307937803760f, 0.786455213599085880f,  -0.618852987960976210f,
    0.785506829564054040f,  -0.620057211763289210f, 0.784556597155575130f,
    -0.621259976511087440f, 0.783604518609638310f,  -0.622461279374149970f,
    0.782650596166575730f,  -0.623661117525694640f, 0.781694832071059280f,
    -0.624859488142386230f, 0.780737228572094600f,  -0.626056388404343520f,
    0.779777787923014440f,  -0.627251815495143860f, 0.778816512381476090f,
    -0.628445766601832600f, 0.777853404209453150f,  -0.629638238914927100f,
    0.776888465673232440f,  -0.630829229628424360f, 0.775921699043407800f,
    -0.632018735939808950f, 0.774953106594873930f,  -0.633206755050057300f,
    0.773982690606822790f,  -0.634393284163645380f, 0.773010453362737100f,
    -0.635578320488556110f, 0.772036397150384520f,  -0.636761861236284310f,
    0.771060524261813710f,  -0.637943903621843940f, 0.770082836993348010f,
    -0.639124444863775730f, 0.769103337645579590f,  -0.640303482184151450f,
    0.768122028523365530f,  -0.641481012808583050f, 0.767138911935820510f,
    -0.642657033966226860f, 0.766153990196312920f,  -0.643831542889791280f,
    0.765167265622459070f,  -0.645004536815543930f, 0.764178740536116790f,
    -0.646176012983316390f, 0.763188417263381270f,  -0.647345968636511950f,
    0.762196298134579010f,  -0.648514401022112440f, 0.761202385484261890f,
    -0.649681307390683300f, 0.760206681651202310f,  -0.650846684996380760f,
    0.759209188978388180f,  -0.652010531096959500f, 0.758209909813015280f,
    -0.653172842953776530f, 0.757208846506484680f,  -0.654333617831800440f,
    0.756206001414394540f,  -0.655492852999615460f, 0.755201376896536550f,
    -0.656650545729428830f, 0.754194975316889280f,  -0.657806693297078640f,
    0.753186799043612520f,  -0.658961292982037430f, 0.752176850449042700f,
    -0.660114342067420370f, 0.751165131909686590f,  -0.661265837839992150f,
    0.750151645806215070f,  -0.662415777590171890f, 0.749136394523459260f,
    -0.663564158612039660f, 0.748119380450403710f,  -0.664710978203344900f,
    0.747100605980180130f,  -0.665856233665509500f, 0.746080073510064000f,
    -0.666999922303637360f, 0.745057785441466060f,  -0.668142041426518560f,
    0.744033744179929290f,  -0.669282588346635900f, 0.743007952135121830f,
    -0.670421560380173090f, 0.741980411720831070f,  -0.671558954847018440f,
    0.740951125354958990f,  -0.672694769070772750f, 0.739920095459516310f,
    -0.673829000378756040f, 0.738887324460615220f,  -0.674961646102012150f,
    0.737852814788465870f,  -0.676092703575315810f, 0.736816568877370020f,
    -0.677222170137180450f, 0.735778589165713480f,  -0.678350043129861250f,
    0.734738878095963610f,  -0.679476319899364970f, 0.733697438114660370f,
    -0.680600997795453020f, 0.732654271672412820f,  -0.681724074171649600f,
    0.731609381223892740f,  -0.682845546385247970f, 0.730562769227827590f,
    -0.683965411797315510f, 0.729514438146997010f,  -0.685083667772700240f,
    0.728464390448225310f,  -0.686200311680038590f, 0.727412628602375770f,
    -0.687315340891759160f, 0.726359155084345900f,  -0.688428752784090330f,
    0.725303972373060880f,  -0.689540544737066940f, 0.724247082951466890f,
    -0.690650714134534380f, 0.723188489306527570f,  -0.691759258364157640f,
    0.722128193929215460f,  -0.692866174817424740f, 0.721066199314508110f,
    -0.693971460889653780f, 0.720002507961381770f,  -0.695075113980000770f,
    0.718937122372804490f,  -0.696177131491462990f, 0.717870045055731710f,
    -0.697277510830886400f, 0.716801278521099650f,  -0.698376249408972800f,
    0.715730825283818710f,  -0.699473344640283880f, 0.714658687862768980f,
    -0.700568793943248220f, 0.713584868780793750f,  -0.701662594740168450f,
    0.712509370564692320f,  -0.702754744457225080f, 0.711432195745216660f,
    -0.703845240524484830f, 0.710353346857062420f,  -0.704934080375904880f,
    0.709272826438865580f,  -0.706021261449339630f, 0.708190637033195510f,
    -0.707106781186547460f, 0.707106781186547570f,  -0.708190637033195400f,
    0.706021261449339740f,  -0.709272826438865470f, 0.704934080375905100f,
    -0.710353346857062310f, 0.703845240524485050f,  -0.711432195745216540f,
    0.702754744457225190f,  -0.712509370564692210f, 0.701662594740168680f,
    -0.713584868780793640f, 0.700568793943248340f,  -0.714658687862768870f,
    0.699473344640283990f,  -0.715730825283818590f, 0.698376249408972920f,
    -0.716801278521099540f, 0.697277510830886520f,  -0.717870045055731600f,
    0.696177131491463100f,  -0.718937122372804380f, 0.695075113980000990f,
    -0.720002507961381650f, 0.693971460889654000f,  -0.721066199314507990f,
    0.692866174817424850f,  -0.722128193929215230f, 0.691759258364157860f,
    -0.723188489306527460f, 0.690650714134534600f,  -0.724247082951466780f,
    0.689540544737067050f,  -0.725303972373060770f, 0.688428752784090440f,
    -0.726359155084345790f, 0.687315340891759270f,  -0.727412628602375650f,
    0.686200311680038700f,  -0.728464390448225200f, 0.685083667772700360f,
    -0.729514438146996790f, 0.683965411797315630f,  -0.730562769227827480f,
    0.682845546385248190f,  -0.731609381223892630f, 0.681724074171649710f,
    -0.732654271672412700f, 0.680600997795453240f,  -0.733697438114660260f,
    0.679476319899365080f,  -0.734738878095963500f, 0.678350043129861360f,
    -0.735778589165713370f, 0.677222170137180560f,  -0.736816568877369900f,
    0.676092703575315920f,  -0.737852814788465760f, 0.674961646102012260f,
    -0.738887324460615000f, 0.673829000378756150f,  -0.739920095459516200f,
    0.672694769070772860f,  -0.740951125354958880f, 0.671558954847018550f,
    -0.741980411720830960f, 0.670421560380173200f,  -0.743007952135121720f,
    0.669282588346636010f,  -0.744033744179929070f, 0.668142041426518670f,
    -0.745057785441465950f, 0.666999922303637580f,  -0.746080073510063890f,
    0.665856233665509610f,  -0.747100605980180020f, 0.664710978203345020f,
    -0.748119380450403600f, 0.663564158612039770f,  -0.749136394523459150f,
    0.662415777590172010f,  -0.750151645806214960f, 0.661265837839992380f,
    -0.751165131909686480f, 0.660114342067420480f,  -0.752176850449042480f,
    0.658961292982037540f,  -0.753186799043612410f, 0.657806693297078750f,
    -0.754194975316889170f, 0.656650545729429050f,  -0.755201376896536440f,
    0.655492852999615570f,  -0.756206001414394420f, 0.654333617831800550f,
    -0.757208846506484570f, 0.653172842953776640f,  -0.758209909813015170f,
    0.652010531096959720f,  -0.759209188978388070f, 0.650846684996380990f,
    -0.760206681651202200f, 0.649681307390683420f,  -0.761202385484261670f,
    0.648514401022112550f,  -0.762196298134578900f, 0.647345968636512060f,
    -0.763188417263381050f, 0.646176012983316620f,  -0.764178740536116670f,
    0.645004536815544040f,  -0.765167265622458960f, 0.643831542889791390f,
    -0.766153990196312700f, 0.642657033966227090f,  -0.767138911935820290f,
    0.641481012808583160f,  -0.768122028523365420f, 0.640303482184151560f,
    -0.769103337645579480f, 0.639124444863775840f,  -0.770082836993347900f,
    0.637943903621844060f,  -0.771060524261813600f, 0.636761861236284420f,
    -0.772036397150384410f, 0.635578320488556230f,  -0.773010453362736990f,
    0.634393284163645490f,  -0.773982690606822680f, 0.633206755050057410f,
    -0.774953106594873820f, 0.632018735939809060f,  -0.775921699043407690f,
    0.630829229628424470f,  -0.776888465673232330f, 0.629638238914927210f,
    -0.777853404209453040f, 0.628445766601832710f,  -0.778816512381475980f,
    0.627251815495144080f,  -0.779777787923014330f, 0.626056388404343630f,
    -0.780737228572094490f, 0.624859488142386340f,  -0.781694832071059160f,
    0.623661117525694860f,  -0.782650596166575620f, 0.622461279374150080f,
    -0.783604518609638200f, 0.621259976511087550f,  -0.784556597155575020f,
    0.620057211763289430f,  -0.785506829564053930f, 0.618852987960976430f,
    -0.786455213599085770f, 0.617647307937803870f,  -0.787401747029031210f,
    0.616440174530853760f,  -0.788346427626606230f, 0.615231590580626930f,
    -0.789289253168885760f, 0.614021558931038380f,  -0.790230221437309920f,
    0.612810082429409820f,  -0.791169330217690200f, 0.611597163926461910f,
    -0.792106577300212170f, 0.610382806276309700f,  -0.793041960479443530f,
    0.609167012336453320f,  -0.793975477554337170f, 0.607949784967773630f,
    -0.794907126328236790f, 0.606731127034524700f,  -0.795836904608883460f,
    0.605511041404325660f,  -0.796764810208418830f, 0.604289530948155960f,
    -0.797690840943390930f, 0.603066598540348390f,  -0.798614994634760820f,
    0.601842247058580140f,  -0.799537269107905120f, 0.600616479383868860f,
    -0.800457662192622710f, 0.599389298400564650f,  -0.801376171723140240f,
    0.598160706996342380f,  -0.802292795538115500f, 0.596930708062196720f,
    -0.803207531480644830f, 0.595699304492433470f,  -0.804120377398265700f,
    0.594466499184664430f,  -0.805031331142963440f, 0.593232295039800020f,
    -0.805940390571176280f, 0.591996694962040990f,  -0.806847553543799330f,
    0.590759701858874160f,  -0.807752817926190250f, 0.589521318641064050f,
    -0.808656181588174980f, 0.588281548222645330f,  -0.809557642404051370f,
    0.587040393520917970f,  -0.810457198252594660f, 0.585797857456438980f,
    -0.811354847017063730f, 0.584553942953015330f,  -0.812250586585203770f,
    0.583308652937698510f,  -0.813144414849253480f, 0.582061990340775660f,
    -0.814036329705948410f, 0.580813958095764530f,  -0.814926329056526400f,
    0.579564559139405970f,  -0.815814410806733670f, 0.578313796411655700f,
    -0.816700572866827850f, 0.577061672855679440f,  -0.817584813151583600f,
    0.575808191417845450f,  -0.818467129580298660f, 0.574553355047715870f,
    -0.819347520076797010f, 0.573297166698042090f,  -0.820225982569434580f,
    0.572039629324757270f,  -0.821102514991104650f, 0.570780745886967260f,
    -0.821977115279241330f, 0.569520519346947470f,  -0.822849781375826210f,
    0.568258952670131710f,  -0.823720511227391430f, 0.566996048825108680f,
    -0.824589302785025070f, 0.565731810783613450f,  -0.825456154004377440f,
    0.564466241520519500f,  -0.826321062845663530f, 0.563199344013834090f,
    -0.827184027273669020f, 0.561931121244689580f,  -0.828045045257755690f,
    0.560661576197336140f,  -0.828904114771864990f, 0.559390711859136030f,
    -0.829761233794522930f, 0.558118531220556320f,  -0.830616400308846310f,
    0.556845037275160100f,  -0.831469612302545350f, 0.555570233019602180f,
    -0.832320867767929570f, 0.554294121453620230f,  -0.833170164701913190f,
    0.553016705580027580f,  -0.834017501106018020f, 0.551737988404707670f,
    -0.834862874986380010f, 0.550457972936604920f,  -0.835706284353752600f,
    0.549176662187719660f,  -0.836547727223511890f, 0.547894059173100410f,
    -0.837387201615661820f, 0.546610166910834970f,  -0.838224705554838080f,
    0.545324988422046350f,  -0.839060237070312630f, 0.544038526730884040f,
    -0.839893794195999520f, 0.542750784864515890f,  -0.840725374970458070f,
    0.541461765853123330f,  -0.841554977436898330f, 0.540171472729892970f,
    -0.842382599643185850f, 0.538879908531008420f,  -0.843208239641845330f,
    0.537587076295645730f,  -0.844031895490066410f, 0.536292979065963290f,
    -0.844853565249707120f, 0.534997619887097150f,  -0.845673246987298950f,
    0.533701001807153190f,  -0.846490938774052020f, 0.532403127877198010f,
    -0.847306638685858430f, 0.531104001151254890f,  -0.848120344803297120f,
    0.529803624686294830f,  -0.848932055211639610f, 0.528502001542228480f,
    -0.849741768000852550f, 0.527199134781901280f,  -0.850549481265603370f,
    0.525895027471084850f,  -0.851355193105265200f, 0.524589682678468950f,
    -0.852158901623919610f, 0.523283103475656650f,  -0.852960604930363630f,
    0.521975292937154500f,  -0.853760301138111410f, 0.520666254140367160f,
    -0.854557988365400420f, 0.519355990165589750f,  -0.855353664735195920f,
    0.518044504095999450f,  -0.856147328375194470f, 0.516731799017649760f,
    -0.856938977417828650f, 0.515417878019463150f,  -0.857728610000272010f,
    0.514102744193221770f,  -0.858516224264442850f, 0.512786400633562960f,
    -0.859301818357008360f, 0.511468850437970520f,  -0.860085390429390140f,
    0.510150096706766810f,  -0.860866938637767090f, 0.508830142543107320f,
    -0.861646461143081300f, 0.507508991052970980f,  -0.862423956111040500f,
    0.506186645345155230f,  -0.863199421712124050f, 0.504863108531267700f,
    -0.863972856121586700f, 0.503538383725717690f,  -0.864744257519462380f,
    0.502212474045710680f,  -0.865513624090568980f, 0.500885382611240940f,
    -0.866280954024512990f, 0.499557112545081950f,  -0.867046245515692760f,
    0.498227666972781760f,  -0.867809496763303210f, 0.496897049022654690f,
    -0.868570705971340900f, 0.495565261825772540f,  -0.869329871348606620f,
    0.494232308515960010f,  -0.870086991108711350f, 0.492898192229784150f,
    -0.870842063470078980f, 0.491562916106549900f,  -0.871595086655950870f,
    0.490226483288291380f,  -0.872346058894391430f, 0.488888896919763280f,
    -0.873094978418290090f, 0.487550160148435880f,  -0.873841843465366750f,
    0.486210276124486580f,  -0.874586652278176110f, 0.484869248000791120f,
    -0.875329403104110890f, 0.483527078932918630f,  -0.876070094195406490f,
    0.482183772079122890f,  -0.876808723809145650f, 0.480839330600333960f,
    -0.877545290207261130f, 0.479493757660153290f,  -0.878279791656541460f,
    0.478147056424843180f,  -0.879012226428633530f, 0.476799230063322090f,
    -0.879742592800047300f, 0.475450281747156090f,  -0.880470889052160750f,
    0.474100214650550080f,  -0.881197113471222090f, 0.472749031950342740f,
    -0.881921264348354940f, 0.471396736825997860f,  -0.882643339979562790f,
    0.470043332459595680f,  -0.883363338665731690f, 0.468688822035827850f,
    -0.884081258712634880f, 0.467333208741988580f,  -0.884797098430937790f,
    0.465976495767966180f,  -0.885510856136199840f, 0.464618686306238090f,
    -0.886222530148880530f, 0.463259783551860320f,  -0.886932118794342190f,
    0.461899790702462730f,  -0.887639620402853820f, 0.460538710958240230f,
    -0.888345033309596240f, 0.459176547521944200f,  -0.889048355854664570f,
    0.457813303598877170f,  -0.889749586383072670f, 0.456448982396884140f,
    -0.890448723244757880f, 0.455083587126343890f,  -0.891145764794583290f,
    0.453717121000163760f,  -0.891840709392342610f, 0.452349587233771060f,
    -0.892533555402764580f, 0.450980989045103860f,  -0.893224301195515210f,
    0.449611329654606870f,  -0.893912945145203140f, 0.448240612285220050f,
    -0.894599485631382700f, 0.446868840162374160f,  -0.895283921038557360f,
    0.445496016513981960f,  -0.895966249756185110f, 0.444122144570429310f,
    -0.896646470178680270f, 0.442747227564569970f,  -0.897324580705418210f,
    0.441371268731716890f,  -0.898000579740739770f, 0.439994271309633310f,
    -0.898674465693953930f, 0.438616238538527550f,  -0.899346236979341460f,
    0.437237173661044250f,  -0.900015892016160170f, 0.435857079922255530f,
    -0.900683429228646750f, 0.434475960569655980f,  -0.901348847046021920f,
    0.433093818853152070f,  -0.902012143902493180f, 0.431710658025057260f,
    -0.902673318237258710f, 0.430326481340082890f,  -0.903332368494511820f,
    0.428941292055329600f,  -0.903989293123443340f, 0.427555093430282030f,
    -0.904644090578246130f, 0.426167888726799840f,  -0.905296759318118700f,
    0.424779681209108860f,  -0.905947297807268460f, 0.423390474143795940f,
    -0.906595704514915330f, 0.422000270799799850f,  -0.907241977915295820f,
    0.420609074448402560f,  -0.907886116487666040f, 0.419216888363224240f,
    -0.908528118716306120f, 0.417823715820212440f,  -0.909167983090522380f,
    0.416429560097637150f,  -0.909805708104652110f, 0.415034424476081850f,
    -0.910441292258067140f, 0.413638312238434610f,  -0.911074734055176360f,
    0.412241226669882830f,  -0.911706032005429770f, 0.410843171057904130f,
    -0.912335184623322750f, 0.409444148692257650f,  -0.912962190428398210f,
    0.408044162864978580f,  -0.913587047945250700f, 0.406643216870369200f,
    -0.914209755703530690f, 0.405241314004989920f,  -0.914830312237945980f,
    0.403838457567654410f,  -0.915448716088267720f, 0.402434650859418600f,
    -0.916064965799331720f, 0.401029897183575620f,  -0.916679059921042590f,
    0.399624199845647070f,  -0.917290997008377910f, 0.398217562153373670f,
    -0.917900775621390500f, 0.396809987416710250f,  -0.918508394325212140f,
    0.395401478947816520f,  -0.919113851690057770f, 0.393992040061048150f,
    -0.919717146291227360f, 0.392581674072951410f,  -0.920318276709110480f,
    0.391170384302254040f,  -0.920917241529189410f, 0.389758174069856470f,
    -0.921514039342041790f, 0.388345046698826580f,  -0.922108668743345070f,
    0.386931005514388750f,  -0.922701128333878630f, 0.385516053843918850f,
    -0.923291416719527520f, 0.384100195016935320f,  -0.923879532511286740f,
    0.382683432365089890f,  -0.924465474325262600f, 0.381265769222162320f,
    -0.925049240782677470f, 0.379847208924051380f,  -0.925630830509872720f,
    0.378427754808765670f,  -0.926210242138311380f, 0.377007410216418150f,
    -0.926787474304581750f, 0.375586178489217380f,  -0.927362525650401110f,
    0.374164062971458040f,  -0.927935394822617780f, 0.372741067009516090f,
    -0.928506080473215480f, 0.371317193951837710f,  -0.929074581259315750f,
    0.369892447148934100f,  -0.929640895843181210f, 0.368466829953372600f,
    -0.930205022892219070f, 0.367040345719767290f,  -0.930766961078983710f,
    0.365612997804773800f,  -0.931326709081180320f, 0.364184789567080110f,
    -0.931884265581668040f, 0.362755724367397280f,  -0.932439629268462470f,
    0.361325805568454170f,  -0.932992798834738850f, 0.359895036534988330f,
    -0.933543772978836170f, 0.358463420633736600f,  -0.934092550404258760f,
    0.357030961233430310f,  -0.934639129819680670f, 0.355597661704784020f,
    -0.935183509938947610f, 0.354163525420490400f,  -0.935725689481080260f,
    0.352728555755210950f,  -0.936265667170278260f, 0.351292756085567200f,
    -0.936803441735921670f, 0.349856129790134860f,  -0.937339011912574850f,
    0.348418680249434790f,  -0.937872376439989770f, 0.346980410845923740f,
    -0.938403534063108170f, 0.345541324963988980f,  -0.938932483532064490f,
    0.344101425989939040f,  -0.939459223602189920f, 0.342660717311994430f,
    -0.939983753034013820f, 0.341219202320282690f,  -0.940506070593268300f,
    0.339776884406827020f,  -0.941026175050889260f, 0.338333766965541180f,
    -0.941544065183020700f, 0.336889853392220330f,  -0.942059739771017310f,
    0.335445147084531710f,  -0.942573197601446870f, 0.333999651442009380f,
    -0.943084437466093380f, 0.332553369866044450f,  -0.943593458161960390f,
    0.331106305759876480f,  -0.944100258491272660f, 0.329658462528587440f,
    -0.944604837261480150f, 0.328209843579092720f,  -0.945107193285260610f,
    0.326760452320131840f,  -0.945607325380521170f, 0.325310292162263260f,
    -0.946105232370403340f, 0.323859366517853020f,  -0.946600913083283530f,
    0.322407678801069850f,  -0.947094366352777110f, 0.320955232427875490f,
    -0.947585591017741090f, 0.319502030816015800f,  -0.948074585922276230f,
    0.318048077385014890f,  -0.948561349915730270f, 0.316593375556166070f,
    -0.949045881852700560f, 0.315137928752522500f,  -0.949528180593036670f,
    0.313681740398891410f,  -0.950008245001843000f, 0.312224813921825110f,
    -0.950486073949481700f, 0.310767152749611530f,  -0.950961666311574970f,
    0.309308760312269000f,  -0.951435020969008340f, 0.307849640041535030f,
    -0.951906136807932350f, 0.306389795370860920f,  -0.952375012719765770f,
    0.304929229735402650f,  -0.952841647601198600f, 0.303467946572011430f,
    -0.953306040354193860f, 0.302005949319228030f,  -0.953768189885990210f,
    0.300543241417273680f,  -0.954228095109105560f, 0.299079826308040530f,
    -0.954685754941338340f, 0.297615707435086140f,  -0.955141168305770670f,
    0.296150888243624010f,  -0.955594334130771110f, 0.294685372180514380f,
    -0.956045251349996290f, 0.293219162694258960f,  -0.956493918902394990f,
    0.291752263234989430f,  -0.956940335732208820f, 0.290284677254462390f,
    -0.957384500788975860f, 0.288816408206049760f,  -0.957826413027532910f,
    0.287347459544729620f,  -0.958266071408017670f, 0.285877834727080560f,
    -0.958703474895871490f, 0.284407537211272100f,  -0.959138622461841890f,
    0.282936570457055450f,  -0.959571513081984520f, 0.281464937925757890f,
    -0.960002145737665850f, 0.279992643080273440f,  -0.960430519415565790f,
    0.278519689385053170f,  -0.960856633107679550f, 0.277046080306100230f,
    -0.961280485811320640f, 0.275571819310958310f,  -0.961702076529122540f,
    0.274096909868706380f,  -0.962121404269041470f, 0.272621355449949250f,
    -0.962538468044359160f, 0.271145159526808120f,  -0.962953266873683880f,
    0.269668325572915090f,  -0.963365799780953940f, 0.268190857063403400f,
    -0.963776065795439840f, 0.266712757474898480f,  -0.964184063951745830f,
    0.265234030285511730f,  -0.964589793289812650f, 0.263754678974831570f,
    -0.964993252854920320f, 0.262274707023913700f,  -0.965394441697689290f,
    0.260794117915275850f,  -0.965793358874083570f, 0.259312915132886400f,
    -0.966190003445412500f, 0.257831102162158990f,  -0.966584374478333010f,
    0.256348682489943190f,  -0.966976471044852070f, 0.254865659604514680f,
    -0.967366292222328510f, 0.253382036995570100f,  -0.967753837093475400f,
    0.251897818154217190f,  -0.968139104746362330f, 0.250413006572965340f,
    -0.968522094274417380f, 0.248927605745720090f,  -0.968902804776428870f,
    0.247441619167773490f,  -0.969281235356548420f, 0.245955050335794650f,
    -0.969657385124292340f, 0.244467902747824480f,  -0.970031253194543970f,
    0.242980179903264070f,  -0.970402838687555500f, 0.241491885302869360f,
    -0.970772140728950240f, 0.240003022448741780f,  -0.971139158449725090f,
    0.238513594844318550f,  -0.971503890986251780f, 0.237023605994367170f,
    -0.971866337480279290f, 0.235533059404975740f,  -0.972226497078936270f,
    0.234041958583543510f,  -0.972584368934732210f, 0.232550307038775160f,
    -0.972939952205560070f, 0.231058108280671330f,  -0.973293246054698250f,
    0.229565365820518920f,  -0.973644249650811870f, 0.228072083170886060f,
    -0.973992962167955830f, 0.226578263845610170f,  -0.974339382785575860f,
    0.225083911359792830f,  -0.974683510688510670f, 0.223589029229790290f,
    -0.975025345066994120f, 0.222093620973203650f,  -0.975364885116656980f,
    0.220597690108873510f,  -0.975702130038528460f, 0.219101240156870050f,
    -0.976037079039039020f, 0.217604274638483720f,  -0.976369731330021140f,
    0.216106797076219440f,  -0.976700086128711730f, 0.214608810993786980f,
    -0.977028142657754390f, 0.213110319916091420f,  -0.977353900145199960f,
    0.211611327369227890f,  -0.977677357824509930f, 0.210111836880469800f,
    -0.977998514934557140f, 0.208611851978263510f,  -0.978317370719627540f,
    0.207111376192218840f,  -0.978633924429423100f, 0.205610413053099380f,
    -0.978948175319062200f, 0.204108966092816840f,  -0.979260122649082020f,
    0.202607038844421380f,  -0.979569765685440520f, 0.201104634842092010f,
    -0.979877103699517640f, 0.199601757621130920f,  -0.980182135968117320f,
    0.198098410717953810f,  -0.980484861773469380f, 0.196594597670080280f,
    -0.980785280403230430f, 0.195090322016128610f,  -0.981083391150486590f,
    0.193585587295803800f,  -0.981379193313754560f, 0.192080397049892470f,
    -0.981672686196983110f, 0.190574754820253070f,  -0.981963869109555240f,
    0.189068664149806360f,  -0.982252741366289370f, 0.187562128582529570f,
    -0.982539302287441240f, 0.186055151663446910f,  -0.982823551198705240f,
    0.184547736938619700f,  -0.983105487431216290f, 0.183039887955140900f,
    -0.983385110321551180f, 0.181531608261125220f,  -0.983662419211730250f,
    0.180022901405699570f,  -0.983937413449218920f, 0.178513770938997420f,
    -0.984210092386929030f, 0.177004220412148940f,  -0.984480455383220930f,
    0.175494253377271450f,  -0.984748501801904210f, 0.173983873387464130f,
    -0.985014231012239840f, 0.172473083996796120f,  -0.985277642388941220f,
    0.170961888760301220f,  -0.985538735312176060f, 0.169450291233968210f,
    -0.985797509167567370f, 0.167938294974731280f,  -0.986053963346195440f,
    0.166425903540464050f,  -0.986308097244598560f, 0.164913120489970140f,
    -0.986559910264775410f, 0.163399949382973280f,  -0.986809401814185530f,
    0.161886393780111740f,  -0.987056571305750970f, 0.160372457242928450f,
    -0.987301418157858430f, 0.158858143333861470f,  -0.987543941794359230f,
    0.157343455616238550f,  -0.987784141644572180f, 0.155828397654265370f,
    -0.988022017143283530f, 0.154312973013020080f,  -0.988257567730749460f,
    0.152797185258443690f,  -0.988490792852696590f, 0.151281037957330310f,
    -0.988721691960323780f, 0.149764534677321450f,  -0.988950264510302990f,
    0.148247678986896250f,  -0.989176509964781010f, 0.146730474455361800f,
    -0.989400427791380380f, 0.145212924652847350f,  -0.989622017463200780f,
    0.143695033150294640f,  -0.989841278458820530f, 0.142176803519448090f,
    -0.990058210262297010f, 0.140658239332849540f,  -0.990272812363169110f,
    0.139139344163826340f,  -0.990485084256457090f, 0.137620121586486040f,
    -0.990695025442664630f, 0.136100575175706480f,  -0.990902635427780010f,
    0.134580708507126280f,  -0.991107913723276890f, 0.133060525157139010f,
    -0.991310859846115440f, 0.131540028702883340f,  -0.991511473318743900f,
    0.130019222722233430f,  -0.991709753669099530f, 0.128498110793793090f,
    -0.991905700430609330f, 0.126976696496886060f,  -0.992099313142191800f,
    0.125454983411546260f,  -0.992290591348257260f, 0.123932975118512480f,
    -0.992479534598709970f, 0.122410675199216350f,  -0.992666142448948020f,
    0.120888087235777060f,  -0.992850414459865100f, 0.119365214810991630f,
    -0.993032350197851410f, 0.117842061508325090f,  -0.993211949234794500f,
    0.116318630911904710f,  -0.993389211148080650f, 0.114794926606510310f,
    -0.993564135520595300f, 0.113270952177564420f,  -0.993736721940724600f,
    0.111746711211126500f,  -0.993906970002356060f, 0.110222207293883240f,
    -0.994074879304879370f, 0.108697444013138740f,  -0.994240449453187900f,
    0.107172424956809160f,  -0.994403680057679100f, 0.105647153713410750f,
    -0.994564570734255420f, 0.104121633872054570f,  -0.994723121104325700f,
    0.102595869022436560f,  -0.994879330794805620f, 0.101069862754827930f,
    -0.995033199438118630f, 0.099543618660069277f,  -0.995184726672196820f,
    0.098017140329560826f,  -0.995333912140482280f, 0.096490431355252662f,
    -0.995480755491926940f, 0.094963495329638908f,  -0.995625256380994310f,
    0.093436335845747967f,  -0.995767414467659820f, 0.091908956497132752f,
    -0.995907229417411720f, 0.090381360877865288f,  -0.996044700901251970f,
    0.088853552582524753f,  -0.996179828595696980f, 0.087325535206192059f,
    -0.996312612182778000f, 0.085797312344440158f,  -0.996443051350042630f,
    0.084268887593324182f,  -0.996571145790554840f, 0.082740264549375636f,
    -0.996696895202896060f, 0.081211446809592663f,  -0.996820299291165670f,
    0.079682437971430195f,  -0.996941357764982160f, 0.078153241632794149f,
    -0.997060070339482960f, 0.076623861392031686f,  -0.997176436735326080f,
    0.075094300847921347f,  -0.997290456678690210f, 0.073564563599667732f,
    -0.997402129901275300f, 0.072034653246889471f,  -0.997511456140303450f,
    0.070504573389613856f,  -0.997618435138519550f, 0.068974327628267024f,
    -0.997723066644191640f, 0.067443919563664176f,  -0.997825350411111640f,
    0.065913352797003763f,  -0.997925286198596000f, 0.064382630929857701f,
    -0.998022873771486240f, 0.062851757564161490f,  -0.998118112900149180f,
    0.061320736302208488f,  -0.998211003360478190f, 0.059789570746640069f,
    -0.998301544933892890f, 0.058258264500435794f,  -0.998389737407340160f,
    0.056726821166908067f,  -0.998475580573294770f, 0.055195244349690094f,
    -0.998559074229759310f, 0.053663537652730520f,  -0.998640218180265160f,
    0.052131704680283594f,  -0.998719012233872940f, 0.050599749036899393f,
    -0.998795456205172410f, 0.049067674327417966f,  -0.998869549914283560f,
    0.047535484156959538f,  -0.998941293186856870f, 0.046003182130914706f,
    -0.999010685854073380f, 0.044470771854938584f,  -0.999077727752645360f,
    0.042938256934941021f,  -0.999142418724816910f, 0.041405640977076774f,
    -0.999204758618363890f, 0.039872927587740130f,  -0.999264747286594420f,
    0.038340120373552854f,  -0.999322384588349540f, 0.036807222941358832f,
    -0.999377670388002850f, 0.035274238898214232f,  -0.999430604555461730f,
    0.033741171851377705f,  -0.999481186966166950f, 0.032208025408304544f,
    -0.999529417501093140f, 0.030674803176636865f,  -0.999575296046749220f,
    0.029141508764193802f,  -0.999618822495178640f, 0.027608145778965660f,
    -0.999659996743959220f, 0.026074717829104099f,  -0.999698818696204250f,
    0.024541228522912326f,  -0.999735288260561680f, 0.023007681468839695f,
    -0.999769405351215280f, 0.021474080275469667f,  -0.999801169887884260f,
    0.019940428551514438f,  -0.999830581795823400f, 0.018406729905805101f,
    -0.999857641005823860f, 0.016872987947281835f,  -0.999882347454212560f,
    0.015339206284988060f,  -0.999904701082852790f, 0.013805388528060632f,
    -0.999924701839144500f, 0.012271538285720007f,  -0.999942349676023910f,
    0.010737659167264411f,  -0.999957644551963900f, 0.009203754782060021f,
    -0.999970586430974140f, 0.007669828739531138f,  -0.999981175282601110f,
    0.006135884649154799f,  -0.999989411081928400f, 0.004601926120448733f,
    -0.999995293809576190f, 0.003067956762965977f,  -0.999998823451701880f,
    0.001533980186285049f,  -1.000000000000000000f, 0.000000000000000122f,
    -0.999998823451701880f, -0.001533980186284804f, -0.999995293809576190f,
    -0.003067956762965732f, -0.999989411081928400f, -0.004601926120448488f,
    -0.999981175282601110f, -0.006135884649154554f, -0.999970586430974140f,
    -0.007669828739530893f, -0.999957644551963900f, -0.009203754782059776f,
    -0.999942349676023910f, -0.010737659167264166f, -0.999924701839144500f,
    -0.012271538285719762f, -0.999904701082852900f, -0.013805388528060387f,
    -0.999882347454212560f, -0.015339206284987816f, -0.999857641005823860f,
    -0.016872987947281589f, -0.999830581795823400f, -0.018406729905804858f,
    -0.999801169887884260f, -0.019940428551514192f, -0.999769405351215280f,
    -0.021474080275469421f, -0.999735288260561680f, -0.023007681468839448f,
    -0.999698818696204250f, -0.024541228522912080f, -0.999659996743959220f,
    -0.026074717829103856f, -0.999618822495178640f, -0.027608145778965414f,
    -0.999575296046749220f, -0.029141508764193556f, -0.999529417501093140f,
    -0.030674803176636619f, -0.999481186966166950f, -0.032208025408304294f,
    -0.999430604555461730f, -0.033741171851377455f, -0.999377670388002850f,
    -0.035274238898213982f, -0.999322384588349540f, -0.036807222941358582f,
    -0.999264747286594420f, -0.038340120373552611f, -0.999204758618363890f,
    -0.039872927587739887f, -0.999142418724816910f, -0.041405640977076531f,
    -0.999077727752645360f, -0.042938256934940779f, -0.999010685854073380f,
    -0.044470771854938335f, -0.998941293186856870f, -0.046003182130914456f,
    -0.998869549914283560f, -0.047535484156959296f, -0.998795456205172410f,
    -0.049067674327417724f, -0.998719012233872940f, -0.050599749036899150f,
    -0.998640218180265270f, -0.052131704680283351f, -0.998559074229759310f,
    -0.053663537652730277f, -0.998475580573294770f, -0.055195244349689851f,
    -0.998389737407340160f, -0.056726821166907818f, -0.998301544933892890f,
    -0.058258264500435551f, -0.998211003360478190f, -0.059789570746639827f,
    -0.998118112900149180f, -0.061320736302208245f, -0.998022873771486240f,
    -0.062851757564161240f, -0.997925286198596000f, -0.064382630929857451f,
    -0.997825350411111640f, -0.065913352797003527f, -0.997723066644191640f,
    -0.067443919563663926f, -0.997618435138519550f, -0.068974327628266774f,
    -0.997511456140303450f, -0.070504573389613606f, -0.997402129901275300f,
    -0.072034653246889235f, -0.997290456678690210f, -0.073564563599667496f,
    -0.997176436735326190f, -0.075094300847921097f, -0.997060070339482960f,
    -0.076623861392031437f, -0.996941357764982160f, -0.078153241632793899f,
    -0.996820299291165780f, -0.079682437971429945f, -0.996696895202896060f,
    -0.081211446809592427f, -0.996571145790554840f, -0.082740264549375400f,
    -0.996443051350042630f, -0.084268887593323932f, -0.996312612182778000f,
    -0.085797312344439922f, -0.996179828595696980f, -0.087325535206191809f,
    -0.996044700901251970f, -0.088853552582524503f, -0.995907229417411720f,
    -0.090381360877865052f, -0.995767414467659820f, -0.091908956497132516f,
    -0.995625256380994310f, -0.093436335845747731f, -0.995480755491926940f,
    -0.094963495329638659f, -0.995333912140482280f, -0.096490431355252412f,
    -0.995184726672196930f, -0.098017140329560590f, -0.995033199438118630f,
    -0.099543618660069041f, -0.994879330794805620f, -0.101069862754827680f,
    -0.994723121104325700f, -0.102595869022436310f, -0.994564570734255530f,
    -0.104121633872054320f, -0.994403680057679100f, -0.105647153713410520f,
    -0.994240449453187900f, -0.107172424956808910f, -0.994074879304879480f,
    -0.108697444013138490f, -0.993906970002356060f, -0.110222207293883000f,
    -0.993736721940724710f, -0.111746711211126250f, -0.993564135520595300f,
    -0.113270952177564170f, -0.993389211148080650f, -0.114794926606510070f,
    -0.993211949234794610f, -0.116318630911904470f, -0.993032350197851410f,
    -0.117842061508324840f, -0.992850414459865100f, -0.119365214810991380f,
    -0.992666142448948020f, -0.120888087235776820f, -0.992479534598709970f,
    -0.122410675199216100f, -0.992290591348257370f, -0.123932975118512230f,
    -0.992099313142191800f, -0.125454983411546010f, -0.991905700430609330f,
    -0.126976696496885810f, -0.991709753669099530f, -0.128498110793792840f,
    -0.991511473318744010f, -0.130019222722233180f, -0.991310859846115440f,
    -0.131540028702883090f, -0.991107913723276890f, -0.133060525157138760f,
    -0.990902635427780010f, -0.134580708507126060f, -0.990695025442664630f,
    -0.136100575175706230f, -0.990485084256457090f, -0.137620121586485790f,
    -0.990272812363169110f, -0.139139344163826120f, -0.990058210262297120f,
    -0.140658239332849290f, -0.989841278458820530f, -0.142176803519447840f,
    -0.989622017463200890f, -0.143695033150294390f, -0.989400427791380380f,
    -0.145212924652847130f, -0.989176509964781010f, -0.146730474455361580f,
    -0.988950264510302990f, -0.148247678986896030f, -0.988721691960323780f,
    -0.149764534677321200f, -0.988490792852696700f, -0.151281037957330080f,
    -0.988257567730749460f, -0.152797185258443440f, -0.988022017143283640f,
    -0.154312973013019830f, -0.987784141644572180f, -0.155828397654265120f,
    -0.987543941794359230f, -0.157343455616238300f, -0.987301418157858430f,
    -0.158858143333861220f, -0.987056571305750970f, -0.160372457242928200f,
    -0.986809401814185530f, -0.161886393780111490f, -0.986559910264775520f,
    -0.163399949382973060f, -0.986308097244598670f, -0.164913120489969890f,
    -0.986053963346195440f, -0.166425903540463830f, -0.985797509167567480f,
    -0.167938294974731030f, -0.985538735312176060f, -0.169450291233967990f,
    -0.985277642388941330f, -0.170961888760300970f, -0.985014231012239840f,
    -0.172473083996795870f, -0.984748501801904210f, -0.173983873387463880f,
    -0.984480455383220930f, -0.175494253377271200f, -0.984210092386929140f,
    -0.177004220412148690f, -0.983937413449218920f, -0.178513770938997170f,
    -0.983662419211730250f, -0.180022901405699350f, -0.983385110321551180f,
    -0.181531608261124970f, -0.983105487431216400f, -0.183039887955140650f,
    -0.982823551198705350f, -0.184547736938619480f, -0.982539302287441240f,
    -0.186055151663446660f, -0.982252741366289480f, -0.187562128582529320f,
    -0.981963869109555240f, -0.189068664149806110f, -0.981672686196983110f,
    -0.190574754820252820f, -0.981379193313754670f, -0.192080397049892220f,
    -0.981083391150486710f, -0.193585587295803550f, -0.980785280403230430f,
    -0.195090322016128360f, -0.980484861773469380f, -0.196594597670080030f,
    -0.980182135968117430f, -0.198098410717953560f, -0.979877103699517750f,
    -0.199601757621130670f, -0.979569765685440520f, -0.201104634842091760f,
    -0.979260122649082020f, -0.202607038844421130f, -0.978948175319062200f,
    -0.204108966092816620f, -0.978633924429423210f, -0.205610413053099160f,
    -0.978317370719627650f, -0.207111376192218590f, -0.977998514934557140f,
    -0.208611851978263260f, -0.977677357824510040f, -0.210111836880469550f,
    -0.977353900145199960f, -0.211611327369227660f, -0.977028142657754390f,
    -0.213110319916091200f, -0.976700086128711840f, -0.214608810993786730f,
    -0.976369731330021250f, -0.216106797076219210f, -0.976037079039039130f,
    -0.217604274638483470f, -0.975702130038528570f, -0.219101240156869800f,
    -0.975364885116656980f, -0.220597690108873260f, -0.975025345066994120f,
    -0.222093620973203430f, -0.974683510688510670f, -0.223589029229790040f,
    -0.974339382785575860f, -0.225083911359792610f, -0.973992962167955940f,
    -0.226578263845609920f, -0.973644249650811870f, -0.228072083170885810f,
    -0.973293246054698250f, -0.229565365820518700f, -0.972939952205560180f,
    -0.231058108280671080f, -0.972584368934732320f, -0.232550307038774940f,
    -0.972226497078936380f, -0.234041958583543260f, -0.971866337480279400f,
    -0.235533059404975510f, -0.971503890986251890f, -0.237023605994366950f,
    -0.971139158449725200f, -0.238513594844318330f, -0.970772140728950240f,
    -0.240003022448741530f, -0.970402838687555500f, -0.241491885302869110f,
    -0.970031253194543970f, -0.242980179903263820f, -0.969657385124292450f,
    -0.244467902747824260f, -0.969281235356548530f, -0.245955050335794430f,
    -0.968902804776428870f, -0.247441619167773270f, -0.968522094274417380f,
    -0.248927605745719870f, -0.968139104746362440f, -0.250413006572965110f,
    -0.967753837093475510f, -0.251897818154216970f, -0.967366292222328620f,
    -0.253382036995569880f, -0.966976471044852180f, -0.254865659604514460f,
    -0.966584374478333120f, -0.256348682489942910f, -0.966190003445412620f,
    -0.257831102162158770f, -0.965793358874083680f, -0.259312915132886180f,
    -0.965394441697689400f, -0.260794117915275630f, -0.964993252854920440f,
    -0.262274707023913420f, -0.964589793289812760f, -0.263754678974831350f,
    -0.964184063951745830f, -0.265234030285511510f, -0.963776065795439950f,
    -0.266712757474898250f, -0.963365799780954050f, -0.268190857063403180f,
    -0.962953266873683990f, -0.269668325572914810f, -0.962538468044359160f,
    -0.271145159526807900f, -0.962121404269041580f, -0.272621355449949030f,
    -0.961702076529122540f, -0.274096909868706160f, -0.961280485811320640f,
    -0.275571819310958090f, -0.960856633107679550f, -0.277046080306100010f,
    -0.960430519415565900f, -0.278519689385052890f, -0.960002145737665960f,
    -0.279992643080273220f, -0.959571513081984630f, -0.281464937925757660f,
    -0.959138622461842010f, -0.282936570457055170f, -0.958703474895871600f,
    -0.284407537211271820f, -0.958266071408017780f, -0.285877834727080340f,
    -0.957826413027532910f, -0.287347459544729400f, -0.957384500788975860f,
    -0.288816408206049540f, -0.956940335732208940f, -0.290284677254462110f,
    -0.956493918902395100f, -0.291752263234989210f, -0.956045251349996410f,
    -0.293219162694258740f, -0.955594334130771110f, -0.294685372180514160f,
    -0.955141168305770780f, -0.296150888243623790f, -0.954685754941338450f,
    -0.297615707435085920f, -0.954228095109105670f, -0.299079826308040310f,
    -0.953768189885990330f, -0.300543241417273450f, -0.953306040354193970f,
    -0.302005949319227810f, -0.952841647601198720f, -0.303467946572011200f,
    -0.952375012719765880f, -0.304929229735402430f, -0.951906136807932350f,
    -0.306389795370860700f, -0.951435020969008450f, -0.307849640041534810f,
    -0.950961666311575080f, -0.309308760312268780f, -0.950486073949481810f,
    -0.310767152749611310f, -0.950008245001843000f, -0.312224813921824880f,
    -0.949528180593036790f, -0.313681740398891180f, -0.949045881852700670f,
    -0.315137928752522220f, -0.948561349915730270f, -0.316593375556165850f,
    -0.948074585922276340f, -0.318048077385014670f, -0.947585591017741200f,
    -0.319502030816015580f, -0.947094366352777220f, -0.320955232427875270f,
    -0.946600913083283650f, -0.322407678801069630f, -0.946105232370403450f,
    -0.323859366517852800f, -0.945607325380521280f, -0.325310292162262980f,
    -0.945107193285260610f, -0.326760452320131570f, -0.944604837261480260f,
    -0.328209843579092500f, -0.944100258491272770f, -0.329658462528587210f,
    -0.943593458161960390f, -0.331106305759876260f, -0.943084437466093490f,
    -0.332553369866044220f, -0.942573197601446980f, -0.333999651442009100f,
    -0.942059739771017420f, -0.335445147084531490f, -0.941544065183020810f,
    -0.336889853392220110f, -0.941026175050889370f, -0.338333766965540910f,
    -0.940506070593268410f, -0.339776884406826800f, -0.939983753034013940f,
    -0.341219202320282470f, -0.939459223602190030f, -0.342660717311994210f,
    -0.938932483532064600f, -0.344101425989938810f, -0.938403534063108280f,
    -0.345541324963988760f, -0.937872376439989890f, -0.346980410845923510f,
    -0.937339011912574960f, -0.348418680249434560f, -0.936803441735921670f,
    -0.349856129790134640f, -0.936265667170278260f, -0.351292756085566980f,
    -0.935725689481080370f, -0.352728555755210730f, -0.935183509938947720f,
    -0.354163525420490120f, -0.934639129819680780f, -0.355597661704783800f,
    -0.934092550404258870f, -0.357030961233430090f, -0.933543772978836280f,
    -0.358463420633736370f, -0.932992798834738960f, -0.359895036534988110f,
    -0.932439629268462470f, -0.361325805568453950f, -0.931884265581668150f,
    -0.362755724367397060f, -0.931326709081180430f, -0.364184789567079890f,
    -0.930766961078983820f, -0.365612997804773580f, -0.930205022892219070f,
    -0.367040345719767070f, -0.929640895843181210f, -0.368466829953372380f,
    -0.929074581259315860f, -0.369892447148933880f, -0.928506080473215590f,
    -0.371317193951837430f, -0.927935394822617780f, -0.372741067009515870f,
    -0.927362525650401110f, -0.374164062971457820f, -0.926787474304581860f,
    -0.375586178489217160f, -0.926210242138311490f, -0.377007410216417930f,
    -0.925630830509872830f, -0.378427754808765390f, -0.925049240782677580f,
    -0.379847208924051160f, -0.924465474325262710f, -0.381265769222162100f,
    -0.923879532511286850f, -0.382683432365089670f, -0.923291416719527640f,
    -0.384100195016935100f, -0.922701128333878630f, -0.385516053843918630f,
    -0.922108668743345180f, -0.386931005514388530f, -0.921514039342041900f,
    -0.388345046698826360f, -0.920917241529189520f, -0.389758174069856240f,
    -0.920318276709110590f, -0.391170384302253820f, -0.919717146291227470f,
    -0.392581674072951190f, -0.919113851690057770f, -0.393992040061047930f,
    -0.918508394325212250f, -0.395401478947816300f, -0.917900775621390610f,
    -0.396809987416710030f, -0.917290997008378020f, -0.398217562153373450f,
    -0.916679059921042700f, -0.399624199845646840f, -0.916064965799331830f,
    -0.401029897183575400f, -0.915448716088267830f, -0.402434650859418370f,
    -0.914830312237946090f, -0.403838457567654190f, -0.914209755703530690f,
    -0.405241314004989690f, -0.913587047945250810f, -0.406643216870368970f,
    -0.912962190428398320f, -0.408044162864978350f, -0.912335184623322860f,
    -0.409444148692257430f, -0.911706032005429880f, -0.410843171057903910f,
    -0.911074734055176470f, -0.412241226669882610f, -0.910441292258067250f,
    -0.413638312238434390f, -0.909805708104652220f, -0.415034424476081630f,
    -0.909167983090522490f, -0.416429560097636930f, -0.908528118716306230f,
    -0.417823715820212220f, -0.907886116487666150f, -0.419216888363224020f,
    -0.907241977915295930f, -0.420609074448402340f, -0.906595704514915450f,
    -0.422000270799799630f, -0.905947297807268570f, -0.423390474143795710f,
    -0.905296759318118820f, -0.424779681209108640f, -0.904644090578246240f,
    -0.426167888726799620f, -0.903989293123443450f, -0.427555093430281810f,
    -0.903332368494511930f, -0.428941292055329380f, -0.902673318237258830f,
    -0.430326481340082670f, -0.902012143902493290f, -0.431710658025057040f,
    -0.901348847046022030f, -0.433093818853151850f, -0.900683429228646860f,
    -0.434475960569655760f, -0.900015892016160280f, -0.435857079922255310f,
    -0.899346236979341570f, -0.437237173661044030f, -0.898674465693954040f,
    -0.438616238538527330f, -0.898000579740739880f, -0.439994271309633090f,
    -0.897324580705418320f, -0.441371268731716670f, -0.896646470178680380f,
    -0.442747227564569750f, -0.895966249756185220f, -0.444122144570429090f,
    -0.895283921038557470f, -0.445496016513981740f, -0.894599485631382810f,
    -0.446868840162373940f, -0.893912945145203250f, -0.448240612285219830f,
    -0.893224301195515320f, -0.449611329654606650f, -0.892533555402764690f,
    -0.450980989045103640f, -0.891840709392342720f, -0.452349587233770830f,
    -0.891145764794583410f, -0.453717121000163540f, -0.890448723244757990f,
    -0.455083587126343670f, -0.889749586383072780f, -0.456448982396883920f,
    -0.889048355854664680f, -0.457813303598876950f, -0.888345033309596350f,
    -0.459176547521943980f, -0.887639620402853930f, -0.460538710958240060f,
    -0.886932118794342310f, -0.461899790702462510f, -0.886222530148880640f,
    -0.463259783551860090f, -0.885510856136199950f, -0.464618686306237870f,
    -0.884797098430937900f, -0.465976495767965960f, -0.884081258712634990f,
    -0.467333208741988360f, -0.883363338665731800f, -0.468688822035827620f,
    -0.882643339979562900f, -0.470043332459595450f, -0.881921264348355050f,
    -0.471396736825997640f, -0.881197113471222200f, -0.472749031950342510f,
    -0.880470889052160870f, -0.474100214650549860f, -0.879742592800047410f,
    -0.475450281747155920f, -0.879012226428633640f, -0.476799230063321870f,
    -0.878279791656541580f, -0.478147056424842950f, -0.877545290207261240f,
    -0.479493757660153060f, -0.876808723809145760f, -0.480839330600333740f,
    -0.876070094195406600f, -0.482183772079122660f, -0.875329403104111000f,
    -0.483527078932918410f, -0.874586652278176220f, -0.484869248000790950f,
    -0.873841843465366860f, -0.486210276124486360f, -0.873094978418290200f,
    -0.487550160148435660f, -0.872346058894391540f, -0.488888896919763060f,
    -0.871595086655950980f, -0.490226483288291160f, -0.870842063470079090f,
    -0.491562916106549730f, -0.870086991108711460f, -0.492898192229783930f,
    -0.869329871348606730f, -0.494232308515959780f, -0.868570705971341010f,
    -0.495565261825772320f, -0.867809496763303320f, -0.496897049022654470f,
    -0.867046245515692870f, -0.498227666972781540f, -0.866280954024513110f,
    -0.499557112545081730f, -0.865513624090569090f, -0.500885382611240710f,
    -0.864744257519462490f, -0.502212474045710570f, -0.863972856121586810f,
    -0.503538383725717460f, -0.863199421712124160f, -0.504863108531267590f,
    -0.862423956111040720f, -0.506186645345155010f, -0.861646461143081410f,
    -0.507508991052970760f, -0.860866938637767310f, -0.508830142543107100f,
    -0.860085390429390250f, -0.510150096706766590f, -0.859301818357008470f,
    -0.511468850437970300f, -0.858516224264442960f, -0.512786400633562730f,
    -0.857728610000272120f, -0.514102744193221550f, -0.856938977417828760f,
    -0.515417878019462930f, -0.856147328375194690f, -0.516731799017649650f,
    -0.855353664735196140f, -0.518044504095999230f, -0.854557988365400530f,
    -0.519355990165589640f, -0.853760301138111520f, -0.520666254140366940f,
    -0.852960604930363740f, -0.521975292937154280f, -0.852158901623919830f,
    -0.523283103475656430f, -0.851355193105265310f, -0.524589682678468730f,
    -0.850549481265603480f, -0.525895027471084630f, -0.849741768000852660f,
    -0.527199134781901060f, -0.848932055211639720f, -0.528502001542228260f,
    -0.848120344803297230f, -0.529803624686294610f, -0.847306638685858540f,
    -0.531104001151254670f, -0.846490938774052130f, -0.532403127877197790f,
    -0.845673246987299070f, -0.533701001807152960f, -0.844853565249707230f,
    -0.534997619887096930f, -0.844031895490066520f, -0.536292979065963070f,
    -0.843208239641845440f, -0.537587076295645510f, -0.842382599643185960f,
    -0.538879908531008200f, -0.841554977436898440f, -0.540171472729892850f,
    -0.840725374970458180f, -0.541461765853123220f, -0.839893794195999630f,
    -0.542750784864515780f, -0.839060237070312740f, -0.544038526730883820f,
    -0.838224705554838190f, -0.545324988422046130f, -0.837387201615662050f,
    -0.546610166910834750f, -0.836547727223512010f, -0.547894059173100190f,
    -0.835706284353752720f, -0.549176662187719540f, -0.834862874986380120f,
    -0.550457972936604700f, -0.834017501106018130f, -0.551737988404707450f,
    -0.833170164701913300f, -0.553016705580027360f, -0.832320867767929680f,
    -0.554294121453620000f, -0.831469612302545460f, -0.555570233019601960f,
    -0.830616400308846430f, -0.556845037275159880f, -0.829761233794523050f,
    -0.558118531220556100f, -0.828904114771865100f, -0.559390711859135800f,
    -0.828045045257755800f, -0.560661576197335920f, -0.827184027273669130f,
    -0.561931121244689360f, -0.826321062845663650f, -0.563199344013833870f,
    -0.825456154004377550f, -0.564466241520519390f, -0.824589302785025290f,
    -0.565731810783613230f, -0.823720511227391540f, -0.566996048825108460f,
    -0.822849781375826430f, -0.568258952670131490f, -0.821977115279241550f,
    -0.569520519346947250f, -0.821102514991104760f, -0.570780745886967140f,
    -0.820225982569434690f, -0.572039629324757050f, -0.819347520076797120f,
    -0.573297166698041980f, -0.818467129580298770f, -0.574553355047715650f,
    -0.817584813151583710f, -0.575808191417845340f, -0.816700572866827960f,
    -0.577061672855679330f, -0.815814410806733890f, -0.578313796411655480f,
    -0.814926329056526620f, -0.579564559139405740f, -0.814036329705948520f,
    -0.580813958095764300f, -0.813144414849253590f, -0.582061990340775440f,
    -0.812250586585203880f, -0.583308652937698400f, -0.811354847017063840f,
    -0.584553942953015100f, -0.810457198252594770f, -0.585797857456438860f,
    -0.809557642404051480f, -0.587040393520917750f, -0.808656181588175090f,
    -0.588281548222645110f, -0.807752817926190360f, -0.589521318641063940f,
    -0.806847553543799450f, -0.590759701858873940f, -0.805940390571176390f,
    -0.591996694962040880f, -0.805031331142963550f, -0.593232295039799800f,
    -0.804120377398265920f, -0.594466499184664210f, -0.803207531480644940f,
    -0.595699304492433250f, -0.802292795538115720f, -0.596930708062196500f,
    -0.801376171723140350f, -0.598160706996342160f, -0.800457662192622820f,
    -0.599389298400564540f, -0.799537269107905240f, -0.600616479383868640f,
    -0.798614994634760930f, -0.601842247058579920f, -0.797690840943391160f,
    -0.603066598540348160f, -0.796764810208418940f, -0.604289530948155850f,
    -0.795836904608883570f, -0.605511041404325430f, -0.794907126328236900f,
    -0.606731127034524480f, -0.793975477554337280f, -0.607949784967773410f,
    -0.793041960479443750f, -0.609167012336453100f, -0.792106577300212280f,
    -0.610382806276309480f, -0.791169330217690310f, -0.611597163926461800f,
    -0.790230221437310140f, -0.612810082429409710f, -0.789289253168885870f,
    -0.614021558931038160f, -0.788346427626606340f, -0.615231590580626710f,
    -0.787401747029031430f, -0.616440174530853650f, -0.786455213599085990f,
    -0.617647307937803650f, -0.785506829564054040f, -0.618852987960976210f,
    -0.784556597155575240f, -0.620057211763289210f, -0.783604518609638420f,
    -0.621259976511087440f, -0.782650596166575840f, -0.622461279374149860f,
    -0.781694832071059390f, -0.623661117525694640f, -0.780737228572094600f,
    -0.624859488142386230f, -0.779777787923014550f, -0.626056388404343520f,
    -0.778816512381476200f, -0.627251815495143860f, -0.777853404209453150f,
    -0.628445766601832600f, -0.776888465673232440f, -0.629638238914926980f,
    -0.775921699043407800f, -0.630829229628424250f, -0.774953106594873930f,
    -0.632018735939808950f, -0.773982690606822790f, -0.633206755050057300f,
    -0.773010453362737100f, -0.634393284163645270f, -0.772036397150384520f,
    -0.635578320488556110f, -0.771060524261813710f, -0.636761861236284310f,
    -0.770082836993348120f, -0.637943903621843940f, -0.769103337645579700f,
    -0.639124444863775730f, -0.768122028523365640f, -0.640303482184151450f,
    -0.767138911935820510f, -0.641481012808583050f, -0.766153990196312920f,
    -0.642657033966226860f, -0.765167265622459070f, -0.643831542889791280f,
    -0.764178740536116790f, -0.645004536815543820f, -0.763188417263381270f,
    -0.646176012983316390f, -0.762196298134579120f, -0.647345968636511840f,
    -0.761202385484261890f, -0.648514401022112330f, -0.760206681651202420f,
    -0.649681307390683190f, -0.759209188978388180f, -0.650846684996380760f,
    -0.758209909813015280f, -0.652010531096959500f, -0.757208846506484790f,
    -0.653172842953776530f, -0.756206001414394650f, -0.654333617831800330f,
    -0.755201376896536550f, -0.655492852999615350f, -0.754194975316889390f,
    -0.656650545729428830f, -0.753186799043612630f, -0.657806693297078530f,
    -0.752176850449042700f, -0.658961292982037320f, -0.751165131909686590f,
    -0.660114342067420260f, -0.750151645806215070f, -0.661265837839992150f,
    -0.749136394523459260f, -0.662415777590171780f, -0.748119380450403710f,
    -0.663564158612039660f, -0.747100605980180130f, -0.664710978203344790f,
    -0.746080073510064000f, -0.665856233665509390f, -0.745057785441466060f,
    -0.666999922303637360f, -0.744033744179929290f, -0.668142041426518450f,
    -0.743007952135121940f, -0.669282588346635790f, -0.741980411720831070f,
    -0.670421560380172980f, -0.740951125354959110f, -0.671558954847018440f,
    -0.739920095459516310f, -0.672694769070772750f, -0.738887324460615220f,
    -0.673829000378756040f, -0.737852814788465980f, -0.674961646102012040f,
    -0.736816568877370020f, -0.676092703575315810f, -0.735778589165713590f,
    -0.677222170137180330f, -0.734738878095963720f, -0.678350043129861250f,
    -0.733697438114660370f, -0.679476319899364860f, -0.732654271672412820f,
    -0.680600997795453020f, -0.731609381223892740f, -0.681724074171649600f,
    -0.730562769227827700f, -0.682845546385247970f, -0.729514438146997010f,
    -0.683965411797315400f, -0.728464390448225420f, -0.685083667772700130f,
    -0.727412628602375880f, -0.686200311680038480f, -0.726359155084345900f,
    -0.687315340891759160f, -0.725303972373060880f, -0.688428752784090330f,
    -0.724247082951467000f, -0.689540544737066830f, -0.723188489306527680f,
    -0.690650714134534380f, -0.722128193929215460f, -0.691759258364157640f,
    -0.721066199314508110f, -0.692866174817424630f, -0.720002507961381880f,
    -0.693971460889653780f, -0.718937122372804490f, -0.695075113980000770f,
    -0.717870045055731710f, -0.696177131491462990f, -0.716801278521099650f,
    -0.697277510830886400f, -0.715730825283818710f, -0.698376249408972800f,
    -0.714658687862768980f, -0.699473344640283880f, -0.713584868780793750f,
    -0.700568793943248220f, -0.712509370564692320f, -0.701662594740168450f,
    -0.711432195745216660f, -0.702754744457225080f, -0.710353346857062420f,
    -0.703845240524484830f, -0.709272826438865690f, -0.704934080375904880f,
    -0.708190637033195510f, -0.706021261449339520f, -0.707106781186547680f,
    -0.707106781186547460f, -0.706021261449339740f, -0.708190637033195290f,
    -0.704934080375905100f, -0.709272826438865470f, -0.703845240524485050f,
    -0.710353346857062310f, -0.702754744457225300f, -0.711432195745216430f,
    -0.701662594740168680f, -0.712509370564692210f, -0.700568793943248450f,
    -0.713584868780793520f, -0.699473344640284100f, -0.714658687862768760f,
    -0.698376249408973030f, -0.715730825283818480f, -0.697277510830886630f,
    -0.716801278521099540f, -0.696177131491463210f, -0.717870045055731490f,
    -0.695075113980000990f, -0.718937122372804380f, -0.693971460889654000f,
    -0.720002507961381650f, -0.692866174817424850f, -0.721066199314507880f,
    -0.691759258364157860f, -0.722128193929215230f, -0.690650714134534600f,
    -0.723188489306527460f, -0.689540544737067050f, -0.724247082951466780f,
    -0.688428752784090550f, -0.725303972373060660f, -0.687315340891759390f,
    -0.726359155084345680f, -0.686200311680038700f, -0.727412628602375650f,
    -0.685083667772700360f, -0.728464390448225200f, -0.683965411797315630f,
    -0.729514438146996790f, -0.682845546385248190f, -0.730562769227827480f,
    -0.681724074171649820f, -0.731609381223892520f, -0.680600997795453240f,
    -0.732654271672412590f, -0.679476319899365080f, -0.733697438114660150f,
    -0.678350043129861470f, -0.734738878095963500f, -0.677222170137180560f,
    -0.735778589165713370f, -0.676092703575316030f, -0.736816568877369790f,
    -0.674961646102012260f, -0.737852814788465760f, -0.673829000378756260f,
    -0.738887324460615000f, -0.672694769070772970f, -0.739920095459516090f,
    -0.671558954847018660f, -0.740951125354958880f, -0.670421560380173200f,
    -0.741980411720830960f, -0.669282588346636120f, -0.743007952135121720f,
    -0.668142041426518670f, -0.744033744179929070f, -0.666999922303637580f,
    -0.745057785441465840f, -0.665856233665509610f, -0.746080073510063780f,
    -0.664710978203345020f, -0.747100605980180020f, -0.663564158612039880f,
    -0.748119380450403490f, -0.662415777590172010f, -0.749136394523459040f,
    -0.661265837839992380f, -0.750151645806214960f, -0.660114342067420480f,
    -0.751165131909686370f, -0.658961292982037540f, -0.752176850449042480f,
    -0.657806693297078750f, -0.753186799043612410f, -0.656650545729429050f,
    -0.754194975316889170f, -0.655492852999615570f, -0.755201376896536320f,
    -0.654333617831800660f, -0.756206001414394420f, -0.653172842953777090f,
    -0.757208846506484230f, -0.652010531096959720f, -0.758209909813015170f,
    -0.650846684996380990f, -0.759209188978387960f, -0.649681307390683080f,
    -0.760206681651202420f, -0.648514401022112220f, -0.761202385484262000f,
    -0.647345968636512500f, -0.762196298134578560f, -0.646176012983316620f,
    -0.763188417263381050f, -0.645004536815544040f, -0.764178740536116560f,
    -0.643831542889791500f, -0.765167265622458960f, -0.642657033966226750f,
    -0.766153990196313030f, -0.641481012808583610f, -0.767138911935820070f,
    -0.640303482184152010f, -0.768122028523365090f, -0.639124444863775950f,
    -0.769103337645579480f, -0.637943903621844170f, -0.770082836993347900f,
    -0.636761861236284200f, -0.771060524261813820f, -0.635578320488556000f,
    -0.772036397150384630f, -0.634393284163645930f, -0.773010453362736660f,
    -0.633206755050057520f, -0.773982690606822570f, -0.632018735939809170f,
    -0.774953106594873820f, -0.630829229628424580f, -0.775921699043407580f,
    -0.629638238914926870f, -0.776888465673232550f, -0.628445766601833160f,
    -0.777853404209452700f, -0.627251815495144420f, -0.778816512381475650f,
    -0.626056388404343740f, -0.779777787923014330f, -0.624859488142386450f,
    -0.780737228572094380f, -0.623661117525694530f, -0.781694832071059500f,
    -0.622461279374149750f, -0.782650596166575840f, -0.621259976511088000f,
    -0.783604518609637980f, -0.620057211763289430f, -0.784556597155575020f,
    -0.618852987960976430f, -0.785506829564053820f, -0.617647307937803980f,
    -0.786455213599085770f, -0.616440174530853540f, -0.787401747029031430f,
    -0.615231590580627260f, -0.788346427626605890f, -0.614021558931038710f,
    -0.789289253168885430f, -0.612810082429409930f, -0.790230221437309920f,
    -0.611597163926462020f, -0.791169330217690090f, -0.610382806276309360f,
    -0.792106577300212390f, -0.609167012336452980f, -0.793041960479443860f,
    -0.607949784967774080f, -0.793975477554336840f, -0.606731127034524810f,
    -0.794907126328236790f, -0.605511041404325660f, -0.795836904608883460f,
    -0.604289530948156070f, -0.796764810208418720f, -0.603066598540348050f,
    -0.797690840943391160f, -0.601842247058580470f, -0.798614994634760490f,
    -0.600616479383869310f, -0.799537269107904790f, -0.599389298400564760f,
    -0.800457662192622600f, -0.598160706996342380f, -0.801376171723140130f,
    -0.596930708062196390f, -0.802292795538115720f, -0.595699304492433130f,
    -0.803207531480645050f, -0.594466499184664880f, -0.804120377398265470f,
    -0.593232295039800130f, -0.805031331142963440f, -0.591996694962041100f,
    -0.805940390571176170f, -0.590759701858874280f, -0.806847553543799220f,
    -0.589521318641063830f, -0.807752817926190470f, -0.588281548222645780f,
    -0.808656181588174650f, -0.587040393520918300f, -0.809557642404051040f,
    -0.585797857456439090f, -0.810457198252594660f, -0.584553942953015330f,
    -0.811354847017063730f, -0.583308652937698290f, -0.812250586585203990f,
    -0.582061990340775330f, -0.813144414849253700f, -0.580813958095764970f,
    -0.814036329705948080f, -0.579564559139405970f, -0.814926329056526400f,
    -0.578313796411655700f, -0.815814410806733670f, -0.577061672855679550f,
    -0.816700572866827730f, -0.575808191417845230f, -0.817584813151583820f,
    -0.574553355047716320f, -0.818467129580298320f, -0.573297166698042540f,
    -0.819347520076796680f, -0.572039629324757270f, -0.820225982569434460f,
    -0.570780745886967370f, -0.821102514991104650f, -0.569520519346947140f,
    -0.821977115279241550f, -0.568258952670131380f, -0.822849781375826430f,
    -0.566996048825109010f, -0.823720511227391090f, -0.565731810783613450f,
    -0.824589302785025070f, -0.564466241520519610f, -0.825456154004377440f,
    -0.563199344013834090f, -0.826321062845663420f, -0.561931121244689250f,
    -0.827184027273669240f, -0.560661576197336480f, -0.828045045257755460f,
    -0.559390711859136470f, -0.828904114771864650f, -0.558118531220556320f,
    -0.829761233794522930f, -0.556845037275160100f, -0.830616400308846200f,
    -0.555570233019602180f, -0.831469612302545240f, -0.554294121453619890f,
    -0.832320867767929800f, -0.553016705580027910f, -0.833170164701912960f,
    -0.551737988404707670f, -0.834017501106017910f, -0.550457972936604920f,
    -0.834862874986380010f, -0.549176662187719770f, -0.835706284353752600f,
    -0.547894059173100080f, -0.836547727223512120f, -0.546610166910835420f,
    -0.837387201615661600f, -0.545324988422046800f, -0.838224705554837860f,
    -0.544038526730884150f, -0.839060237070312520f, -0.542750784864516000f,
    -0.839893794195999410f, -0.541461765853123440f, -0.840725374970458070f,
    -0.540171472729892740f, -0.841554977436898550f, -0.538879908531008870f,
    -0.842382599643185630f, -0.537587076295645730f, -0.843208239641845210f,
    -0.536292979065963290f, -0.844031895490066300f, -0.534997619887097260f,
    -0.844853565249707010f, -0.533701001807152850f, -0.845673246987299180f,
    -0.532403127877198460f, -0.846490938774051790f, -0.531104001151255330f,
    -0.847306638685858090f, -0.529803624686294940f, -0.848120344803297120f,
    -0.528502001542228590f, -0.848932055211639610f, -0.527199134781901280f,
    -0.849741768000852550f, -0.525895027471084520f, -0.850549481265603590f,
    -0.524589682678469390f, -0.851355193105264860f, -0.523283103475656760f,
    -0.852158901623919610f, -0.521975292937154500f, -0.852960604930363520f,
    -0.520666254140367160f, -0.853760301138111410f, -0.519355990165589420f,
    -0.854557988365400640f, -0.518044504095999890f, -0.855353664735195700f,
    -0.516731799017650210f, -0.856147328375194250f, -0.515417878019463260f,
    -0.856938977417828540f, -0.514102744193221770f, -0.857728610000272010f,
    -0.512786400633562960f, -0.858516224264442850f, -0.511468850437970190f,
    -0.859301818357008470f, -0.510150096706767250f, -0.860085390429389920f,
    -0.508830142543107320f, -0.860866938637767090f, -0.507508991052970980f,
    -0.861646461143081190f, -0.506186645345155340f, -0.862423956111040500f,
    -0.504863108531267370f, -0.863199421712124270f, -0.503538383725718020f,
    -0.863972856121586470f, -0.502212474045711120f, -0.864744257519462160f,
    -0.500885382611241050f, -0.865513624090568980f, -0.499557112545082000f,
    -0.866280954024512880f, -0.498227666972781810f, -0.867046245515692650f,
    -0.496897049022654360f, -0.867809496763303320f, -0.495565261825772980f,
    -0.868570705971340670f, -0.494232308515960060f, -0.869329871348606620f,
    -0.492898192229784200f, -0.870086991108711350f, -0.491562916106549950f,
    -0.870842063470078860f, -0.490226483288291050f, -0.871595086655951090f,
    -0.488888896919763730f, -0.872346058894391210f, -0.487550160148436330f,
    -0.873094978418289870f, -0.486210276124486640f, -0.873841843465366750f,
    -0.484869248000791180f, -0.874586652278176110f, -0.483527078932918690f,
    -0.875329403104110890f, -0.482183772079122550f, -0.876070094195406710f,
    -0.480839330600334400f, -0.876808723809145430f, -0.479493757660153340f,
    -0.877545290207261130f, -0.478147056424843230f, -0.878279791656541460f,
    -0.476799230063322140f, -0.879012226428633410f, -0.475450281747155760f,
    -0.879742592800047520f, -0.474100214650550520f, -0.880470889052160530f,
    -0.472749031950343180f, -0.881197113471221870f, -0.471396736825997860f,
    -0.881921264348354940f, -0.470043332459595730f, -0.882643339979562680f,
    -0.468688822035827900f, -0.883363338665731580f, -0.467333208741988250f,
    -0.884081258712635100f, -0.465976495767966630f, -0.884797098430937570f,
    -0.464618686306238150f, -0.885510856136199730f, -0.463259783551860370f,
    -0.886222530148880530f, -0.461899790702462790f, -0.886932118794342190f,
    -0.460538710958239890f, -0.887639620402854050f, -0.459176547521944640f,
    -0.888345033309596020f, -0.457813303598877620f, -0.889048355854664350f,
    -0.456448982396884200f, -0.889749586383072670f, -0.455083587126343950f,
    -0.890448723244757880f, -0.453717121000163810f, -0.891145764794583290f,
    -0.452349587233770670f, -0.891840709392342830f, -0.450980989045104310f,
    -0.892533555402764360f, -0.449611329654606930f, -0.893224301195515210f,
    -0.448240612285220110f, -0.893912945145203140f, -0.446868840162374210f,
    -0.894599485631382700f, -0.445496016513981630f, -0.895283921038557580f,
    -0.444122144570429760f, -0.895966249756184880f, -0.442747227564570410f,
    -0.896646470178680040f, -0.441371268731716950f, -0.897324580705418210f,
    -0.439994271309633370f, -0.898000579740739770f, -0.438616238538527600f,
    -0.898674465693953820f, -0.437237173661043920f, -0.899346236979341680f,
    -0.435857079922255970f, -0.900015892016159950f, -0.434475960569656040f,
    -0.900683429228646750f, -0.433093818853152120f, -0.901348847046021920f,
    -0.431710658025057310f, -0.902012143902493180f, -0.430326481340082500f,
    -0.902673318237258830f, -0.428941292055330050f, -0.903332368494511600f,
    -0.427555093430282470f, -0.903989293123443120f, -0.426167888726799890f,
    -0.904644090578246130f, -0.424779681209108920f, -0.905296759318118700f,
    -0.423390474143795990f, -0.905947297807268460f, -0.422000270799799520f,
    -0.906595704514915450f, -0.420609074448403010f, -0.907241977915295590f,
    -0.419216888363224290f, -0.907886116487666040f, -0.417823715820212490f,
    -0.908528118716306010f, -0.416429560097637210f, -0.909167983090522380f,
    -0.415034424476081520f, -0.909805708104652330f, -0.413638312238435110f,
    -0.910441292258066910f, -0.412241226669883280f, -0.911074734055176140f,
    -0.410843171057904190f, -0.911706032005429770f, -0.409444148692257760f,
    -0.912335184623322750f, -0.408044162864978630f, -0.912962190428398210f,
    -0.406643216870368810f, -0.913587047945250920f, -0.405241314004990360f,
    -0.914209755703530470f, -0.403838457567654460f, -0.914830312237945980f,
    -0.402434650859418650f, -0.915448716088267720f, -0.401029897183575680f,
    -0.916064965799331720f, -0.399624199845646730f, -0.916679059921042700f,
    -0.398217562153374170f, -0.917290997008377680f, -0.396809987416710750f,
    -0.917900775621390270f, -0.395401478947816580f, -0.918508394325212140f,
    -0.393992040061048210f, -0.919113851690057660f, -0.392581674072951470f,
    -0.919717146291227360f, -0.391170384302253700f, -0.920318276709110590f,
    -0.389758174069856970f, -0.920917241529189300f, -0.388345046698826630f,
    -0.921514039342041790f, -0.386931005514388800f, -0.922108668743345070f,
    -0.385516053843918900f, -0.922701128333878520f, -0.384100195016934930f,
    -0.923291416719527640f, -0.382683432365090340f, -0.923879532511286520f,
    -0.381265769222162760f, -0.924465474325262490f, -0.379847208924051440f,
    -0.925049240782677470f, -0.378427754808765730f, -0.925630830509872720f,
    -0.377007410216418200f, -0.926210242138311380f, -0.375586178489217050f,
    -0.926787474304581860f, -0.374164062971458490f, -0.927362525650400890f,
    -0.372741067009516150f, -0.927935394822617670f, -0.371317193951837770f,
    -0.928506080473215480f, -0.369892447148934160f, -0.929074581259315750f,
    -0.368466829953372210f, -0.929640895843181330f, -0.367040345719766960f,
    -0.930205022892219180f, -0.365612997804774300f, -0.930766961078983600f,
    -0.364184789567080170f, -0.931326709081180320f, -0.362755724367397340f,
    -0.931884265581668040f, -0.361325805568454230f, -0.932439629268462360f,
    -0.359895036534987940f, -0.932992798834738960f, -0.358463420633737040f,
    -0.933543772978835950f, -0.357030961233430370f, -0.934092550404258760f,
    -0.355597661704784070f, -0.934639129819680670f, -0.354163525420490450f,
    -0.935183509938947610f, -0.352728555755210620f, -0.935725689481080370f,
    -0.351292756085566870f, -0.936265667170278370f, -0.349856129790135360f,
    -0.936803441735921450f, -0.348418680249434840f, -0.937339011912574850f,
    -0.346980410845923790f, -0.937872376439989770f, -0.345541324963989040f,
    -0.938403534063108170f, -0.344101425989938650f, -0.938932483532064600f,
    -0.342660717311994880f, -0.939459223602189700f, -0.341219202320282740f,
    -0.939983753034013820f, -0.339776884406827070f, -0.940506070593268300f,
    -0.338333766965541240f, -0.941026175050889260f, -0.336889853392219940f,
    -0.941544065183020810f, -0.335445147084531380f, -0.942059739771017420f,
    -0.333999651442009830f, -0.942573197601446760f, -0.332553369866044500f,
    -0.943084437466093380f, -0.331106305759876540f, -0.943593458161960270f,
    -0.329658462528587490f, -0.944100258491272660f, -0.328209843579092330f,
    -0.944604837261480370f, -0.326760452320132290f, -0.945107193285260380f,
    -0.325310292162263310f, -0.945607325380521170f, -0.323859366517853080f,
    -0.946105232370403340f, -0.322407678801069910f, -0.946600913083283530f,
    -0.320955232427875160f, -0.947094366352777220f, -0.319502030816015410f,
    -0.947585591017741200f, -0.318048077385015390f, -0.948074585922276110f,
    -0.316593375556166180f, -0.948561349915730160f, -0.315137928752522560f,
    -0.949045881852700560f, -0.313681740398891460f, -0.949528180593036670f,
    -0.312224813921824770f, -0.950008245001843110f, -0.310767152749612030f,
    -0.950486073949481590f, -0.309308760312269060f, -0.950961666311574970f,
    -0.307849640041535090f, -0.951435020969008340f, -0.306389795370861030f,
    -0.951906136807932350f, -0.304929229735402320f, -0.952375012719765880f,
    -0.303467946572011040f, -0.952841647601198720f, -0.302005949319228530f,
    -0.953306040354193750f, -0.300543241417273730f, -0.953768189885990210f,
    -0.299079826308040590f, -0.954228095109105560f, -0.297615707435086200f,
    -0.954685754941338340f, -0.296150888243623680f, -0.955141168305770780f,
    -0.294685372180514880f, -0.955594334130770880f, -0.293219162694259020f,
    -0.956045251349996290f, -0.291752263234989480f, -0.956493918902394990f,
    -0.290284677254462440f, -0.956940335732208820f, -0.288816408206049370f,
    -0.957384500788975970f, -0.287347459544729290f, -0.957826413027533020f,
    -0.285877834727081060f, -0.958266071408017560f, -0.284407537211272150f,
    -0.958703474895871490f, -0.282936570457055500f, -0.959138622461841890f,
    -0.281464937925757940f, -0.959571513081984520f, -0.279992643080273050f,
    -0.960002145737665960f, -0.278519689385053610f, -0.960430519415565680f,
    -0.277046080306100280f, -0.960856633107679550f, -0.275571819310958370f,
    -0.961280485811320530f, -0.274096909868706440f, -0.961702076529122540f,
    -0.272621355449948870f, -0.962121404269041580f, -0.271145159526807790f,
    -0.962538468044359270f, -0.269668325572915530f, -0.962953266873683770f,
    -0.268190857063403510f, -0.963365799780953940f, -0.266712757474898530f,
    -0.963776065795439840f, -0.265234030285511790f, -0.964184063951745830f,
    -0.263754678974831240f, -0.964589793289812760f, -0.262274707023914140f,
    -0.964993252854920210f, -0.260794117915275900f, -0.965394441697689290f,
    -0.259312915132886460f, -0.965793358874083570f, -0.257831102162159040f,
    -0.966190003445412500f, -0.256348682489942800f, -0.966584374478333120f,
    -0.254865659604514350f, -0.966976471044852180f, -0.253382036995570600f,
    -0.967366292222328390f, -0.251897818154217250f, -0.967753837093475400f,
    -0.250413006572965390f, -0.968139104746362330f, -0.248927605745720150f,
    -0.968522094274417270f, -0.247441619167773130f, -0.968902804776428870f,
    -0.245955050335795150f, -0.969281235356548310f, -0.244467902747824540f,
    -0.969657385124292340f, -0.242980179903264120f, -0.970031253194543970f,
    -0.241491885302869410f, -0.970402838687555500f, -0.240003022448741390f,
    -0.970772140728950350f, -0.238513594844318190f, -0.971139158449725200f,
    -0.237023605994367670f, -0.971503890986251670f, -0.235533059404975790f,
    -0.971866337480279290f, -0.234041958583543570f, -0.972226497078936270f,
    -0.232550307038775220f, -0.972584368934732210f, -0.231058108280670940f,
    -0.972939952205560180f, -0.229565365820519420f, -0.973293246054698140f,
    -0.228072083170886120f, -0.973644249650811870f, -0.226578263845610220f,
    -0.973992962167955830f, -0.225083911359792920f, -0.974339382785575860f,
    -0.223589029229789900f, -0.974683510688510670f, -0.222093620973203290f,
    -0.975025345066994230f, -0.220597690108873980f, -0.975364885116656870f,
    -0.219101240156870100f, -0.975702130038528460f, -0.217604274638483780f,
    -0.976037079039039020f, -0.216106797076219490f, -0.976369731330021140f,
    -0.214608810993786620f, -0.976700086128711840f, -0.213110319916091920f,
    -0.977028142657754280f, -0.211611327369227970f, -0.977353900145199960f,
    -0.210111836880469860f, -0.977677357824509930f, -0.208611851978263570f,
    -0.977998514934557030f, -0.207111376192218480f, -0.978317370719627650f,
    -0.205610413053099020f, -0.978633924429423210f, -0.204108966092817340f,
    -0.978948175319062090f, -0.202607038844421440f, -0.979260122649082020f,
    -0.201104634842092070f, -0.979569765685440520f, -0.199601757621130970f,
    -0.979877103699517640f, -0.198098410717953420f, -0.980182135968117430f,
    -0.196594597670080780f, -0.980484861773469270f, -0.195090322016128660f,
    -0.980785280403230320f, -0.193585587295803860f, -0.981083391150486590f,
    -0.192080397049892520f, -0.981379193313754560f, -0.190574754820252680f,
    -0.981672686196983110f, -0.189068664149805970f, -0.981963869109555350f,
    -0.187562128582530070f, -0.982252741366289370f, -0.186055151663446970f,
    -0.982539302287441240f, -0.184547736938619780f, -0.982823551198705240f,
    -0.183039887955140950f, -0.983105487431216290f, -0.181531608261124830f,
    -0.983385110321551290f, -0.180022901405700070f, -0.983662419211730140f,
    -0.178513770938997920f, -0.983937413449218810f, -0.177004220412149000f,
    -0.984210092386929030f, -0.175494253377271510f, -0.984480455383220930f,
    -0.173983873387463740f, -0.984748501801904210f, -0.172473083996795730f,
    -0.985014231012239840f, -0.170961888760301690f, -0.985277642388941110f,
    -0.169450291233968290f, -0.985538735312176060f, -0.167938294974731340f,
    -0.985797509167567370f, -0.166425903540464100f, -0.986053963346195440f,
    -0.164913120489969760f, -0.986308097244598670f, -0.163399949382973780f,
    -0.986559910264775410f, -0.161886393780112240f, -0.986809401814185420f,
    -0.160372457242928510f, -0.987056571305750970f, -0.158858143333861530f,
    -0.987301418157858320f, -0.157343455616238190f, -0.987543941794359340f,
    -0.155828397654264980f, -0.987784141644572180f, -0.154312973013020580f,
    -0.988022017143283530f, -0.152797185258443740f, -0.988257567730749460f,
    -0.151281037957330360f, -0.988490792852696590f, -0.149764534677321510f,
    -0.988721691960323780f, -0.148247678986895890f, -0.988950264510302990f,
    -0.146730474455362300f, -0.989176509964780900f, -0.145212924652847850f,
    -0.989400427791380270f, -0.143695033150294690f, -0.989622017463200780f,
    -0.142176803519448140f, -0.989841278458820530f, -0.140658239332849160f,
    -0.990058210262297120f, -0.139139344163825980f, -0.990272812363169110f,
    -0.137620121586486540f, -0.990485084256456980f, -0.136100575175706530f,
    -0.990695025442664630f, -0.134580708507126360f, -0.990902635427780010f,
    -0.133060525157139060f, -0.991107913723276890f, -0.131540028702882950f,
    -0.991310859846115440f, -0.130019222722233930f, -0.991511473318743900f,
    -0.128498110793793590f, -0.991709753669099530f, -0.126976696496886120f,
    -0.991905700430609330f, -0.125454983411546320f, -0.992099313142191800f,
    -0.123932975118512090f, -0.992290591348257370f, -0.122410675199215960f,
    -0.992479534598710080f, -0.120888087235777570f, -0.992666142448947910f,
    -0.119365214810991690f, -0.992850414459865100f, -0.117842061508325140f,
    -0.993032350197851410f, -0.116318630911904770f, -0.993211949234794500f,
    -0.114794926606509930f, -0.993389211148080650f, -0.113270952177564920f,
    -0.993564135520595300f, -0.111746711211127000f, -0.993736721940724600f,
    -0.110222207293883310f, -0.993906970002356060f, -0.108697444013138800f,
    -0.994074879304879370f, -0.107172424956808770f, -0.994240449453187900f,
    -0.105647153713410380f, -0.994403680057679100f, -0.104121633872055070f,
    -0.994564570734255420f, -0.102595869022436610f, -0.994723121104325700f,
    -0.101069862754827990f, -0.994879330794805620f, -0.099543618660069347f,
    -0.995033199438118630f, -0.098017140329560451f, -0.995184726672196930f,
    -0.096490431355253162f, -0.995333912140482170f, -0.094963495329639408f,
    -0.995480755491926940f, -0.093436335845748036f, -0.995625256380994310f,
    -0.091908956497132821f, -0.995767414467659820f, -0.090381360877864914f,
    -0.995907229417411720f, -0.088853552582524364f, -0.996044700901251970f,
    -0.087325535206192559f, -0.996179828595696870f, -0.085797312344440227f,
    -0.996312612182778000f, -0.084268887593324238f, -0.996443051350042630f,
    -0.082740264549375706f, -0.996571145790554840f, -0.081211446809592289f,
    -0.996696895202896060f, -0.079682437971430695f, -0.996820299291165670f,
    -0.078153241632794648f, -0.996941357764982050f, -0.076623861392031742f,
    -0.997060070339482960f, -0.075094300847921402f, -0.997176436735326080f,
    -0.073564563599667357f, -0.997290456678690210f, -0.072034653246889097f,
    -0.997402129901275300f, -0.070504573389614356f, -0.997511456140303450f,
    -0.068974327628267079f, -0.997618435138519550f, -0.067443919563664231f,
    -0.997723066644191640f, -0.065913352797003832f, -0.997825350411111640f,
    -0.064382630929857312f, -0.997925286198596000f, -0.062851757564161989f,
    -0.998022873771486130f, -0.061320736302208995f, -0.998118112900149180f,
    -0.059789570746640132f, -0.998211003360478190f, -0.058258264500435857f,
    -0.998301544933892780f, -0.056726821166907686f, -0.998389737407340160f,
    -0.055195244349689712f, -0.998475580573294770f, -0.053663537652731026f,
    -0.998559074229759310f, -0.052131704680283657f, -0.998640218180265160f,
    -0.050599749036899455f, -0.998719012233872940f, -0.049067674327418029f,
    -0.998795456205172410f, -0.047535484156959157f, -0.998869549914283560f,
    -0.046003182130915206f, -0.998941293186856870f, -0.044470771854939084f,
    -0.999010685854073270f, -0.042938256934941084f, -0.999077727752645360f,
    -0.041405640977076837f, -0.999142418724816910f, -0.039872927587739748f,
    -0.999204758618363890f, -0.038340120373552472f, -0.999264747286594420f,
    -0.036807222941359331f, -0.999322384588349430f, -0.035274238898214294f,
    -0.999377670388002850f, -0.033741171851377760f, -0.999430604555461730f,
    -0.032208025408304600f, -0.999481186966166950f, -0.030674803176636484f,
    -0.999529417501093140f, -0.029141508764194309f, -0.999575296046749220f,
    -0.027608145778966163f, -0.999618822495178640f, -0.026074717829104161f,
    -0.999659996743959220f, -0.024541228522912389f, -0.999698818696204250f,
    -0.023007681468839310f, -0.999735288260561680f, -0.021474080275469286f,
    -0.999769405351215280f, -0.019940428551514944f, -0.999801169887884260f,
    -0.018406729905805164f, -0.999830581795823400f, -0.016872987947281894f,
    -0.999857641005823860f, -0.015339206284988121f, -0.999882347454212560f,
    -0.013805388528060250f, -0.999904701082852900f, -0.012271538285720512f,
    -0.999924701839144500f, -0.010737659167264916f, -0.999942349676023910f,
    -0.009203754782060083f, -0.999957644551963900f, -0.007669828739531199f,
    -0.999970586430974140f, -0.006135884649154416f, -0.999981175282601110f,
    -0.004601926120448350f, -0.999989411081928400f, -0.003067956762966483f,
    -0.999995293809576190f, -0.001533980186285111f, -0.999998823451701880f,
};

/*
 * @brief  Q31 Twiddle factors Table
 */

/**
 * \par
 * Example code for Q31 Twiddle factors Generation::
 * \par
 * <pre>for(i = 0; i< 3N/4; i++)
 * {
 *    twiddleCoefQ31[2*i]= cos(i * 2*PI/(float)N);
 *    twiddleCoefQ31[2*i+1]= sin(i * 2*PI/(float)N);
 * } </pre>
 * \par
 * where N = 4096	and PI = 3.14159265358979
 * \par
 * Cos and Sin values are interleaved fashion
 * \par
 * Convert Floating point to Q31(Fixed point 1.31):
 *	round(twiddleCoefQ31(i) * pow(2, 31))
 *
 */

const q31_t twiddleCoefQ31[6144] = {
    0x7fffffff, 0x0,        0x7ffff621, 0x3243f5,   0x7fffd886, 0x6487e3,
    0x7fffa72c, 0x96cbc1,   0x7fff6216, 0xc90f88,   0x7fff0943, 0xfb5330,
    0x7ffe9cb2, 0x12d96b1,  0x7ffe1c65, 0x15fda03,  0x7ffd885a, 0x1921d20,
    0x7ffce093, 0x1c45ffe,  0x7ffc250f, 0x1f6a297,  0x7ffb55ce, 0x228e4e2,
    0x7ffa72d1, 0x25b26d7,  0x7ff97c18, 0x28d6870,  0x7ff871a2, 0x2bfa9a4,
    0x7ff75370, 0x2f1ea6c,  0x7ff62182, 0x3242abf,  0x7ff4dbd9, 0x3566a96,
    0x7ff38274, 0x388a9ea,  0x7ff21553, 0x3bae8b2,  0x7ff09478, 0x3ed26e6,
    0x7feeffe1, 0x41f6480,  0x7fed5791, 0x451a177,  0x7feb9b85, 0x483ddc3,
    0x7fe9cbc0, 0x4b6195d,  0x7fe7e841, 0x4e8543e,  0x7fe5f108, 0x51a8e5c,
    0x7fe3e616, 0x54cc7b1,  0x7fe1c76b, 0x57f0035,  0x7fdf9508, 0x5b137df,
    0x7fdd4eec, 0x5e36ea9,  0x7fdaf519, 0x615a48b,  0x7fd8878e, 0x647d97c,
    0x7fd6064c, 0x67a0d76,  0x7fd37153, 0x6ac406f,  0x7fd0c8a3, 0x6de7262,
    0x7fce0c3e, 0x710a345,  0x7fcb3c23, 0x742d311,  0x7fc85854, 0x77501be,
    0x7fc560cf, 0x7a72f45,  0x7fc25596, 0x7d95b9e,  0x7fbf36aa, 0x80b86c2,
    0x7fbc040a, 0x83db0a7,  0x7fb8bdb8, 0x86fd947,  0x7fb563b3, 0x8a2009a,
    0x7fb1f5fc, 0x8d42699,  0x7fae7495, 0x9064b3a,  0x7faadf7c, 0x9386e78,
    0x7fa736b4, 0x96a9049,  0x7fa37a3c, 0x99cb0a7,  0x7f9faa15, 0x9cecf89,
    0x7f9bc640, 0xa00ece8,  0x7f97cebd, 0xa3308bd,  0x7f93c38c, 0xa6522fe,
    0x7f8fa4b0, 0xa973ba5,  0x7f8b7227, 0xac952aa,  0x7f872bf3, 0xafb6805,
    0x7f82d214, 0xb2d7baf,  0x7f7e648c, 0xb5f8d9f,  0x7f79e35a, 0xb919dcf,
    0x7f754e80, 0xbc3ac35,  0x7f70a5fe, 0xbf5b8cb,  0x7f6be9d4, 0xc27c389,
    0x7f671a05, 0xc59cc68,  0x7f62368f, 0xc8bd35e,  0x7f5d3f75, 0xcbdd865,
    0x7f5834b7, 0xcefdb76,  0x7f531655, 0xd21dc87,  0x7f4de451, 0xd53db92,
    0x7f489eaa, 0xd85d88f,  0x7f434563, 0xdb7d376,  0x7f3dd87c, 0xde9cc40,
    0x7f3857f6, 0xe1bc2e4,  0x7f32c3d1, 0xe4db75b,  0x7f2d1c0e, 0xe7fa99e,
    0x7f2760af, 0xeb199a4,  0x7f2191b4, 0xee38766,  0x7f1baf1e, 0xf1572dc,
    0x7f15b8ee, 0xf475bff,  0x7f0faf25, 0xf7942c7,  0x7f0991c4, 0xfab272b,
    0x7f0360cb, 0xfdd0926,  0x7efd1c3c, 0x100ee8ad, 0x7ef6c418, 0x1040c5bb,
    0x7ef05860, 0x1072a048, 0x7ee9d914, 0x10a4784b, 0x7ee34636, 0x10d64dbd,
    0x7edc9fc6, 0x11082096, 0x7ed5e5c6, 0x1139f0cf, 0x7ecf1837, 0x116bbe60,
    0x7ec8371a, 0x119d8941, 0x7ec14270, 0x11cf516a, 0x7eba3a39, 0x120116d5,
    0x7eb31e78, 0x1232d979, 0x7eabef2c, 0x1264994e, 0x7ea4ac58, 0x1296564d,
    0x7e9d55fc, 0x12c8106f, 0x7e95ec1a, 0x12f9c7aa, 0x7e8e6eb2, 0x132b7bf9,
    0x7e86ddc6, 0x135d2d53, 0x7e7f3957, 0x138edbb1, 0x7e778166, 0x13c0870a,
    0x7e6fb5f4, 0x13f22f58, 0x7e67d703, 0x1423d492, 0x7e5fe493, 0x145576b1,
    0x7e57dea7, 0x148715ae, 0x7e4fc53e, 0x14b8b17f, 0x7e47985b, 0x14ea4a1f,
    0x7e3f57ff, 0x151bdf86, 0x7e37042a, 0x154d71aa, 0x7e2e9cdf, 0x157f0086,
    0x7e26221f, 0x15b08c12, 0x7e1d93ea, 0x15e21445, 0x7e14f242, 0x16139918,
    0x7e0c3d29, 0x16451a83, 0x7e0374a0, 0x1676987f, 0x7dfa98a8, 0x16a81305,
    0x7df1a942, 0x16d98a0c, 0x7de8a670, 0x170afd8d, 0x7ddf9034, 0x173c6d80,
    0x7dd6668f, 0x176dd9de, 0x7dcd2981, 0x179f429f, 0x7dc3d90d, 0x17d0a7bc,
    0x7dba7534, 0x1802092c, 0x7db0fdf8, 0x183366e9, 0x7da77359, 0x1864c0ea,
    0x7d9dd55a, 0x18961728, 0x7d9423fc, 0x18c7699b, 0x7d8a5f40, 0x18f8b83c,
    0x7d808728, 0x192a0304, 0x7d769bb5, 0x195b49ea, 0x7d6c9ce9, 0x198c8ce7,
    0x7d628ac6, 0x19bdcbf3, 0x7d58654d, 0x19ef0707, 0x7d4e2c7f, 0x1a203e1b,
    0x7d43e05e, 0x1a517128, 0x7d3980ec, 0x1a82a026, 0x7d2f0e2b, 0x1ab3cb0d,
    0x7d24881b, 0x1ae4f1d6, 0x7d19eebf, 0x1b161479, 0x7d0f4218, 0x1b4732ef,
    0x7d048228, 0x1b784d30, 0x7cf9aef0, 0x1ba96335, 0x7ceec873, 0x1bda74f6,
    0x7ce3ceb2, 0x1c0b826a, 0x7cd8c1ae, 0x1c3c8b8c, 0x7ccda169, 0x1c6d9053,
    0x7cc26de5, 0x1c9e90b8, 0x7cb72724, 0x1ccf8cb3, 0x7cabcd28, 0x1d00843d,
    0x7ca05ff1, 0x1d31774d, 0x7c94df83, 0x1d6265dd, 0x7c894bde, 0x1d934fe5,
    0x7c7da505, 0x1dc4355e, 0x7c71eaf9, 0x1df5163f, 0x7c661dbc, 0x1e25f282,
    0x7c5a3d50, 0x1e56ca1e, 0x7c4e49b7, 0x1e879d0d, 0x7c4242f2, 0x1eb86b46,
    0x7c362904, 0x1ee934c3, 0x7c29fbee, 0x1f19f97b, 0x7c1dbbb3, 0x1f4ab968,
    0x7c116853, 0x1f7b7481, 0x7c0501d2, 0x1fac2abf, 0x7bf88830, 0x1fdcdc1b,
    0x7bebfb70, 0x200d888d, 0x7bdf5b94, 0x203e300d, 0x7bd2a89e, 0x206ed295,
    0x7bc5e290, 0x209f701c, 0x7bb9096b, 0x20d0089c, 0x7bac1d31, 0x21009c0c,
    0x7b9f1de6, 0x21312a65, 0x7b920b89, 0x2161b3a0, 0x7b84e61f, 0x219237b5,
    0x7b77ada8, 0x21c2b69c, 0x7b6a6227, 0x21f3304f, 0x7b5d039e, 0x2223a4c5,
    0x7b4f920e, 0x225413f8, 0x7b420d7a, 0x22847de0, 0x7b3475e5, 0x22b4e274,
    0x7b26cb4f, 0x22e541af, 0x7b190dbc, 0x23159b88, 0x7b0b3d2c, 0x2345eff8,
    0x7afd59a4, 0x23763ef7, 0x7aef6323, 0x23a6887f, 0x7ae159ae, 0x23d6cc87,
    0x7ad33d45, 0x24070b08, 0x7ac50dec, 0x243743fa, 0x7ab6cba4, 0x24677758,
    0x7aa8766f, 0x2497a517, 0x7a9a0e50, 0x24c7cd33, 0x7a8b9348, 0x24f7efa2,
    0x7a7d055b, 0x25280c5e, 0x7a6e648a, 0x2558235f, 0x7a5fb0d8, 0x2588349d,
    0x7a50ea47, 0x25b84012, 0x7a4210d8, 0x25e845b6, 0x7a332490, 0x26184581,
    0x7a24256f, 0x26483f6c, 0x7a151378, 0x26783370, 0x7a05eead, 0x26a82186,
    0x79f6b711, 0x26d809a5, 0x79e76ca7, 0x2707ebc7, 0x79d80f6f, 0x2737c7e3,
    0x79c89f6e, 0x27679df4, 0x79b91ca4, 0x27976df1, 0x79a98715, 0x27c737d3,
    0x7999dec4, 0x27f6fb92, 0x798a23b1, 0x2826b928, 0x797a55e0, 0x2856708d,
    0x796a7554, 0x288621b9, 0x795a820e, 0x28b5cca5, 0x794a7c12, 0x28e5714b,
    0x793a6361, 0x29150fa1, 0x792a37fe, 0x2944a7a2, 0x7919f9ec, 0x29743946,
    0x7909a92d, 0x29a3c485, 0x78f945c3, 0x29d34958, 0x78e8cfb2, 0x2a02c7b8,
    0x78d846fb, 0x2a323f9e, 0x78c7aba2, 0x2a61b101, 0x78b6fda8, 0x2a911bdc,
    0x78a63d11, 0x2ac08026, 0x789569df, 0x2aefddd8, 0x78848414, 0x2b1f34eb,
    0x78738bb3, 0x2b4e8558, 0x786280bf, 0x2b7dcf17, 0x7851633b, 0x2bad1221,
    0x78403329, 0x2bdc4e6f, 0x782ef08b, 0x2c0b83fa, 0x781d9b65, 0x2c3ab2b9,
    0x780c33b8, 0x2c69daa6, 0x77fab989, 0x2c98fbba, 0x77e92cd9, 0x2cc815ee,
    0x77d78daa, 0x2cf72939, 0x77c5dc01, 0x2d263596, 0x77b417df, 0x2d553afc,
    0x77a24148, 0x2d843964, 0x7790583e, 0x2db330c7, 0x777e5cc3, 0x2de2211e,
    0x776c4edb, 0x2e110a62, 0x775a2e89, 0x2e3fec8b, 0x7747fbce, 0x2e6ec792,
    0x7735b6af, 0x2e9d9b70, 0x77235f2d, 0x2ecc681e, 0x7710f54c, 0x2efb2d95,
    0x76fe790e, 0x2f29ebcc, 0x76ebea77, 0x2f58a2be, 0x76d94989, 0x2f875262,
    0x76c69647, 0x2fb5fab2, 0x76b3d0b4, 0x2fe49ba7, 0x76a0f8d2, 0x30133539,
    0x768e0ea6, 0x3041c761, 0x767b1231, 0x30705217, 0x76680376, 0x309ed556,
    0x7654e279, 0x30cd5115, 0x7641af3d, 0x30fbc54d, 0x762e69c4, 0x312a31f8,
    0x761b1211, 0x3158970e, 0x7607a828, 0x3186f487, 0x75f42c0b, 0x31b54a5e,
    0x75e09dbd, 0x31e39889, 0x75ccfd42, 0x3211df04, 0x75b94a9c, 0x32401dc6,
    0x75a585cf, 0x326e54c7, 0x7591aedd, 0x329c8402, 0x757dc5ca, 0x32caab6f,
    0x7569ca99, 0x32f8cb07, 0x7555bd4c, 0x3326e2c3, 0x75419de7, 0x3354f29b,
    0x752d6c6c, 0x3382fa88, 0x751928e0, 0x33b0fa84, 0x7504d345, 0x33def287,
    0x74f06b9e, 0x340ce28b, 0x74dbf1ef, 0x343aca87, 0x74c7663a, 0x3468aa76,
    0x74b2c884, 0x34968250, 0x749e18cd, 0x34c4520d, 0x7489571c, 0x34f219a8,
    0x74748371, 0x351fd918, 0x745f9dd1, 0x354d9057, 0x744aa63f, 0x357b3f5d,
    0x74359cbd, 0x35a8e625, 0x74208150, 0x35d684a6, 0x740b53fb, 0x36041ad9,
    0x73f614c0, 0x3631a8b8, 0x73e0c3a3, 0x365f2e3b, 0x73cb60a8, 0x368cab5c,
    0x73b5ebd1, 0x36ba2014, 0x73a06522, 0x36e78c5b, 0x738acc9e, 0x3714f02a,
    0x73752249, 0x37424b7b, 0x735f6626, 0x376f9e46, 0x73499838, 0x379ce885,
    0x7333b883, 0x37ca2a30, 0x731dc70a, 0x37f76341, 0x7307c3d0, 0x382493b0,
    0x72f1aed9, 0x3851bb77, 0x72db8828, 0x387eda8e, 0x72c54fc1, 0x38abf0ef,
    0x72af05a7, 0x38d8fe93, 0x7298a9dd, 0x39060373, 0x72823c67, 0x3932ff87,
    0x726bbd48, 0x395ff2c9, 0x72552c85, 0x398cdd32, 0x723e8a20, 0x39b9bebc,
    0x7227d61c, 0x39e6975e, 0x7211107e, 0x3a136712, 0x71fa3949, 0x3a402dd2,
    0x71e35080, 0x3a6ceb96, 0x71cc5626, 0x3a99a057, 0x71b54a41, 0x3ac64c0f,
    0x719e2cd2, 0x3af2eeb7, 0x7186fdde, 0x3b1f8848, 0x716fbd68, 0x3b4c18ba,
    0x71586b74, 0x3b78a007, 0x71410805, 0x3ba51e29, 0x7129931f, 0x3bd19318,
    0x71120cc5, 0x3bfdfecd, 0x70fa74fc, 0x3c2a6142, 0x70e2cbc6, 0x3c56ba70,
    0x70cb1128, 0x3c830a50, 0x70b34525, 0x3caf50da, 0x709b67c0, 0x3cdb8e09,
    0x708378ff, 0x3d07c1d6, 0x706b78e3, 0x3d33ec39, 0x70536771, 0x3d600d2c,
    0x703b44ad, 0x3d8c24a8, 0x7023109a, 0x3db832a6, 0x700acb3c, 0x3de4371f,
    0x6ff27497, 0x3e10320d, 0x6fda0cae, 0x3e3c2369, 0x6fc19385, 0x3e680b2c,
    0x6fa90921, 0x3e93e950, 0x6f906d84, 0x3ebfbdcd, 0x6f77c0b3, 0x3eeb889c,
    0x6f5f02b2, 0x3f1749b8, 0x6f463383, 0x3f430119, 0x6f2d532c, 0x3f6eaeb8,
    0x6f1461b0, 0x3f9a5290, 0x6efb5f12, 0x3fc5ec98, 0x6ee24b57, 0x3ff17cca,
    0x6ec92683, 0x401d0321, 0x6eaff099, 0x40487f94, 0x6e96a99d, 0x4073f21d,
    0x6e7d5193, 0x409f5ab6, 0x6e63e87f, 0x40cab958, 0x6e4a6e66, 0x40f60dfb,
    0x6e30e34a, 0x4121589b, 0x6e174730, 0x414c992f, 0x6dfd9a1c, 0x4177cfb1,
    0x6de3dc11, 0x41a2fc1a, 0x6dca0d14, 0x41ce1e65, 0x6db02d29, 0x41f93689,
    0x6d963c54, 0x42244481, 0x6d7c3a98, 0x424f4845, 0x6d6227fa, 0x427a41d0,
    0x6d48047e, 0x42a5311b, 0x6d2dd027, 0x42d0161e, 0x6d138afb, 0x42faf0d4,
    0x6cf934fc, 0x4325c135, 0x6cdece2f, 0x4350873c, 0x6cc45698, 0x437b42e1,
    0x6ca9ce3b, 0x43a5f41e, 0x6c8f351c, 0x43d09aed, 0x6c748b3f, 0x43fb3746,
    0x6c59d0a9, 0x4425c923, 0x6c3f055d, 0x4450507e, 0x6c242960, 0x447acd50,
    0x6c093cb6, 0x44a53f93, 0x6bee3f62, 0x44cfa740, 0x6bd3316a, 0x44fa0450,
    0x6bb812d1, 0x452456bd, 0x6b9ce39b, 0x454e9e80, 0x6b81a3cd, 0x4578db93,
    0x6b66536b, 0x45a30df0, 0x6b4af279, 0x45cd358f, 0x6b2f80fb, 0x45f7526b,
    0x6b13fef5, 0x4621647d, 0x6af86c6c, 0x464b6bbe, 0x6adcc964, 0x46756828,
    0x6ac115e2, 0x469f59b4, 0x6aa551e9, 0x46c9405c, 0x6a897d7d, 0x46f31c1a,
    0x6a6d98a4, 0x471cece7, 0x6a51a361, 0x4746b2bc, 0x6a359db9, 0x47706d93,
    0x6a1987b0, 0x479a1d67, 0x69fd614a, 0x47c3c22f, 0x69e12a8c, 0x47ed5be6,
    0x69c4e37a, 0x4816ea86, 0x69a88c19, 0x48406e08, 0x698c246c, 0x4869e665,
    0x696fac78, 0x48935397, 0x69532442, 0x48bcb599, 0x69368bce, 0x48e60c62,
    0x6919e320, 0x490f57ee, 0x68fd2a3d, 0x49389836, 0x68e06129, 0x4961cd33,
    0x68c387e9, 0x498af6df, 0x68a69e81, 0x49b41533, 0x6889a4f6, 0x49dd282a,
    0x686c9b4b, 0x4a062fbd, 0x684f8186, 0x4a2f2be6, 0x683257ab, 0x4a581c9e,
    0x68151dbe, 0x4a8101de, 0x67f7d3c5, 0x4aa9dba2, 0x67da79c3, 0x4ad2a9e2,
    0x67bd0fbd, 0x4afb6c98, 0x679f95b7, 0x4b2423be, 0x67820bb7, 0x4b4ccf4d,
    0x676471c0, 0x4b756f40, 0x6746c7d8, 0x4b9e0390, 0x67290e02, 0x4bc68c36,
    0x670b4444, 0x4bef092d, 0x66ed6aa1, 0x4c177a6e, 0x66cf8120, 0x4c3fdff4,
    0x66b187c3, 0x4c6839b7, 0x66937e91, 0x4c9087b1, 0x6675658c, 0x4cb8c9dd,
    0x66573cbb, 0x4ce10034, 0x66390422, 0x4d092ab0, 0x661abbc5, 0x4d31494b,
    0x65fc63a9, 0x4d595bfe, 0x65ddfbd3, 0x4d8162c4, 0x65bf8447, 0x4da95d96,
    0x65a0fd0b, 0x4dd14c6e, 0x65826622, 0x4df92f46, 0x6563bf92, 0x4e210617,
    0x6545095f, 0x4e48d0dd, 0x6526438f, 0x4e708f8f, 0x65076e25, 0x4e984229,
    0x64e88926, 0x4ebfe8a5, 0x64c99498, 0x4ee782fb, 0x64aa907f, 0x4f0f1126,
    0x648b7ce0, 0x4f369320, 0x646c59bf, 0x4f5e08e3, 0x644d2722, 0x4f857269,
    0x642de50d, 0x4faccfab, 0x640e9386, 0x4fd420a4, 0x63ef3290, 0x4ffb654d,
    0x63cfc231, 0x50229da1, 0x63b0426d, 0x5049c999, 0x6390b34a, 0x5070e92f,
    0x637114cc, 0x5097fc5e, 0x635166f9, 0x50bf031f, 0x6331a9d4, 0x50e5fd6d,
    0x6311dd64, 0x510ceb40, 0x62f201ac, 0x5133cc94, 0x62d216b3, 0x515aa162,
    0x62b21c7b, 0x518169a5, 0x6292130c, 0x51a82555, 0x6271fa69, 0x51ced46e,
    0x6251d298, 0x51f576ea, 0x62319b9d, 0x521c0cc2, 0x6211557e, 0x524295f0,
    0x61f1003f, 0x5269126e, 0x61d09be5, 0x528f8238, 0x61b02876, 0x52b5e546,
    0x618fa5f7, 0x52dc3b92, 0x616f146c, 0x53028518, 0x614e73da, 0x5328c1d0,
    0x612dc447, 0x534ef1b5, 0x610d05b7, 0x537514c2, 0x60ec3830, 0x539b2af0,
    0x60cb5bb7, 0x53c13439, 0x60aa7050, 0x53e73097, 0x60897601, 0x540d2005,
    0x60686ccf, 0x5433027d, 0x604754bf, 0x5458d7f9, 0x60262dd6, 0x547ea073,
    0x6004f819, 0x54a45be6, 0x5fe3b38d, 0x54ca0a4b, 0x5fc26038, 0x54efab9c,
    0x5fa0fe1f, 0x55153fd4, 0x5f7f8d46, 0x553ac6ee, 0x5f5e0db3, 0x556040e2,
    0x5f3c7f6b, 0x5585adad, 0x5f1ae274, 0x55ab0d46, 0x5ef936d1, 0x55d05faa,
    0x5ed77c8a, 0x55f5a4d2, 0x5eb5b3a2, 0x561adcb9, 0x5e93dc1f, 0x56400758,
    0x5e71f606, 0x566524aa, 0x5e50015d, 0x568a34a9, 0x5e2dfe29, 0x56af3750,
    0x5e0bec6e, 0x56d42c99, 0x5de9cc33, 0x56f9147e, 0x5dc79d7c, 0x571deefa,
    0x5da5604f, 0x5742bc06, 0x5d8314b1, 0x57677b9d, 0x5d60baa7, 0x578c2dba,
    0x5d3e5237, 0x57b0d256, 0x5d1bdb65, 0x57d5696d, 0x5cf95638, 0x57f9f2f8,
    0x5cd6c2b5, 0x581e6ef1, 0x5cb420e0, 0x5842dd54, 0x5c9170bf, 0x58673e1b,
    0x5c6eb258, 0x588b9140, 0x5c4be5b0, 0x58afd6bd, 0x5c290acc, 0x58d40e8c,
    0x5c0621b2, 0x58f838a9, 0x5be32a67, 0x591c550e, 0x5bc024f0, 0x594063b5,
    0x5b9d1154, 0x59646498, 0x5b79ef96, 0x598857b2, 0x5b56bfbd, 0x59ac3cfd,
    0x5b3381ce, 0x59d01475, 0x5b1035cf, 0x59f3de12, 0x5aecdbc5, 0x5a1799d1,
    0x5ac973b5, 0x5a3b47ab, 0x5aa5fda5, 0x5a5ee79a, 0x5a82799a, 0x5a82799a,
    0x5a5ee79a, 0x5aa5fda5, 0x5a3b47ab, 0x5ac973b5, 0x5a1799d1, 0x5aecdbc5,
    0x59f3de12, 0x5b1035cf, 0x59d01475, 0x5b3381ce, 0x59ac3cfd, 0x5b56bfbd,
    0x598857b2, 0x5b79ef96, 0x59646498, 0x5b9d1154, 0x594063b5, 0x5bc024f0,
    0x591c550e, 0x5be32a67, 0x58f838a9, 0x5c0621b2, 0x58d40e8c, 0x5c290acc,
    0x58afd6bd, 0x5c4be5b0, 0x588b9140, 0x5c6eb258, 0x58673e1b, 0x5c9170bf,
    0x5842dd54, 0x5cb420e0, 0x581e6ef1, 0x5cd6c2b5, 0x57f9f2f8, 0x5cf95638,
    0x57d5696d, 0x5d1bdb65, 0x57b0d256, 0x5d3e5237, 0x578c2dba, 0x5d60baa7,
    0x57677b9d, 0x5d8314b1, 0x5742bc06, 0x5da5604f, 0x571deefa, 0x5dc79d7c,
    0x56f9147e, 0x5de9cc33, 0x56d42c99, 0x5e0bec6e, 0x56af3750, 0x5e2dfe29,
    0x568a34a9, 0x5e50015d, 0x566524aa, 0x5e71f606, 0x56400758, 0x5e93dc1f,
    0x561adcb9, 0x5eb5b3a2, 0x55f5a4d2, 0x5ed77c8a, 0x55d05faa, 0x5ef936d1,
    0x55ab0d46, 0x5f1ae274, 0x5585adad, 0x5f3c7f6b, 0x556040e2, 0x5f5e0db3,
    0x553ac6ee, 0x5f7f8d46, 0x55153fd4, 0x5fa0fe1f, 0x54efab9c, 0x5fc26038,
    0x54ca0a4b, 0x5fe3b38d, 0x54a45be6, 0x6004f819, 0x547ea073, 0x60262dd6,
    0x5458d7f9, 0x604754bf, 0x5433027d, 0x60686ccf, 0x540d2005, 0x60897601,
    0x53e73097, 0x60aa7050, 0x53c13439, 0x60cb5bb7, 0x539b2af0, 0x60ec3830,
    0x537514c2, 0x610d05b7, 0x534ef1b5, 0x612dc447, 0x5328c1d0, 0x614e73da,
    0x53028518, 0x616f146c, 0x52dc3b92, 0x618fa5f7, 0x52b5e546, 0x61b02876,
    0x528f8238, 0x61d09be5, 0x5269126e, 0x61f1003f, 0x524295f0, 0x6211557e,
    0x521c0cc2, 0x62319b9d, 0x51f576ea, 0x6251d298, 0x51ced46e, 0x6271fa69,
    0x51a82555, 0x6292130c, 0x518169a5, 0x62b21c7b, 0x515aa162, 0x62d216b3,
    0x5133cc94, 0x62f201ac, 0x510ceb40, 0x6311dd64, 0x50e5fd6d, 0x6331a9d4,
    0x50bf031f, 0x635166f9, 0x5097fc5e, 0x637114cc, 0x5070e92f, 0x6390b34a,
    0x5049c999, 0x63b0426d, 0x50229da1, 0x63cfc231, 0x4ffb654d, 0x63ef3290,
    0x4fd420a4, 0x640e9386, 0x4faccfab, 0x642de50d, 0x4f857269, 0x644d2722,
    0x4f5e08e3, 0x646c59bf, 0x4f369320, 0x648b7ce0, 0x4f0f1126, 0x64aa907f,
    0x4ee782fb, 0x64c99498, 0x4ebfe8a5, 0x64e88926, 0x4e984229, 0x65076e25,
    0x4e708f8f, 0x6526438f, 0x4e48d0dd, 0x6545095f, 0x4e210617, 0x6563bf92,
    0x4df92f46, 0x65826622, 0x4dd14c6e, 0x65a0fd0b, 0x4da95d96, 0x65bf8447,
    0x4d8162c4, 0x65ddfbd3, 0x4d595bfe, 0x65fc63a9, 0x4d31494b, 0x661abbc5,
    0x4d092ab0, 0x66390422, 0x4ce10034, 0x66573cbb, 0x4cb8c9dd, 0x6675658c,
    0x4c9087b1, 0x66937e91, 0x4c6839b7, 0x66b187c3, 0x4c3fdff4, 0x66cf8120,
    0x4c177a6e, 0x66ed6aa1, 0x4bef092d, 0x670b4444, 0x4bc68c36, 0x67290e02,
    0x4b9e0390, 0x6746c7d8, 0x4b756f40, 0x676471c0, 0x4b4ccf4d, 0x67820bb7,
    0x4b2423be, 0x679f95b7, 0x4afb6c98, 0x67bd0fbd, 0x4ad2a9e2, 0x67da79c3,
    0x4aa9dba2, 0x67f7d3c5, 0x4a8101de, 0x68151dbe, 0x4a581c9e, 0x683257ab,
    0x4a2f2be6, 0x684f8186, 0x4a062fbd, 0x686c9b4b, 0x49dd282a, 0x6889a4f6,
    0x49b41533, 0x68a69e81, 0x498af6df, 0x68c387e9, 0x4961cd33, 0x68e06129,
    0x49389836, 0x68fd2a3d, 0x490f57ee, 0x6919e320, 0x48e60c62, 0x69368bce,
    0x48bcb599, 0x69532442, 0x48935397, 0x696fac78, 0x4869e665, 0x698c246c,
    0x48406e08, 0x69a88c19, 0x4816ea86, 0x69c4e37a, 0x47ed5be6, 0x69e12a8c,
    0x47c3c22f, 0x69fd614a, 0x479a1d67, 0x6a1987b0, 0x47706d93, 0x6a359db9,
    0x4746b2bc, 0x6a51a361, 0x471cece7, 0x6a6d98a4, 0x46f31c1a, 0x6a897d7d,
    0x46c9405c, 0x6aa551e9, 0x469f59b4, 0x6ac115e2, 0x46756828, 0x6adcc964,
    0x464b6bbe, 0x6af86c6c, 0x4621647d, 0x6b13fef5, 0x45f7526b, 0x6b2f80fb,
    0x45cd358f, 0x6b4af279, 0x45a30df0, 0x6b66536b, 0x4578db93, 0x6b81a3cd,
    0x454e9e80, 0x6b9ce39b, 0x452456bd, 0x6bb812d1, 0x44fa0450, 0x6bd3316a,
    0x44cfa740, 0x6bee3f62, 0x44a53f93, 0x6c093cb6, 0x447acd50, 0x6c242960,
    0x4450507e, 0x6c3f055d, 0x4425c923, 0x6c59d0a9, 0x43fb3746, 0x6c748b3f,
    0x43d09aed, 0x6c8f351c, 0x43a5f41e, 0x6ca9ce3b, 0x437b42e1, 0x6cc45698,
    0x4350873c, 0x6cdece2f, 0x4325c135, 0x6cf934fc, 0x42faf0d4, 0x6d138afb,
    0x42d0161e, 0x6d2dd027, 0x42a5311b, 0x6d48047e, 0x427a41d0, 0x6d6227fa,
    0x424f4845, 0x6d7c3a98, 0x42244481, 0x6d963c54, 0x41f93689, 0x6db02d29,
    0x41ce1e65, 0x6dca0d14, 0x41a2fc1a, 0x6de3dc11, 0x4177cfb1, 0x6dfd9a1c,
    0x414c992f, 0x6e174730, 0x4121589b, 0x6e30e34a, 0x40f60dfb, 0x6e4a6e66,
    0x40cab958, 0x6e63e87f, 0x409f5ab6, 0x6e7d5193, 0x4073f21d, 0x6e96a99d,
    0x40487f94, 0x6eaff099, 0x401d0321, 0x6ec92683, 0x3ff17cca, 0x6ee24b57,
    0x3fc5ec98, 0x6efb5f12, 0x3f9a5290, 0x6f1461b0, 0x3f6eaeb8, 0x6f2d532c,
    0x3f430119, 0x6f463383, 0x3f1749b8, 0x6f5f02b2, 0x3eeb889c, 0x6f77c0b3,
    0x3ebfbdcd, 0x6f906d84, 0x3e93e950, 0x6fa90921, 0x3e680b2c, 0x6fc19385,
    0x3e3c2369, 0x6fda0cae, 0x3e10320d, 0x6ff27497, 0x3de4371f, 0x700acb3c,
    0x3db832a6, 0x7023109a, 0x3d8c24a8, 0x703b44ad, 0x3d600d2c, 0x70536771,
    0x3d33ec39, 0x706b78e3, 0x3d07c1d6, 0x708378ff, 0x3cdb8e09, 0x709b67c0,
    0x3caf50da, 0x70b34525, 0x3c830a50, 0x70cb1128, 0x3c56ba70, 0x70e2cbc6,
    0x3c2a6142, 0x70fa74fc, 0x3bfdfecd, 0x71120cc5, 0x3bd19318, 0x7129931f,
    0x3ba51e29, 0x71410805, 0x3b78a007, 0x71586b74, 0x3b4c18ba, 0x716fbd68,
    0x3b1f8848, 0x7186fdde, 0x3af2eeb7, 0x719e2cd2, 0x3ac64c0f, 0x71b54a41,
    0x3a99a057, 0x71cc5626, 0x3a6ceb96, 0x71e35080, 0x3a402dd2, 0x71fa3949,
    0x3a136712, 0x7211107e, 0x39e6975e, 0x7227d61c, 0x39b9bebc, 0x723e8a20,
    0x398cdd32, 0x72552c85, 0x395ff2c9, 0x726bbd48, 0x3932ff87, 0x72823c67,
    0x39060373, 0x7298a9dd, 0x38d8fe93, 0x72af05a7, 0x38abf0ef, 0x72c54fc1,
    0x387eda8e, 0x72db8828, 0x3851bb77, 0x72f1aed9, 0x382493b0, 0x7307c3d0,
    0x37f76341, 0x731dc70a, 0x37ca2a30, 0x7333b883, 0x379ce885, 0x73499838,
    0x376f9e46, 0x735f6626, 0x37424b7b, 0x73752249, 0x3714f02a, 0x738acc9e,
    0x36e78c5b, 0x73a06522, 0x36ba2014, 0x73b5ebd1, 0x368cab5c, 0x73cb60a8,
    0x365f2e3b, 0x73e0c3a3, 0x3631a8b8, 0x73f614c0, 0x36041ad9, 0x740b53fb,
    0x35d684a6, 0x74208150, 0x35a8e625, 0x74359cbd, 0x357b3f5d, 0x744aa63f,
    0x354d9057, 0x745f9dd1, 0x351fd918, 0x74748371, 0x34f219a8, 0x7489571c,
    0x34c4520d, 0x749e18cd, 0x34968250, 0x74b2c884, 0x3468aa76, 0x74c7663a,
    0x343aca87, 0x74dbf1ef, 0x340ce28b, 0x74f06b9e, 0x33def287, 0x7504d345,
    0x33b0fa84, 0x751928e0, 0x3382fa88, 0x752d6c6c, 0x3354f29b, 0x75419de7,
    0x3326e2c3, 0x7555bd4c, 0x32f8cb07, 0x7569ca99, 0x32caab6f, 0x757dc5ca,
    0x329c8402, 0x7591aedd, 0x326e54c7, 0x75a585cf, 0x32401dc6, 0x75b94a9c,
    0x3211df04, 0x75ccfd42, 0x31e39889, 0x75e09dbd, 0x31b54a5e, 0x75f42c0b,
    0x3186f487, 0x7607a828, 0x3158970e, 0x761b1211, 0x312a31f8, 0x762e69c4,
    0x30fbc54d, 0x7641af3d, 0x30cd5115, 0x7654e279, 0x309ed556, 0x76680376,
    0x30705217, 0x767b1231, 0x3041c761, 0x768e0ea6, 0x30133539, 0x76a0f8d2,
    0x2fe49ba7, 0x76b3d0b4, 0x2fb5fab2, 0x76c69647, 0x2f875262, 0x76d94989,
    0x2f58a2be, 0x76ebea77, 0x2f29ebcc, 0x76fe790e, 0x2efb2d95, 0x7710f54c,
    0x2ecc681e, 0x77235f2d, 0x2e9d9b70, 0x7735b6af, 0x2e6ec792, 0x7747fbce,
    0x2e3fec8b, 0x775a2e89, 0x2e110a62, 0x776c4edb, 0x2de2211e, 0x777e5cc3,
    0x2db330c7, 0x7790583e, 0x2d843964, 0x77a24148, 0x2d553afc, 0x77b417df,
    0x2d263596, 0x77c5dc01, 0x2cf72939, 0x77d78daa, 0x2cc815ee, 0x77e92cd9,
    0x2c98fbba, 0x77fab989, 0x2c69daa6, 0x780c33b8, 0x2c3ab2b9, 0x781d9b65,
    0x2c0b83fa, 0x782ef08b, 0x2bdc4e6f, 0x78403329, 0x2bad1221, 0x7851633b,
    0x2b7dcf17, 0x786280bf, 0x2b4e8558, 0x78738bb3, 0x2b1f34eb, 0x78848414,
    0x2aefddd8, 0x789569df, 0x2ac08026, 0x78a63d11, 0x2a911bdc, 0x78b6fda8,
    0x2a61b101, 0x78c7aba2, 0x2a323f9e, 0x78d846fb, 0x2a02c7b8, 0x78e8cfb2,
    0x29d34958, 0x78f945c3, 0x29a3c485, 0x7909a92d, 0x29743946, 0x7919f9ec,
    0x2944a7a2, 0x792a37fe, 0x29150fa1, 0x793a6361, 0x28e5714b, 0x794a7c12,
    0x28b5cca5, 0x795a820e, 0x288621b9, 0x796a7554, 0x2856708d, 0x797a55e0,
    0x2826b928, 0x798a23b1, 0x27f6fb92, 0x7999dec4, 0x27c737d3, 0x79a98715,
    0x27976df1, 0x79b91ca4, 0x27679df4, 0x79c89f6e, 0x2737c7e3, 0x79d80f6f,
    0x2707ebc7, 0x79e76ca7, 0x26d809a5, 0x79f6b711, 0x26a82186, 0x7a05eead,
    0x26783370, 0x7a151378, 0x26483f6c, 0x7a24256f, 0x26184581, 0x7a332490,
    0x25e845b6, 0x7a4210d8, 0x25b84012, 0x7a50ea47, 0x2588349d, 0x7a5fb0d8,
    0x2558235f, 0x7a6e648a, 0x25280c5e, 0x7a7d055b, 0x24f7efa2, 0x7a8b9348,
    0x24c7cd33, 0x7a9a0e50, 0x2497a517, 0x7aa8766f, 0x24677758, 0x7ab6cba4,
    0x243743fa, 0x7ac50dec, 0x24070b08, 0x7ad33d45, 0x23d6cc87, 0x7ae159ae,
    0x23a6887f, 0x7aef6323, 0x23763ef7, 0x7afd59a4, 0x2345eff8, 0x7b0b3d2c,
    0x23159b88, 0x7b190dbc, 0x22e541af, 0x7b26cb4f, 0x22b4e274, 0x7b3475e5,
    0x22847de0, 0x7b420d7a, 0x225413f8, 0x7b4f920e, 0x2223a4c5, 0x7b5d039e,
    0x21f3304f, 0x7b6a6227, 0x21c2b69c, 0x7b77ada8, 0x219237b5, 0x7b84e61f,
    0x2161b3a0, 0x7b920b89, 0x21312a65, 0x7b9f1de6, 0x21009c0c, 0x7bac1d31,
    0x20d0089c, 0x7bb9096b, 0x209f701c, 0x7bc5e290, 0x206ed295, 0x7bd2a89e,
    0x203e300d, 0x7bdf5b94, 0x200d888d, 0x7bebfb70, 0x1fdcdc1b, 0x7bf88830,
    0x1fac2abf, 0x7c0501d2, 0x1f7b7481, 0x7c116853, 0x1f4ab968, 0x7c1dbbb3,
    0x1f19f97b, 0x7c29fbee, 0x1ee934c3, 0x7c362904, 0x1eb86b46, 0x7c4242f2,
    0x1e879d0d, 0x7c4e49b7, 0x1e56ca1e, 0x7c5a3d50, 0x1e25f282, 0x7c661dbc,
    0x1df5163f, 0x7c71eaf9, 0x1dc4355e, 0x7c7da505, 0x1d934fe5, 0x7c894bde,
    0x1d6265dd, 0x7c94df83, 0x1d31774d, 0x7ca05ff1, 0x1d00843d, 0x7cabcd28,
    0x1ccf8cb3, 0x7cb72724, 0x1c9e90b8, 0x7cc26de5, 0x1c6d9053, 0x7ccda169,
    0x1c3c8b8c, 0x7cd8c1ae, 0x1c0b826a, 0x7ce3ceb2, 0x1bda74f6, 0x7ceec873,
    0x1ba96335, 0x7cf9aef0, 0x1b784d30, 0x7d048228, 0x1b4732ef, 0x7d0f4218,
    0x1b161479, 0x7d19eebf, 0x1ae4f1d6, 0x7d24881b, 0x1ab3cb0d, 0x7d2f0e2b,
    0x1a82a026, 0x7d3980ec, 0x1a517128, 0x7d43e05e, 0x1a203e1b, 0x7d4e2c7f,
    0x19ef0707, 0x7d58654d, 0x19bdcbf3, 0x7d628ac6, 0x198c8ce7, 0x7d6c9ce9,
    0x195b49ea, 0x7d769bb5, 0x192a0304, 0x7d808728, 0x18f8b83c, 0x7d8a5f40,
    0x18c7699b, 0x7d9423fc, 0x18961728, 0x7d9dd55a, 0x1864c0ea, 0x7da77359,
    0x183366e9, 0x7db0fdf8, 0x1802092c, 0x7dba7534, 0x17d0a7bc, 0x7dc3d90d,
    0x179f429f, 0x7dcd2981, 0x176dd9de, 0x7dd6668f, 0x173c6d80, 0x7ddf9034,
    0x170afd8d, 0x7de8a670, 0x16d98a0c, 0x7df1a942, 0x16a81305, 0x7dfa98a8,
    0x1676987f, 0x7e0374a0, 0x16451a83, 0x7e0c3d29, 0x16139918, 0x7e14f242,
    0x15e21445, 0x7e1d93ea, 0x15b08c12, 0x7e26221f, 0x157f0086, 0x7e2e9cdf,
    0x154d71aa, 0x7e37042a, 0x151bdf86, 0x7e3f57ff, 0x14ea4a1f, 0x7e47985b,
    0x14b8b17f, 0x7e4fc53e, 0x148715ae, 0x7e57dea7, 0x145576b1, 0x7e5fe493,
    0x1423d492, 0x7e67d703, 0x13f22f58, 0x7e6fb5f4, 0x13c0870a, 0x7e778166,
    0x138edbb1, 0x7e7f3957, 0x135d2d53, 0x7e86ddc6, 0x132b7bf9, 0x7e8e6eb2,
    0x12f9c7aa, 0x7e95ec1a, 0x12c8106f, 0x7e9d55fc, 0x1296564d, 0x7ea4ac58,
    0x1264994e, 0x7eabef2c, 0x1232d979, 0x7eb31e78, 0x120116d5, 0x7eba3a39,
    0x11cf516a, 0x7ec14270, 0x119d8941, 0x7ec8371a, 0x116bbe60, 0x7ecf1837,
    0x1139f0cf, 0x7ed5e5c6, 0x11082096, 0x7edc9fc6, 0x10d64dbd, 0x7ee34636,
    0x10a4784b, 0x7ee9d914, 0x1072a048, 0x7ef05860, 0x1040c5bb, 0x7ef6c418,
    0x100ee8ad, 0x7efd1c3c, 0xfdd0926,  0x7f0360cb, 0xfab272b,  0x7f0991c4,
    0xf7942c7,  0x7f0faf25, 0xf475bff,  0x7f15b8ee, 0xf1572dc,  0x7f1baf1e,
    0xee38766,  0x7f2191b4, 0xeb199a4,  0x7f2760af, 0xe7fa99e,  0x7f2d1c0e,
    0xe4db75b,  0x7f32c3d1, 0xe1bc2e4,  0x7f3857f6, 0xde9cc40,  0x7f3dd87c,
    0xdb7d376,  0x7f434563, 0xd85d88f,  0x7f489eaa, 0xd53db92,  0x7f4de451,
    0xd21dc87,  0x7f531655, 0xcefdb76,  0x7f5834b7, 0xcbdd865,  0x7f5d3f75,
    0xc8bd35e,  0x7f62368f, 0xc59cc68,  0x7f671a05, 0xc27c389,  0x7f6be9d4,
    0xbf5b8cb,  0x7f70a5fe, 0xbc3ac35,  0x7f754e80, 0xb919dcf,  0x7f79e35a,
    0xb5f8d9f,  0x7f7e648c, 0xb2d7baf,  0x7f82d214, 0xafb6805,  0x7f872bf3,
    0xac952aa,  0x7f8b7227, 0xa973ba5,  0x7f8fa4b0, 0xa6522fe,  0x7f93c38c,
    0xa3308bd,  0x7f97cebd, 0xa00ece8,  0x7f9bc640, 0x9cecf89,  0x7f9faa15,
    0x99cb0a7,  0x7fa37a3c, 0x96a9049,  0x7fa736b4, 0x9386e78,  0x7faadf7c,
    0x9064b3a,  0x7fae7495, 0x8d42699,  0x7fb1f5fc, 0x8a2009a,  0x7fb563b3,
    0x86fd947,  0x7fb8bdb8, 0x83db0a7,  0x7fbc040a, 0x80b86c2,  0x7fbf36aa,
    0x7d95b9e,  0x7fc25596, 0x7a72f45,  0x7fc560cf, 0x77501be,  0x7fc85854,
    0x742d311,  0x7fcb3c23, 0x710a345,  0x7fce0c3e, 0x6de7262,  0x7fd0c8a3,
    0x6ac406f,  0x7fd37153, 0x67a0d76,  0x7fd6064c, 0x647d97c,  0x7fd8878e,
    0x615a48b,  0x7fdaf519, 0x5e36ea9,  0x7fdd4eec, 0x5b137df,  0x7fdf9508,
    0x57f0035,  0x7fe1c76b, 0x54cc7b1,  0x7fe3e616, 0x51a8e5c,  0x7fe5f108,
    0x4e8543e,  0x7fe7e841, 0x4b6195d,  0x7fe9cbc0, 0x483ddc3,  0x7feb9b85,
    0x451a177,  0x7fed5791, 0x41f6480,  0x7feeffe1, 0x3ed26e6,  0x7ff09478,
    0x3bae8b2,  0x7ff21553, 0x388a9ea,  0x7ff38274, 0x3566a96,  0x7ff4dbd9,
    0x3242abf,  0x7ff62182, 0x2f1ea6c,  0x7ff75370, 0x2bfa9a4,  0x7ff871a2,
    0x28d6870,  0x7ff97c18, 0x25b26d7,  0x7ffa72d1, 0x228e4e2,  0x7ffb55ce,
    0x1f6a297,  0x7ffc250f, 0x1c45ffe,  0x7ffce093, 0x1921d20,  0x7ffd885a,
    0x15fda03,  0x7ffe1c65, 0x12d96b1,  0x7ffe9cb2, 0xfb5330,   0x7fff0943,
    0xc90f88,   0x7fff6216, 0x96cbc1,   0x7fffa72c, 0x6487e3,   0x7fffd886,
    0x3243f5,   0x7ffff621, 0x0,        0x7fffffff, 0xffcdbc0b, 0x7ffff621,
    0xff9b781d, 0x7fffd886, 0xff69343f, 0x7fffa72c, 0xff36f078, 0x7fff6216,
    0xff04acd0, 0x7fff0943, 0xfed2694f, 0x7ffe9cb2, 0xfea025fd, 0x7ffe1c65,
    0xfe6de2e0, 0x7ffd885a, 0xfe3ba002, 0x7ffce093, 0xfe095d69, 0x7ffc250f,
    0xfdd71b1e, 0x7ffb55ce, 0xfda4d929, 0x7ffa72d1, 0xfd729790, 0x7ff97c18,
    0xfd40565c, 0x7ff871a2, 0xfd0e1594, 0x7ff75370, 0xfcdbd541, 0x7ff62182,
    0xfca9956a, 0x7ff4dbd9, 0xfc775616, 0x7ff38274, 0xfc45174e, 0x7ff21553,
    0xfc12d91a, 0x7ff09478, 0xfbe09b80, 0x7feeffe1, 0xfbae5e89, 0x7fed5791,
    0xfb7c223d, 0x7feb9b85, 0xfb49e6a3, 0x7fe9cbc0, 0xfb17abc2, 0x7fe7e841,
    0xfae571a4, 0x7fe5f108, 0xfab3384f, 0x7fe3e616, 0xfa80ffcb, 0x7fe1c76b,
    0xfa4ec821, 0x7fdf9508, 0xfa1c9157, 0x7fdd4eec, 0xf9ea5b75, 0x7fdaf519,
    0xf9b82684, 0x7fd8878e, 0xf985f28a, 0x7fd6064c, 0xf953bf91, 0x7fd37153,
    0xf9218d9e, 0x7fd0c8a3, 0xf8ef5cbb, 0x7fce0c3e, 0xf8bd2cef, 0x7fcb3c23,
    0xf88afe42, 0x7fc85854, 0xf858d0bb, 0x7fc560cf, 0xf826a462, 0x7fc25596,
    0xf7f4793e, 0x7fbf36aa, 0xf7c24f59, 0x7fbc040a, 0xf79026b9, 0x7fb8bdb8,
    0xf75dff66, 0x7fb563b3, 0xf72bd967, 0x7fb1f5fc, 0xf6f9b4c6, 0x7fae7495,
    0xf6c79188, 0x7faadf7c, 0xf6956fb7, 0x7fa736b4, 0xf6634f59, 0x7fa37a3c,
    0xf6313077, 0x7f9faa15, 0xf5ff1318, 0x7f9bc640, 0xf5ccf743, 0x7f97cebd,
    0xf59add02, 0x7f93c38c, 0xf568c45b, 0x7f8fa4b0, 0xf536ad56, 0x7f8b7227,
    0xf50497fb, 0x7f872bf3, 0xf4d28451, 0x7f82d214, 0xf4a07261, 0x7f7e648c,
    0xf46e6231, 0x7f79e35a, 0xf43c53cb, 0x7f754e80, 0xf40a4735, 0x7f70a5fe,
    0xf3d83c77, 0x7f6be9d4, 0xf3a63398, 0x7f671a05, 0xf3742ca2, 0x7f62368f,
    0xf342279b, 0x7f5d3f75, 0xf310248a, 0x7f5834b7, 0xf2de2379, 0x7f531655,
    0xf2ac246e, 0x7f4de451, 0xf27a2771, 0x7f489eaa, 0xf2482c8a, 0x7f434563,
    0xf21633c0, 0x7f3dd87c, 0xf1e43d1c, 0x7f3857f6, 0xf1b248a5, 0x7f32c3d1,
    0xf1805662, 0x7f2d1c0e, 0xf14e665c, 0x7f2760af, 0xf11c789a, 0x7f2191b4,
    0xf0ea8d24, 0x7f1baf1e, 0xf0b8a401, 0x7f15b8ee, 0xf086bd39, 0x7f0faf25,
    0xf054d8d5, 0x7f0991c4, 0xf022f6da, 0x7f0360cb, 0xeff11753, 0x7efd1c3c,
    0xefbf3a45, 0x7ef6c418, 0xef8d5fb8, 0x7ef05860, 0xef5b87b5, 0x7ee9d914,
    0xef29b243, 0x7ee34636, 0xeef7df6a, 0x7edc9fc6, 0xeec60f31, 0x7ed5e5c6,
    0xee9441a0, 0x7ecf1837, 0xee6276bf, 0x7ec8371a, 0xee30ae96, 0x7ec14270,
    0xedfee92b, 0x7eba3a39, 0xedcd2687, 0x7eb31e78, 0xed9b66b2, 0x7eabef2c,
    0xed69a9b3, 0x7ea4ac58, 0xed37ef91, 0x7e9d55fc, 0xed063856, 0x7e95ec1a,
    0xecd48407, 0x7e8e6eb2, 0xeca2d2ad, 0x7e86ddc6, 0xec71244f, 0x7e7f3957,
    0xec3f78f6, 0x7e778166, 0xec0dd0a8, 0x7e6fb5f4, 0xebdc2b6e, 0x7e67d703,
    0xebaa894f, 0x7e5fe493, 0xeb78ea52, 0x7e57dea7, 0xeb474e81, 0x7e4fc53e,
    0xeb15b5e1, 0x7e47985b, 0xeae4207a, 0x7e3f57ff, 0xeab28e56, 0x7e37042a,
    0xea80ff7a, 0x7e2e9cdf, 0xea4f73ee, 0x7e26221f, 0xea1debbb, 0x7e1d93ea,
    0xe9ec66e8, 0x7e14f242, 0xe9bae57d, 0x7e0c3d29, 0xe9896781, 0x7e0374a0,
    0xe957ecfb, 0x7dfa98a8, 0xe92675f4, 0x7df1a942, 0xe8f50273, 0x7de8a670,
    0xe8c39280, 0x7ddf9034, 0xe8922622, 0x7dd6668f, 0xe860bd61, 0x7dcd2981,
    0xe82f5844, 0x7dc3d90d, 0xe7fdf6d4, 0x7dba7534, 0xe7cc9917, 0x7db0fdf8,
    0xe79b3f16, 0x7da77359, 0xe769e8d8, 0x7d9dd55a, 0xe7389665, 0x7d9423fc,
    0xe70747c4, 0x7d8a5f40, 0xe6d5fcfc, 0x7d808728, 0xe6a4b616, 0x7d769bb5,
    0xe6737319, 0x7d6c9ce9, 0xe642340d, 0x7d628ac6, 0xe610f8f9, 0x7d58654d,
    0xe5dfc1e5, 0x7d4e2c7f, 0xe5ae8ed8, 0x7d43e05e, 0xe57d5fda, 0x7d3980ec,
    0xe54c34f3, 0x7d2f0e2b, 0xe51b0e2a, 0x7d24881b, 0xe4e9eb87, 0x7d19eebf,
    0xe4b8cd11, 0x7d0f4218, 0xe487b2d0, 0x7d048228, 0xe4569ccb, 0x7cf9aef0,
    0xe4258b0a, 0x7ceec873, 0xe3f47d96, 0x7ce3ceb2, 0xe3c37474, 0x7cd8c1ae,
    0xe3926fad, 0x7ccda169, 0xe3616f48, 0x7cc26de5, 0xe330734d, 0x7cb72724,
    0xe2ff7bc3, 0x7cabcd28, 0xe2ce88b3, 0x7ca05ff1, 0xe29d9a23, 0x7c94df83,
    0xe26cb01b, 0x7c894bde, 0xe23bcaa2, 0x7c7da505, 0xe20ae9c1, 0x7c71eaf9,
    0xe1da0d7e, 0x7c661dbc, 0xe1a935e2, 0x7c5a3d50, 0xe17862f3, 0x7c4e49b7,
    0xe14794ba, 0x7c4242f2, 0xe116cb3d, 0x7c362904, 0xe0e60685, 0x7c29fbee,
    0xe0b54698, 0x7c1dbbb3, 0xe0848b7f, 0x7c116853, 0xe053d541, 0x7c0501d2,
    0xe02323e5, 0x7bf88830, 0xdff27773, 0x7bebfb70, 0xdfc1cff3, 0x7bdf5b94,
    0xdf912d6b, 0x7bd2a89e, 0xdf608fe4, 0x7bc5e290, 0xdf2ff764, 0x7bb9096b,
    0xdeff63f4, 0x7bac1d31, 0xdeced59b, 0x7b9f1de6, 0xde9e4c60, 0x7b920b89,
    0xde6dc84b, 0x7b84e61f, 0xde3d4964, 0x7b77ada8, 0xde0ccfb1, 0x7b6a6227,
    0xdddc5b3b, 0x7b5d039e, 0xddabec08, 0x7b4f920e, 0xdd7b8220, 0x7b420d7a,
    0xdd4b1d8c, 0x7b3475e5, 0xdd1abe51, 0x7b26cb4f, 0xdcea6478, 0x7b190dbc,
    0xdcba1008, 0x7b0b3d2c, 0xdc89c109, 0x7afd59a4, 0xdc597781, 0x7aef6323,
    0xdc293379, 0x7ae159ae, 0xdbf8f4f8, 0x7ad33d45, 0xdbc8bc06, 0x7ac50dec,
    0xdb9888a8, 0x7ab6cba4, 0xdb685ae9, 0x7aa8766f, 0xdb3832cd, 0x7a9a0e50,
    0xdb08105e, 0x7a8b9348, 0xdad7f3a2, 0x7a7d055b, 0xdaa7dca1, 0x7a6e648a,
    0xda77cb63, 0x7a5fb0d8, 0xda47bfee, 0x7a50ea47, 0xda17ba4a, 0x7a4210d8,
    0xd9e7ba7f, 0x7a332490, 0xd9b7c094, 0x7a24256f, 0xd987cc90, 0x7a151378,
    0xd957de7a, 0x7a05eead, 0xd927f65b, 0x79f6b711, 0xd8f81439, 0x79e76ca7,
    0xd8c8381d, 0x79d80f6f, 0xd898620c, 0x79c89f6e, 0xd868920f, 0x79b91ca4,
    0xd838c82d, 0x79a98715, 0xd809046e, 0x7999dec4, 0xd7d946d8, 0x798a23b1,
    0xd7a98f73, 0x797a55e0, 0xd779de47, 0x796a7554, 0xd74a335b, 0x795a820e,
    0xd71a8eb5, 0x794a7c12, 0xd6eaf05f, 0x793a6361, 0xd6bb585e, 0x792a37fe,
    0xd68bc6ba, 0x7919f9ec, 0xd65c3b7b, 0x7909a92d, 0xd62cb6a8, 0x78f945c3,
    0xd5fd3848, 0x78e8cfb2, 0xd5cdc062, 0x78d846fb, 0xd59e4eff, 0x78c7aba2,
    0xd56ee424, 0x78b6fda8, 0xd53f7fda, 0x78a63d11, 0xd5102228, 0x789569df,
    0xd4e0cb15, 0x78848414, 0xd4b17aa8, 0x78738bb3, 0xd48230e9, 0x786280bf,
    0xd452eddf, 0x7851633b, 0xd423b191, 0x78403329, 0xd3f47c06, 0x782ef08b,
    0xd3c54d47, 0x781d9b65, 0xd396255a, 0x780c33b8, 0xd3670446, 0x77fab989,
    0xd337ea12, 0x77e92cd9, 0xd308d6c7, 0x77d78daa, 0xd2d9ca6a, 0x77c5dc01,
    0xd2aac504, 0x77b417df, 0xd27bc69c, 0x77a24148, 0xd24ccf39, 0x7790583e,
    0xd21ddee2, 0x777e5cc3, 0xd1eef59e, 0x776c4edb, 0xd1c01375, 0x775a2e89,
    0xd191386e, 0x7747fbce, 0xd1626490, 0x7735b6af, 0xd13397e2, 0x77235f2d,
    0xd104d26b, 0x7710f54c, 0xd0d61434, 0x76fe790e, 0xd0a75d42, 0x76ebea77,
    0xd078ad9e, 0x76d94989, 0xd04a054e, 0x76c69647, 0xd01b6459, 0x76b3d0b4,
    0xcfeccac7, 0x76a0f8d2, 0xcfbe389f, 0x768e0ea6, 0xcf8fade9, 0x767b1231,
    0xcf612aaa, 0x76680376, 0xcf32aeeb, 0x7654e279, 0xcf043ab3, 0x7641af3d,
    0xced5ce08, 0x762e69c4, 0xcea768f2, 0x761b1211, 0xce790b79, 0x7607a828,
    0xce4ab5a2, 0x75f42c0b, 0xce1c6777, 0x75e09dbd, 0xcdee20fc, 0x75ccfd42,
    0xcdbfe23a, 0x75b94a9c, 0xcd91ab39, 0x75a585cf, 0xcd637bfe, 0x7591aedd,
    0xcd355491, 0x757dc5ca, 0xcd0734f9, 0x7569ca99, 0xccd91d3d, 0x7555bd4c,
    0xccab0d65, 0x75419de7, 0xcc7d0578, 0x752d6c6c, 0xcc4f057c, 0x751928e0,
    0xcc210d79, 0x7504d345, 0xcbf31d75, 0x74f06b9e, 0xcbc53579, 0x74dbf1ef,
    0xcb97558a, 0x74c7663a, 0xcb697db0, 0x74b2c884, 0xcb3badf3, 0x749e18cd,
    0xcb0de658, 0x7489571c, 0xcae026e8, 0x74748371, 0xcab26fa9, 0x745f9dd1,
    0xca84c0a3, 0x744aa63f, 0xca5719db, 0x74359cbd, 0xca297b5a, 0x74208150,
    0xc9fbe527, 0x740b53fb, 0xc9ce5748, 0x73f614c0, 0xc9a0d1c5, 0x73e0c3a3,
    0xc97354a4, 0x73cb60a8, 0xc945dfec, 0x73b5ebd1, 0xc91873a5, 0x73a06522,
    0xc8eb0fd6, 0x738acc9e, 0xc8bdb485, 0x73752249, 0xc89061ba, 0x735f6626,
    0xc863177b, 0x73499838, 0xc835d5d0, 0x7333b883, 0xc8089cbf, 0x731dc70a,
    0xc7db6c50, 0x7307c3d0, 0xc7ae4489, 0x72f1aed9, 0xc7812572, 0x72db8828,
    0xc7540f11, 0x72c54fc1, 0xc727016d, 0x72af05a7, 0xc6f9fc8d, 0x7298a9dd,
    0xc6cd0079, 0x72823c67, 0xc6a00d37, 0x726bbd48, 0xc67322ce, 0x72552c85,
    0xc6464144, 0x723e8a20, 0xc61968a2, 0x7227d61c, 0xc5ec98ee, 0x7211107e,
    0xc5bfd22e, 0x71fa3949, 0xc593146a, 0x71e35080, 0xc5665fa9, 0x71cc5626,
    0xc539b3f1, 0x71b54a41, 0xc50d1149, 0x719e2cd2, 0xc4e077b8, 0x7186fdde,
    0xc4b3e746, 0x716fbd68, 0xc4875ff9, 0x71586b74, 0xc45ae1d7, 0x71410805,
    0xc42e6ce8, 0x7129931f, 0xc4020133, 0x71120cc5, 0xc3d59ebe, 0x70fa74fc,
    0xc3a94590, 0x70e2cbc6, 0xc37cf5b0, 0x70cb1128, 0xc350af26, 0x70b34525,
    0xc32471f7, 0x709b67c0, 0xc2f83e2a, 0x708378ff, 0xc2cc13c7, 0x706b78e3,
    0xc29ff2d4, 0x70536771, 0xc273db58, 0x703b44ad, 0xc247cd5a, 0x7023109a,
    0xc21bc8e1, 0x700acb3c, 0xc1efcdf3, 0x6ff27497, 0xc1c3dc97, 0x6fda0cae,
    0xc197f4d4, 0x6fc19385, 0xc16c16b0, 0x6fa90921, 0xc1404233, 0x6f906d84,
    0xc1147764, 0x6f77c0b3, 0xc0e8b648, 0x6f5f02b2, 0xc0bcfee7, 0x6f463383,
    0xc0915148, 0x6f2d532c, 0xc065ad70, 0x6f1461b0, 0xc03a1368, 0x6efb5f12,
    0xc00e8336, 0x6ee24b57, 0xbfe2fcdf, 0x6ec92683, 0xbfb7806c, 0x6eaff099,
    0xbf8c0de3, 0x6e96a99d, 0xbf60a54a, 0x6e7d5193, 0xbf3546a8, 0x6e63e87f,
    0xbf09f205, 0x6e4a6e66, 0xbedea765, 0x6e30e34a, 0xbeb366d1, 0x6e174730,
    0xbe88304f, 0x6dfd9a1c, 0xbe5d03e6, 0x6de3dc11, 0xbe31e19b, 0x6dca0d14,
    0xbe06c977, 0x6db02d29, 0xbddbbb7f, 0x6d963c54, 0xbdb0b7bb, 0x6d7c3a98,
    0xbd85be30, 0x6d6227fa, 0xbd5acee5, 0x6d48047e, 0xbd2fe9e2, 0x6d2dd027,
    0xbd050f2c, 0x6d138afb, 0xbcda3ecb, 0x6cf934fc, 0xbcaf78c4, 0x6cdece2f,
    0xbc84bd1f, 0x6cc45698, 0xbc5a0be2, 0x6ca9ce3b, 0xbc2f6513, 0x6c8f351c,
    0xbc04c8ba, 0x6c748b3f, 0xbbda36dd, 0x6c59d0a9, 0xbbafaf82, 0x6c3f055d,
    0xbb8532b0, 0x6c242960, 0xbb5ac06d, 0x6c093cb6, 0xbb3058c0, 0x6bee3f62,
    0xbb05fbb0, 0x6bd3316a, 0xbadba943, 0x6bb812d1, 0xbab16180, 0x6b9ce39b,
    0xba87246d, 0x6b81a3cd, 0xba5cf210, 0x6b66536b, 0xba32ca71, 0x6b4af279,
    0xba08ad95, 0x6b2f80fb, 0xb9de9b83, 0x6b13fef5, 0xb9b49442, 0x6af86c6c,
    0xb98a97d8, 0x6adcc964, 0xb960a64c, 0x6ac115e2, 0xb936bfa4, 0x6aa551e9,
    0xb90ce3e6, 0x6a897d7d, 0xb8e31319, 0x6a6d98a4, 0xb8b94d44, 0x6a51a361,
    0xb88f926d, 0x6a359db9, 0xb865e299, 0x6a1987b0, 0xb83c3dd1, 0x69fd614a,
    0xb812a41a, 0x69e12a8c, 0xb7e9157a, 0x69c4e37a, 0xb7bf91f8, 0x69a88c19,
    0xb796199b, 0x698c246c, 0xb76cac69, 0x696fac78, 0xb7434a67, 0x69532442,
    0xb719f39e, 0x69368bce, 0xb6f0a812, 0x6919e320, 0xb6c767ca, 0x68fd2a3d,
    0xb69e32cd, 0x68e06129, 0xb6750921, 0x68c387e9, 0xb64beacd, 0x68a69e81,
    0xb622d7d6, 0x6889a4f6, 0xb5f9d043, 0x686c9b4b, 0xb5d0d41a, 0x684f8186,
    0xb5a7e362, 0x683257ab, 0xb57efe22, 0x68151dbe, 0xb556245e, 0x67f7d3c5,
    0xb52d561e, 0x67da79c3, 0xb5049368, 0x67bd0fbd, 0xb4dbdc42, 0x679f95b7,
    0xb4b330b3, 0x67820bb7, 0xb48a90c0, 0x676471c0, 0xb461fc70, 0x6746c7d8,
    0xb43973ca, 0x67290e02, 0xb410f6d3, 0x670b4444, 0xb3e88592, 0x66ed6aa1,
    0xb3c0200c, 0x66cf8120, 0xb397c649, 0x66b187c3, 0xb36f784f, 0x66937e91,
    0xb3473623, 0x6675658c, 0xb31effcc, 0x66573cbb, 0xb2f6d550, 0x66390422,
    0xb2ceb6b5, 0x661abbc5, 0xb2a6a402, 0x65fc63a9, 0xb27e9d3c, 0x65ddfbd3,
    0xb256a26a, 0x65bf8447, 0xb22eb392, 0x65a0fd0b, 0xb206d0ba, 0x65826622,
    0xb1def9e9, 0x6563bf92, 0xb1b72f23, 0x6545095f, 0xb18f7071, 0x6526438f,
    0xb167bdd7, 0x65076e25, 0xb140175b, 0x64e88926, 0xb1187d05, 0x64c99498,
    0xb0f0eeda, 0x64aa907f, 0xb0c96ce0, 0x648b7ce0, 0xb0a1f71d, 0x646c59bf,
    0xb07a8d97, 0x644d2722, 0xb0533055, 0x642de50d, 0xb02bdf5c, 0x640e9386,
    0xb0049ab3, 0x63ef3290, 0xafdd625f, 0x63cfc231, 0xafb63667, 0x63b0426d,
    0xaf8f16d1, 0x6390b34a, 0xaf6803a2, 0x637114cc, 0xaf40fce1, 0x635166f9,
    0xaf1a0293, 0x6331a9d4, 0xaef314c0, 0x6311dd64, 0xaecc336c, 0x62f201ac,
    0xaea55e9e, 0x62d216b3, 0xae7e965b, 0x62b21c7b, 0xae57daab, 0x6292130c,
    0xae312b92, 0x6271fa69, 0xae0a8916, 0x6251d298, 0xade3f33e, 0x62319b9d,
    0xadbd6a10, 0x6211557e, 0xad96ed92, 0x61f1003f, 0xad707dc8, 0x61d09be5,
    0xad4a1aba, 0x61b02876, 0xad23c46e, 0x618fa5f7, 0xacfd7ae8, 0x616f146c,
    0xacd73e30, 0x614e73da, 0xacb10e4b, 0x612dc447, 0xac8aeb3e, 0x610d05b7,
    0xac64d510, 0x60ec3830, 0xac3ecbc7, 0x60cb5bb7, 0xac18cf69, 0x60aa7050,
    0xabf2dffb, 0x60897601, 0xabccfd83, 0x60686ccf, 0xaba72807, 0x604754bf,
    0xab815f8d, 0x60262dd6, 0xab5ba41a, 0x6004f819, 0xab35f5b5, 0x5fe3b38d,
    0xab105464, 0x5fc26038, 0xaaeac02c, 0x5fa0fe1f, 0xaac53912, 0x5f7f8d46,
    0xaa9fbf1e, 0x5f5e0db3, 0xaa7a5253, 0x5f3c7f6b, 0xaa54f2ba, 0x5f1ae274,
    0xaa2fa056, 0x5ef936d1, 0xaa0a5b2e, 0x5ed77c8a, 0xa9e52347, 0x5eb5b3a2,
    0xa9bff8a8, 0x5e93dc1f, 0xa99adb56, 0x5e71f606, 0xa975cb57, 0x5e50015d,
    0xa950c8b0, 0x5e2dfe29, 0xa92bd367, 0x5e0bec6e, 0xa906eb82, 0x5de9cc33,
    0xa8e21106, 0x5dc79d7c, 0xa8bd43fa, 0x5da5604f, 0xa8988463, 0x5d8314b1,
    0xa873d246, 0x5d60baa7, 0xa84f2daa, 0x5d3e5237, 0xa82a9693, 0x5d1bdb65,
    0xa8060d08, 0x5cf95638, 0xa7e1910f, 0x5cd6c2b5, 0xa7bd22ac, 0x5cb420e0,
    0xa798c1e5, 0x5c9170bf, 0xa7746ec0, 0x5c6eb258, 0xa7502943, 0x5c4be5b0,
    0xa72bf174, 0x5c290acc, 0xa707c757, 0x5c0621b2, 0xa6e3aaf2, 0x5be32a67,
    0xa6bf9c4b, 0x5bc024f0, 0xa69b9b68, 0x5b9d1154, 0xa677a84e, 0x5b79ef96,
    0xa653c303, 0x5b56bfbd, 0xa62feb8b, 0x5b3381ce, 0xa60c21ee, 0x5b1035cf,
    0xa5e8662f, 0x5aecdbc5, 0xa5c4b855, 0x5ac973b5, 0xa5a11866, 0x5aa5fda5,
    0xa57d8666, 0x5a82799a, 0xa55a025b, 0x5a5ee79a, 0xa5368c4b, 0x5a3b47ab,
    0xa513243b, 0x5a1799d1, 0xa4efca31, 0x59f3de12, 0xa4cc7e32, 0x59d01475,
    0xa4a94043, 0x59ac3cfd, 0xa486106a, 0x598857b2, 0xa462eeac, 0x59646498,
    0xa43fdb10, 0x594063b5, 0xa41cd599, 0x591c550e, 0xa3f9de4e, 0x58f838a9,
    0xa3d6f534, 0x58d40e8c, 0xa3b41a50, 0x58afd6bd, 0xa3914da8, 0x588b9140,
    0xa36e8f41, 0x58673e1b, 0xa34bdf20, 0x5842dd54, 0xa3293d4b, 0x581e6ef1,
    0xa306a9c8, 0x57f9f2f8, 0xa2e4249b, 0x57d5696d, 0xa2c1adc9, 0x57b0d256,
    0xa29f4559, 0x578c2dba, 0xa27ceb4f, 0x57677b9d, 0xa25a9fb1, 0x5742bc06,
    0xa2386284, 0x571deefa, 0xa21633cd, 0x56f9147e, 0xa1f41392, 0x56d42c99,
    0xa1d201d7, 0x56af3750, 0xa1affea3, 0x568a34a9, 0xa18e09fa, 0x566524aa,
    0xa16c23e1, 0x56400758, 0xa14a4c5e, 0x561adcb9, 0xa1288376, 0x55f5a4d2,
    0xa106c92f, 0x55d05faa, 0xa0e51d8c, 0x55ab0d46, 0xa0c38095, 0x5585adad,
    0xa0a1f24d, 0x556040e2, 0xa08072ba, 0x553ac6ee, 0xa05f01e1, 0x55153fd4,
    0xa03d9fc8, 0x54efab9c, 0xa01c4c73, 0x54ca0a4b, 0x9ffb07e7, 0x54a45be6,
    0x9fd9d22a, 0x547ea073, 0x9fb8ab41, 0x5458d7f9, 0x9f979331, 0x5433027d,
    0x9f7689ff, 0x540d2005, 0x9f558fb0, 0x53e73097, 0x9f34a449, 0x53c13439,
    0x9f13c7d0, 0x539b2af0, 0x9ef2fa49, 0x537514c2, 0x9ed23bb9, 0x534ef1b5,
    0x9eb18c26, 0x5328c1d0, 0x9e90eb94, 0x53028518, 0x9e705a09, 0x52dc3b92,
    0x9e4fd78a, 0x52b5e546, 0x9e2f641b, 0x528f8238, 0x9e0effc1, 0x5269126e,
    0x9deeaa82, 0x524295f0, 0x9dce6463, 0x521c0cc2, 0x9dae2d68, 0x51f576ea,
    0x9d8e0597, 0x51ced46e, 0x9d6decf4, 0x51a82555, 0x9d4de385, 0x518169a5,
    0x9d2de94d, 0x515aa162, 0x9d0dfe54, 0x5133cc94, 0x9cee229c, 0x510ceb40,
    0x9cce562c, 0x50e5fd6d, 0x9cae9907, 0x50bf031f, 0x9c8eeb34, 0x5097fc5e,
    0x9c6f4cb6, 0x5070e92f, 0x9c4fbd93, 0x5049c999, 0x9c303dcf, 0x50229da1,
    0x9c10cd70, 0x4ffb654d, 0x9bf16c7a, 0x4fd420a4, 0x9bd21af3, 0x4faccfab,
    0x9bb2d8de, 0x4f857269, 0x9b93a641, 0x4f5e08e3, 0x9b748320, 0x4f369320,
    0x9b556f81, 0x4f0f1126, 0x9b366b68, 0x4ee782fb, 0x9b1776da, 0x4ebfe8a5,
    0x9af891db, 0x4e984229, 0x9ad9bc71, 0x4e708f8f, 0x9abaf6a1, 0x4e48d0dd,
    0x9a9c406e, 0x4e210617, 0x9a7d99de, 0x4df92f46, 0x9a5f02f5, 0x4dd14c6e,
    0x9a407bb9, 0x4da95d96, 0x9a22042d, 0x4d8162c4, 0x9a039c57, 0x4d595bfe,
    0x99e5443b, 0x4d31494b, 0x99c6fbde, 0x4d092ab0, 0x99a8c345, 0x4ce10034,
    0x998a9a74, 0x4cb8c9dd, 0x996c816f, 0x4c9087b1, 0x994e783d, 0x4c6839b7,
    0x99307ee0, 0x4c3fdff4, 0x9912955f, 0x4c177a6e, 0x98f4bbbc, 0x4bef092d,
    0x98d6f1fe, 0x4bc68c36, 0x98b93828, 0x4b9e0390, 0x989b8e40, 0x4b756f40,
    0x987df449, 0x4b4ccf4d, 0x98606a49, 0x4b2423be, 0x9842f043, 0x4afb6c98,
    0x9825863d, 0x4ad2a9e2, 0x98082c3b, 0x4aa9dba2, 0x97eae242, 0x4a8101de,
    0x97cda855, 0x4a581c9e, 0x97b07e7a, 0x4a2f2be6, 0x979364b5, 0x4a062fbd,
    0x97765b0a, 0x49dd282a, 0x9759617f, 0x49b41533, 0x973c7817, 0x498af6df,
    0x971f9ed7, 0x4961cd33, 0x9702d5c3, 0x49389836, 0x96e61ce0, 0x490f57ee,
    0x96c97432, 0x48e60c62, 0x96acdbbe, 0x48bcb599, 0x96905388, 0x48935397,
    0x9673db94, 0x4869e665, 0x965773e7, 0x48406e08, 0x963b1c86, 0x4816ea86,
    0x961ed574, 0x47ed5be6, 0x96029eb6, 0x47c3c22f, 0x95e67850, 0x479a1d67,
    0x95ca6247, 0x47706d93, 0x95ae5c9f, 0x4746b2bc, 0x9592675c, 0x471cece7,
    0x95768283, 0x46f31c1a, 0x955aae17, 0x46c9405c, 0x953eea1e, 0x469f59b4,
    0x9523369c, 0x46756828, 0x95079394, 0x464b6bbe, 0x94ec010b, 0x4621647d,
    0x94d07f05, 0x45f7526b, 0x94b50d87, 0x45cd358f, 0x9499ac95, 0x45a30df0,
    0x947e5c33, 0x4578db93, 0x94631c65, 0x454e9e80, 0x9447ed2f, 0x452456bd,
    0x942cce96, 0x44fa0450, 0x9411c09e, 0x44cfa740, 0x93f6c34a, 0x44a53f93,
    0x93dbd6a0, 0x447acd50, 0x93c0faa3, 0x4450507e, 0x93a62f57, 0x4425c923,
    0x938b74c1, 0x43fb3746, 0x9370cae4, 0x43d09aed, 0x935631c5, 0x43a5f41e,
    0x933ba968, 0x437b42e1, 0x932131d1, 0x4350873c, 0x9306cb04, 0x4325c135,
    0x92ec7505, 0x42faf0d4, 0x92d22fd9, 0x42d0161e, 0x92b7fb82, 0x42a5311b,
    0x929dd806, 0x427a41d0, 0x9283c568, 0x424f4845, 0x9269c3ac, 0x42244481,
    0x924fd2d7, 0x41f93689, 0x9235f2ec, 0x41ce1e65, 0x921c23ef, 0x41a2fc1a,
    0x920265e4, 0x4177cfb1, 0x91e8b8d0, 0x414c992f, 0x91cf1cb6, 0x4121589b,
    0x91b5919a, 0x40f60dfb, 0x919c1781, 0x40cab958, 0x9182ae6d, 0x409f5ab6,
    0x91695663, 0x4073f21d, 0x91500f67, 0x40487f94, 0x9136d97d, 0x401d0321,
    0x911db4a9, 0x3ff17cca, 0x9104a0ee, 0x3fc5ec98, 0x90eb9e50, 0x3f9a5290,
    0x90d2acd4, 0x3f6eaeb8, 0x90b9cc7d, 0x3f430119, 0x90a0fd4e, 0x3f1749b8,
    0x90883f4d, 0x3eeb889c, 0x906f927c, 0x3ebfbdcd, 0x9056f6df, 0x3e93e950,
    0x903e6c7b, 0x3e680b2c, 0x9025f352, 0x3e3c2369, 0x900d8b69, 0x3e10320d,
    0x8ff534c4, 0x3de4371f, 0x8fdcef66, 0x3db832a6, 0x8fc4bb53, 0x3d8c24a8,
    0x8fac988f, 0x3d600d2c, 0x8f94871d, 0x3d33ec39, 0x8f7c8701, 0x3d07c1d6,
    0x8f649840, 0x3cdb8e09, 0x8f4cbadb, 0x3caf50da, 0x8f34eed8, 0x3c830a50,
    0x8f1d343a, 0x3c56ba70, 0x8f058b04, 0x3c2a6142, 0x8eedf33b, 0x3bfdfecd,
    0x8ed66ce1, 0x3bd19318, 0x8ebef7fb, 0x3ba51e29, 0x8ea7948c, 0x3b78a007,
    0x8e904298, 0x3b4c18ba, 0x8e790222, 0x3b1f8848, 0x8e61d32e, 0x3af2eeb7,
    0x8e4ab5bf, 0x3ac64c0f, 0x8e33a9da, 0x3a99a057, 0x8e1caf80, 0x3a6ceb96,
    0x8e05c6b7, 0x3a402dd2, 0x8deeef82, 0x3a136712, 0x8dd829e4, 0x39e6975e,
    0x8dc175e0, 0x39b9bebc, 0x8daad37b, 0x398cdd32, 0x8d9442b8, 0x395ff2c9,
    0x8d7dc399, 0x3932ff87, 0x8d675623, 0x39060373, 0x8d50fa59, 0x38d8fe93,
    0x8d3ab03f, 0x38abf0ef, 0x8d2477d8, 0x387eda8e, 0x8d0e5127, 0x3851bb77,
    0x8cf83c30, 0x382493b0, 0x8ce238f6, 0x37f76341, 0x8ccc477d, 0x37ca2a30,
    0x8cb667c8, 0x379ce885, 0x8ca099da, 0x376f9e46, 0x8c8addb7, 0x37424b7b,
    0x8c753362, 0x3714f02a, 0x8c5f9ade, 0x36e78c5b, 0x8c4a142f, 0x36ba2014,
    0x8c349f58, 0x368cab5c, 0x8c1f3c5d, 0x365f2e3b, 0x8c09eb40, 0x3631a8b8,
    0x8bf4ac05, 0x36041ad9, 0x8bdf7eb0, 0x35d684a6, 0x8bca6343, 0x35a8e625,
    0x8bb559c1, 0x357b3f5d, 0x8ba0622f, 0x354d9057, 0x8b8b7c8f, 0x351fd918,
    0x8b76a8e4, 0x34f219a8, 0x8b61e733, 0x34c4520d, 0x8b4d377c, 0x34968250,
    0x8b3899c6, 0x3468aa76, 0x8b240e11, 0x343aca87, 0x8b0f9462, 0x340ce28b,
    0x8afb2cbb, 0x33def287, 0x8ae6d720, 0x33b0fa84, 0x8ad29394, 0x3382fa88,
    0x8abe6219, 0x3354f29b, 0x8aaa42b4, 0x3326e2c3, 0x8a963567, 0x32f8cb07,
    0x8a823a36, 0x32caab6f, 0x8a6e5123, 0x329c8402, 0x8a5a7a31, 0x326e54c7,
    0x8a46b564, 0x32401dc6, 0x8a3302be, 0x3211df04, 0x8a1f6243, 0x31e39889,
    0x8a0bd3f5, 0x31b54a5e, 0x89f857d8, 0x3186f487, 0x89e4edef, 0x3158970e,
    0x89d1963c, 0x312a31f8, 0x89be50c3, 0x30fbc54d, 0x89ab1d87, 0x30cd5115,
    0x8997fc8a, 0x309ed556, 0x8984edcf, 0x30705217, 0x8971f15a, 0x3041c761,
    0x895f072e, 0x30133539, 0x894c2f4c, 0x2fe49ba7, 0x893969b9, 0x2fb5fab2,
    0x8926b677, 0x2f875262, 0x89141589, 0x2f58a2be, 0x890186f2, 0x2f29ebcc,
    0x88ef0ab4, 0x2efb2d95, 0x88dca0d3, 0x2ecc681e, 0x88ca4951, 0x2e9d9b70,
    0x88b80432, 0x2e6ec792, 0x88a5d177, 0x2e3fec8b, 0x8893b125, 0x2e110a62,
    0x8881a33d, 0x2de2211e, 0x886fa7c2, 0x2db330c7, 0x885dbeb8, 0x2d843964,
    0x884be821, 0x2d553afc, 0x883a23ff, 0x2d263596, 0x88287256, 0x2cf72939,
    0x8816d327, 0x2cc815ee, 0x88054677, 0x2c98fbba, 0x87f3cc48, 0x2c69daa6,
    0x87e2649b, 0x2c3ab2b9, 0x87d10f75, 0x2c0b83fa, 0x87bfccd7, 0x2bdc4e6f,
    0x87ae9cc5, 0x2bad1221, 0x879d7f41, 0x2b7dcf17, 0x878c744d, 0x2b4e8558,
    0x877b7bec, 0x2b1f34eb, 0x876a9621, 0x2aefddd8, 0x8759c2ef, 0x2ac08026,
    0x87490258, 0x2a911bdc, 0x8738545e, 0x2a61b101, 0x8727b905, 0x2a323f9e,
    0x8717304e, 0x2a02c7b8, 0x8706ba3d, 0x29d34958, 0x86f656d3, 0x29a3c485,
    0x86e60614, 0x29743946, 0x86d5c802, 0x2944a7a2, 0x86c59c9f, 0x29150fa1,
    0x86b583ee, 0x28e5714b, 0x86a57df2, 0x28b5cca5, 0x86958aac, 0x288621b9,
    0x8685aa20, 0x2856708d, 0x8675dc4f, 0x2826b928, 0x8666213c, 0x27f6fb92,
    0x865678eb, 0x27c737d3, 0x8646e35c, 0x27976df1, 0x86376092, 0x27679df4,
    0x8627f091, 0x2737c7e3, 0x86189359, 0x2707ebc7, 0x860948ef, 0x26d809a5,
    0x85fa1153, 0x26a82186, 0x85eaec88, 0x26783370, 0x85dbda91, 0x26483f6c,
    0x85ccdb70, 0x26184581, 0x85bdef28, 0x25e845b6, 0x85af15b9, 0x25b84012,
    0x85a04f28, 0x2588349d, 0x85919b76, 0x2558235f, 0x8582faa5, 0x25280c5e,
    0x85746cb8, 0x24f7efa2, 0x8565f1b0, 0x24c7cd33, 0x85578991, 0x2497a517,
    0x8549345c, 0x24677758, 0x853af214, 0x243743fa, 0x852cc2bb, 0x24070b08,
    0x851ea652, 0x23d6cc87, 0x85109cdd, 0x23a6887f, 0x8502a65c, 0x23763ef7,
    0x84f4c2d4, 0x2345eff8, 0x84e6f244, 0x23159b88, 0x84d934b1, 0x22e541af,
    0x84cb8a1b, 0x22b4e274, 0x84bdf286, 0x22847de0, 0x84b06df2, 0x225413f8,
    0x84a2fc62, 0x2223a4c5, 0x84959dd9, 0x21f3304f, 0x84885258, 0x21c2b69c,
    0x847b19e1, 0x219237b5, 0x846df477, 0x2161b3a0, 0x8460e21a, 0x21312a65,
    0x8453e2cf, 0x21009c0c, 0x8446f695, 0x20d0089c, 0x843a1d70, 0x209f701c,
    0x842d5762, 0x206ed295, 0x8420a46c, 0x203e300d, 0x84140490, 0x200d888d,
    0x840777d0, 0x1fdcdc1b, 0x83fafe2e, 0x1fac2abf, 0x83ee97ad, 0x1f7b7481,
    0x83e2444d, 0x1f4ab968, 0x83d60412, 0x1f19f97b, 0x83c9d6fc, 0x1ee934c3,
    0x83bdbd0e, 0x1eb86b46, 0x83b1b649, 0x1e879d0d, 0x83a5c2b0, 0x1e56ca1e,
    0x8399e244, 0x1e25f282, 0x838e1507, 0x1df5163f, 0x83825afb, 0x1dc4355e,
    0x8376b422, 0x1d934fe5, 0x836b207d, 0x1d6265dd, 0x835fa00f, 0x1d31774d,
    0x835432d8, 0x1d00843d, 0x8348d8dc, 0x1ccf8cb3, 0x833d921b, 0x1c9e90b8,
    0x83325e97, 0x1c6d9053, 0x83273e52, 0x1c3c8b8c, 0x831c314e, 0x1c0b826a,
    0x8311378d, 0x1bda74f6, 0x83065110, 0x1ba96335, 0x82fb7dd8, 0x1b784d30,
    0x82f0bde8, 0x1b4732ef, 0x82e61141, 0x1b161479, 0x82db77e5, 0x1ae4f1d6,
    0x82d0f1d5, 0x1ab3cb0d, 0x82c67f14, 0x1a82a026, 0x82bc1fa2, 0x1a517128,
    0x82b1d381, 0x1a203e1b, 0x82a79ab3, 0x19ef0707, 0x829d753a, 0x19bdcbf3,
    0x82936317, 0x198c8ce7, 0x8289644b, 0x195b49ea, 0x827f78d8, 0x192a0304,
    0x8275a0c0, 0x18f8b83c, 0x826bdc04, 0x18c7699b, 0x82622aa6, 0x18961728,
    0x82588ca7, 0x1864c0ea, 0x824f0208, 0x183366e9, 0x82458acc, 0x1802092c,
    0x823c26f3, 0x17d0a7bc, 0x8232d67f, 0x179f429f, 0x82299971, 0x176dd9de,
    0x82206fcc, 0x173c6d80, 0x82175990, 0x170afd8d, 0x820e56be, 0x16d98a0c,
    0x82056758, 0x16a81305, 0x81fc8b60, 0x1676987f, 0x81f3c2d7, 0x16451a83,
    0x81eb0dbe, 0x16139918, 0x81e26c16, 0x15e21445, 0x81d9dde1, 0x15b08c12,
    0x81d16321, 0x157f0086, 0x81c8fbd6, 0x154d71aa, 0x81c0a801, 0x151bdf86,
    0x81b867a5, 0x14ea4a1f, 0x81b03ac2, 0x14b8b17f, 0x81a82159, 0x148715ae,
    0x81a01b6d, 0x145576b1, 0x819828fd, 0x1423d492, 0x81904a0c, 0x13f22f58,
    0x81887e9a, 0x13c0870a, 0x8180c6a9, 0x138edbb1, 0x8179223a, 0x135d2d53,
    0x8171914e, 0x132b7bf9, 0x816a13e6, 0x12f9c7aa, 0x8162aa04, 0x12c8106f,
    0x815b53a8, 0x1296564d, 0x815410d4, 0x1264994e, 0x814ce188, 0x1232d979,
    0x8145c5c7, 0x120116d5, 0x813ebd90, 0x11cf516a, 0x8137c8e6, 0x119d8941,
    0x8130e7c9, 0x116bbe60, 0x812a1a3a, 0x1139f0cf, 0x8123603a, 0x11082096,
    0x811cb9ca, 0x10d64dbd, 0x811626ec, 0x10a4784b, 0x810fa7a0, 0x1072a048,
    0x81093be8, 0x1040c5bb, 0x8102e3c4, 0x100ee8ad, 0x80fc9f35, 0xfdd0926,
    0x80f66e3c, 0xfab272b,  0x80f050db, 0xf7942c7,  0x80ea4712, 0xf475bff,
    0x80e450e2, 0xf1572dc,  0x80de6e4c, 0xee38766,  0x80d89f51, 0xeb199a4,
    0x80d2e3f2, 0xe7fa99e,  0x80cd3c2f, 0xe4db75b,  0x80c7a80a, 0xe1bc2e4,
    0x80c22784, 0xde9cc40,  0x80bcba9d, 0xdb7d376,  0x80b76156, 0xd85d88f,
    0x80b21baf, 0xd53db92,  0x80ace9ab, 0xd21dc87,  0x80a7cb49, 0xcefdb76,
    0x80a2c08b, 0xcbdd865,  0x809dc971, 0xc8bd35e,  0x8098e5fb, 0xc59cc68,
    0x8094162c, 0xc27c389,  0x808f5a02, 0xbf5b8cb,  0x808ab180, 0xbc3ac35,
    0x80861ca6, 0xb919dcf,  0x80819b74, 0xb5f8d9f,  0x807d2dec, 0xb2d7baf,
    0x8078d40d, 0xafb6805,  0x80748dd9, 0xac952aa,  0x80705b50, 0xa973ba5,
    0x806c3c74, 0xa6522fe,  0x80683143, 0xa3308bd,  0x806439c0, 0xa00ece8,
    0x806055eb, 0x9cecf89,  0x805c85c4, 0x99cb0a7,  0x8058c94c, 0x96a9049,
    0x80552084, 0x9386e78,  0x80518b6b, 0x9064b3a,  0x804e0a04, 0x8d42699,
    0x804a9c4d, 0x8a2009a,  0x80474248, 0x86fd947,  0x8043fbf6, 0x83db0a7,
    0x8040c956, 0x80b86c2,  0x803daa6a, 0x7d95b9e,  0x803a9f31, 0x7a72f45,
    0x8037a7ac, 0x77501be,  0x8034c3dd, 0x742d311,  0x8031f3c2, 0x710a345,
    0x802f375d, 0x6de7262,  0x802c8ead, 0x6ac406f,  0x8029f9b4, 0x67a0d76,
    0x80277872, 0x647d97c,  0x80250ae7, 0x615a48b,  0x8022b114, 0x5e36ea9,
    0x80206af8, 0x5b137df,  0x801e3895, 0x57f0035,  0x801c19ea, 0x54cc7b1,
    0x801a0ef8, 0x51a8e5c,  0x801817bf, 0x4e8543e,  0x80163440, 0x4b6195d,
    0x8014647b, 0x483ddc3,  0x8012a86f, 0x451a177,  0x8011001f, 0x41f6480,
    0x800f6b88, 0x3ed26e6,  0x800deaad, 0x3bae8b2,  0x800c7d8c, 0x388a9ea,
    0x800b2427, 0x3566a96,  0x8009de7e, 0x3242abf,  0x8008ac90, 0x2f1ea6c,
    0x80078e5e, 0x2bfa9a4,  0x800683e8, 0x28d6870,  0x80058d2f, 0x25b26d7,
    0x8004aa32, 0x228e4e2,  0x8003daf1, 0x1f6a297,  0x80031f6d, 0x1c45ffe,
    0x800277a6, 0x1921d20,  0x8001e39b, 0x15fda03,  0x8001634e, 0x12d96b1,
    0x8000f6bd, 0xfb5330,   0x80009dea, 0xc90f88,   0x800058d4, 0x96cbc1,
    0x8000277a, 0x6487e3,   0x800009df, 0x3243f5,   0x80000000, 0x0,
    0x800009df, 0xffcdbc0b, 0x8000277a, 0xff9b781d, 0x800058d4, 0xff69343f,
    0x80009dea, 0xff36f078, 0x8000f6bd, 0xff04acd0, 0x8001634e, 0xfed2694f,
    0x8001e39b, 0xfea025fd, 0x800277a6, 0xfe6de2e0, 0x80031f6d, 0xfe3ba002,
    0x8003daf1, 0xfe095d69, 0x8004aa32, 0xfdd71b1e, 0x80058d2f, 0xfda4d929,
    0x800683e8, 0xfd729790, 0x80078e5e, 0xfd40565c, 0x8008ac90, 0xfd0e1594,
    0x8009de7e, 0xfcdbd541, 0x800b2427, 0xfca9956a, 0x800c7d8c, 0xfc775616,
    0x800deaad, 0xfc45174e, 0x800f6b88, 0xfc12d91a, 0x8011001f, 0xfbe09b80,
    0x8012a86f, 0xfbae5e89, 0x8014647b, 0xfb7c223d, 0x80163440, 0xfb49e6a3,
    0x801817bf, 0xfb17abc2, 0x801a0ef8, 0xfae571a4, 0x801c19ea, 0xfab3384f,
    0x801e3895, 0xfa80ffcb, 0x80206af8, 0xfa4ec821, 0x8022b114, 0xfa1c9157,
    0x80250ae7, 0xf9ea5b75, 0x80277872, 0xf9b82684, 0x8029f9b4, 0xf985f28a,
    0x802c8ead, 0xf953bf91, 0x802f375d, 0xf9218d9e, 0x8031f3c2, 0xf8ef5cbb,
    0x8034c3dd, 0xf8bd2cef, 0x8037a7ac, 0xf88afe42, 0x803a9f31, 0xf858d0bb,
    0x803daa6a, 0xf826a462, 0x8040c956, 0xf7f4793e, 0x8043fbf6, 0xf7c24f59,
    0x80474248, 0xf79026b9, 0x804a9c4d, 0xf75dff66, 0x804e0a04, 0xf72bd967,
    0x80518b6b, 0xf6f9b4c6, 0x80552084, 0xf6c79188, 0x8058c94c, 0xf6956fb7,
    0x805c85c4, 0xf6634f59, 0x806055eb, 0xf6313077, 0x806439c0, 0xf5ff1318,
    0x80683143, 0xf5ccf743, 0x806c3c74, 0xf59add02, 0x80705b50, 0xf568c45b,
    0x80748dd9, 0xf536ad56, 0x8078d40d, 0xf50497fb, 0x807d2dec, 0xf4d28451,
    0x80819b74, 0xf4a07261, 0x80861ca6, 0xf46e6231, 0x808ab180, 0xf43c53cb,
    0x808f5a02, 0xf40a4735, 0x8094162c, 0xf3d83c77, 0x8098e5fb, 0xf3a63398,
    0x809dc971, 0xf3742ca2, 0x80a2c08b, 0xf342279b, 0x80a7cb49, 0xf310248a,
    0x80ace9ab, 0xf2de2379, 0x80b21baf, 0xf2ac246e, 0x80b76156, 0xf27a2771,
    0x80bcba9d, 0xf2482c8a, 0x80c22784, 0xf21633c0, 0x80c7a80a, 0xf1e43d1c,
    0x80cd3c2f, 0xf1b248a5, 0x80d2e3f2, 0xf1805662, 0x80d89f51, 0xf14e665c,
    0x80de6e4c, 0xf11c789a, 0x80e450e2, 0xf0ea8d24, 0x80ea4712, 0xf0b8a401,
    0x80f050db, 0xf086bd39, 0x80f66e3c, 0xf054d8d5, 0x80fc9f35, 0xf022f6da,
    0x8102e3c4, 0xeff11753, 0x81093be8, 0xefbf3a45, 0x810fa7a0, 0xef8d5fb8,
    0x811626ec, 0xef5b87b5, 0x811cb9ca, 0xef29b243, 0x8123603a, 0xeef7df6a,
    0x812a1a3a, 0xeec60f31, 0x8130e7c9, 0xee9441a0, 0x8137c8e6, 0xee6276bf,
    0x813ebd90, 0xee30ae96, 0x8145c5c7, 0xedfee92b, 0x814ce188, 0xedcd2687,
    0x815410d4, 0xed9b66b2, 0x815b53a8, 0xed69a9b3, 0x8162aa04, 0xed37ef91,
    0x816a13e6, 0xed063856, 0x8171914e, 0xecd48407, 0x8179223a, 0xeca2d2ad,
    0x8180c6a9, 0xec71244f, 0x81887e9a, 0xec3f78f6, 0x81904a0c, 0xec0dd0a8,
    0x819828fd, 0xebdc2b6e, 0x81a01b6d, 0xebaa894f, 0x81a82159, 0xeb78ea52,
    0x81b03ac2, 0xeb474e81, 0x81b867a5, 0xeb15b5e1, 0x81c0a801, 0xeae4207a,
    0x81c8fbd6, 0xeab28e56, 0x81d16321, 0xea80ff7a, 0x81d9dde1, 0xea4f73ee,
    0x81e26c16, 0xea1debbb, 0x81eb0dbe, 0xe9ec66e8, 0x81f3c2d7, 0xe9bae57d,
    0x81fc8b60, 0xe9896781, 0x82056758, 0xe957ecfb, 0x820e56be, 0xe92675f4,
    0x82175990, 0xe8f50273, 0x82206fcc, 0xe8c39280, 0x82299971, 0xe8922622,
    0x8232d67f, 0xe860bd61, 0x823c26f3, 0xe82f5844, 0x82458acc, 0xe7fdf6d4,
    0x824f0208, 0xe7cc9917, 0x82588ca7, 0xe79b3f16, 0x82622aa6, 0xe769e8d8,
    0x826bdc04, 0xe7389665, 0x8275a0c0, 0xe70747c4, 0x827f78d8, 0xe6d5fcfc,
    0x8289644b, 0xe6a4b616, 0x82936317, 0xe6737319, 0x829d753a, 0xe642340d,
    0x82a79ab3, 0xe610f8f9, 0x82b1d381, 0xe5dfc1e5, 0x82bc1fa2, 0xe5ae8ed8,
    0x82c67f14, 0xe57d5fda, 0x82d0f1d5, 0xe54c34f3, 0x82db77e5, 0xe51b0e2a,
    0x82e61141, 0xe4e9eb87, 0x82f0bde8, 0xe4b8cd11, 0x82fb7dd8, 0xe487b2d0,
    0x83065110, 0xe4569ccb, 0x8311378d, 0xe4258b0a, 0x831c314e, 0xe3f47d96,
    0x83273e52, 0xe3c37474, 0x83325e97, 0xe3926fad, 0x833d921b, 0xe3616f48,
    0x8348d8dc, 0xe330734d, 0x835432d8, 0xe2ff7bc3, 0x835fa00f, 0xe2ce88b3,
    0x836b207d, 0xe29d9a23, 0x8376b422, 0xe26cb01b, 0x83825afb, 0xe23bcaa2,
    0x838e1507, 0xe20ae9c1, 0x8399e244, 0xe1da0d7e, 0x83a5c2b0, 0xe1a935e2,
    0x83b1b649, 0xe17862f3, 0x83bdbd0e, 0xe14794ba, 0x83c9d6fc, 0xe116cb3d,
    0x83d60412, 0xe0e60685, 0x83e2444d, 0xe0b54698, 0x83ee97ad, 0xe0848b7f,
    0x83fafe2e, 0xe053d541, 0x840777d0, 0xe02323e5, 0x84140490, 0xdff27773,
    0x8420a46c, 0xdfc1cff3, 0x842d5762, 0xdf912d6b, 0x843a1d70, 0xdf608fe4,
    0x8446f695, 0xdf2ff764, 0x8453e2cf, 0xdeff63f4, 0x8460e21a, 0xdeced59b,
    0x846df477, 0xde9e4c60, 0x847b19e1, 0xde6dc84b, 0x84885258, 0xde3d4964,
    0x84959dd9, 0xde0ccfb1, 0x84a2fc62, 0xdddc5b3b, 0x84b06df2, 0xddabec08,
    0x84bdf286, 0xdd7b8220, 0x84cb8a1b, 0xdd4b1d8c, 0x84d934b1, 0xdd1abe51,
    0x84e6f244, 0xdcea6478, 0x84f4c2d4, 0xdcba1008, 0x8502a65c, 0xdc89c109,
    0x85109cdd, 0xdc597781, 0x851ea652, 0xdc293379, 0x852cc2bb, 0xdbf8f4f8,
    0x853af214, 0xdbc8bc06, 0x8549345c, 0xdb9888a8, 0x85578991, 0xdb685ae9,
    0x8565f1b0, 0xdb3832cd, 0x85746cb8, 0xdb08105e, 0x8582faa5, 0xdad7f3a2,
    0x85919b76, 0xdaa7dca1, 0x85a04f28, 0xda77cb63, 0x85af15b9, 0xda47bfee,
    0x85bdef28, 0xda17ba4a, 0x85ccdb70, 0xd9e7ba7f, 0x85dbda91, 0xd9b7c094,
    0x85eaec88, 0xd987cc90, 0x85fa1153, 0xd957de7a, 0x860948ef, 0xd927f65b,
    0x86189359, 0xd8f81439, 0x8627f091, 0xd8c8381d, 0x86376092, 0xd898620c,
    0x8646e35c, 0xd868920f, 0x865678eb, 0xd838c82d, 0x8666213c, 0xd809046e,
    0x8675dc4f, 0xd7d946d8, 0x8685aa20, 0xd7a98f73, 0x86958aac, 0xd779de47,
    0x86a57df2, 0xd74a335b, 0x86b583ee, 0xd71a8eb5, 0x86c59c9f, 0xd6eaf05f,
    0x86d5c802, 0xd6bb585e, 0x86e60614, 0xd68bc6ba, 0x86f656d3, 0xd65c3b7b,
    0x8706ba3d, 0xd62cb6a8, 0x8717304e, 0xd5fd3848, 0x8727b905, 0xd5cdc062,
    0x8738545e, 0xd59e4eff, 0x87490258, 0xd56ee424, 0x8759c2ef, 0xd53f7fda,
    0x876a9621, 0xd5102228, 0x877b7bec, 0xd4e0cb15, 0x878c744d, 0xd4b17aa8,
    0x879d7f41, 0xd48230e9, 0x87ae9cc5, 0xd452eddf, 0x87bfccd7, 0xd423b191,
    0x87d10f75, 0xd3f47c06, 0x87e2649b, 0xd3c54d47, 0x87f3cc48, 0xd396255a,
    0x88054677, 0xd3670446, 0x8816d327, 0xd337ea12, 0x88287256, 0xd308d6c7,
    0x883a23ff, 0xd2d9ca6a, 0x884be821, 0xd2aac504, 0x885dbeb8, 0xd27bc69c,
    0x886fa7c2, 0xd24ccf39, 0x8881a33d, 0xd21ddee2, 0x8893b125, 0xd1eef59e,
    0x88a5d177, 0xd1c01375, 0x88b80432, 0xd191386e, 0x88ca4951, 0xd1626490,
    0x88dca0d3, 0xd13397e2, 0x88ef0ab4, 0xd104d26b, 0x890186f2, 0xd0d61434,
    0x89141589, 0xd0a75d42, 0x8926b677, 0xd078ad9e, 0x893969b9, 0xd04a054e,
    0x894c2f4c, 0xd01b6459, 0x895f072e, 0xcfeccac7, 0x8971f15a, 0xcfbe389f,
    0x8984edcf, 0xcf8fade9, 0x8997fc8a, 0xcf612aaa, 0x89ab1d87, 0xcf32aeeb,
    0x89be50c3, 0xcf043ab3, 0x89d1963c, 0xced5ce08, 0x89e4edef, 0xcea768f2,
    0x89f857d8, 0xce790b79, 0x8a0bd3f5, 0xce4ab5a2, 0x8a1f6243, 0xce1c6777,
    0x8a3302be, 0xcdee20fc, 0x8a46b564, 0xcdbfe23a, 0x8a5a7a31, 0xcd91ab39,
    0x8a6e5123, 0xcd637bfe, 0x8a823a36, 0xcd355491, 0x8a963567, 0xcd0734f9,
    0x8aaa42b4, 0xccd91d3d, 0x8abe6219, 0xccab0d65, 0x8ad29394, 0xcc7d0578,
    0x8ae6d720, 0xcc4f057c, 0x8afb2cbb, 0xcc210d79, 0x8b0f9462, 0xcbf31d75,
    0x8b240e11, 0xcbc53579, 0x8b3899c6, 0xcb97558a, 0x8b4d377c, 0xcb697db0,
    0x8b61e733, 0xcb3badf3, 0x8b76a8e4, 0xcb0de658, 0x8b8b7c8f, 0xcae026e8,
    0x8ba0622f, 0xcab26fa9, 0x8bb559c1, 0xca84c0a3, 0x8bca6343, 0xca5719db,
    0x8bdf7eb0, 0xca297b5a, 0x8bf4ac05, 0xc9fbe527, 0x8c09eb40, 0xc9ce5748,
    0x8c1f3c5d, 0xc9a0d1c5, 0x8c349f58, 0xc97354a4, 0x8c4a142f, 0xc945dfec,
    0x8c5f9ade, 0xc91873a5, 0x8c753362, 0xc8eb0fd6, 0x8c8addb7, 0xc8bdb485,
    0x8ca099da, 0xc89061ba, 0x8cb667c8, 0xc863177b, 0x8ccc477d, 0xc835d5d0,
    0x8ce238f6, 0xc8089cbf, 0x8cf83c30, 0xc7db6c50, 0x8d0e5127, 0xc7ae4489,
    0x8d2477d8, 0xc7812572, 0x8d3ab03f, 0xc7540f11, 0x8d50fa59, 0xc727016d,
    0x8d675623, 0xc6f9fc8d, 0x8d7dc399, 0xc6cd0079, 0x8d9442b8, 0xc6a00d37,
    0x8daad37b, 0xc67322ce, 0x8dc175e0, 0xc6464144, 0x8dd829e4, 0xc61968a2,
    0x8deeef82, 0xc5ec98ee, 0x8e05c6b7, 0xc5bfd22e, 0x8e1caf80, 0xc593146a,
    0x8e33a9da, 0xc5665fa9, 0x8e4ab5bf, 0xc539b3f1, 0x8e61d32e, 0xc50d1149,
    0x8e790222, 0xc4e077b8, 0x8e904298, 0xc4b3e746, 0x8ea7948c, 0xc4875ff9,
    0x8ebef7fb, 0xc45ae1d7, 0x8ed66ce1, 0xc42e6ce8, 0x8eedf33b, 0xc4020133,
    0x8f058b04, 0xc3d59ebe, 0x8f1d343a, 0xc3a94590, 0x8f34eed8, 0xc37cf5b0,
    0x8f4cbadb, 0xc350af26, 0x8f649840, 0xc32471f7, 0x8f7c8701, 0xc2f83e2a,
    0x8f94871d, 0xc2cc13c7, 0x8fac988f, 0xc29ff2d4, 0x8fc4bb53, 0xc273db58,
    0x8fdcef66, 0xc247cd5a, 0x8ff534c4, 0xc21bc8e1, 0x900d8b69, 0xc1efcdf3,
    0x9025f352, 0xc1c3dc97, 0x903e6c7b, 0xc197f4d4, 0x9056f6df, 0xc16c16b0,
    0x906f927c, 0xc1404233, 0x90883f4d, 0xc1147764, 0x90a0fd4e, 0xc0e8b648,
    0x90b9cc7d, 0xc0bcfee7, 0x90d2acd4, 0xc0915148, 0x90eb9e50, 0xc065ad70,
    0x9104a0ee, 0xc03a1368, 0x911db4a9, 0xc00e8336, 0x9136d97d, 0xbfe2fcdf,
    0x91500f67, 0xbfb7806c, 0x91695663, 0xbf8c0de3, 0x9182ae6d, 0xbf60a54a,
    0x919c1781, 0xbf3546a8, 0x91b5919a, 0xbf09f205, 0x91cf1cb6, 0xbedea765,
    0x91e8b8d0, 0xbeb366d1, 0x920265e4, 0xbe88304f, 0x921c23ef, 0xbe5d03e6,
    0x9235f2ec, 0xbe31e19b, 0x924fd2d7, 0xbe06c977, 0x9269c3ac, 0xbddbbb7f,
    0x9283c568, 0xbdb0b7bb, 0x929dd806, 0xbd85be30, 0x92b7fb82, 0xbd5acee5,
    0x92d22fd9, 0xbd2fe9e2, 0x92ec7505, 0xbd050f2c, 0x9306cb04, 0xbcda3ecb,
    0x932131d1, 0xbcaf78c4, 0x933ba968, 0xbc84bd1f, 0x935631c5, 0xbc5a0be2,
    0x9370cae4, 0xbc2f6513, 0x938b74c1, 0xbc04c8ba, 0x93a62f57, 0xbbda36dd,
    0x93c0faa3, 0xbbafaf82, 0x93dbd6a0, 0xbb8532b0, 0x93f6c34a, 0xbb5ac06d,
    0x9411c09e, 0xbb3058c0, 0x942cce96, 0xbb05fbb0, 0x9447ed2f, 0xbadba943,
    0x94631c65, 0xbab16180, 0x947e5c33, 0xba87246d, 0x9499ac95, 0xba5cf210,
    0x94b50d87, 0xba32ca71, 0x94d07f05, 0xba08ad95, 0x94ec010b, 0xb9de9b83,
    0x95079394, 0xb9b49442, 0x9523369c, 0xb98a97d8, 0x953eea1e, 0xb960a64c,
    0x955aae17, 0xb936bfa4, 0x95768283, 0xb90ce3e6, 0x9592675c, 0xb8e31319,
    0x95ae5c9f, 0xb8b94d44, 0x95ca6247, 0xb88f926d, 0x95e67850, 0xb865e299,
    0x96029eb6, 0xb83c3dd1, 0x961ed574, 0xb812a41a, 0x963b1c86, 0xb7e9157a,
    0x965773e7, 0xb7bf91f8, 0x9673db94, 0xb796199b, 0x96905388, 0xb76cac69,
    0x96acdbbe, 0xb7434a67, 0x96c97432, 0xb719f39e, 0x96e61ce0, 0xb6f0a812,
    0x9702d5c3, 0xb6c767ca, 0x971f9ed7, 0xb69e32cd, 0x973c7817, 0xb6750921,
    0x9759617f, 0xb64beacd, 0x97765b0a, 0xb622d7d6, 0x979364b5, 0xb5f9d043,
    0x97b07e7a, 0xb5d0d41a, 0x97cda855, 0xb5a7e362, 0x97eae242, 0xb57efe22,
    0x98082c3b, 0xb556245e, 0x9825863d, 0xb52d561e, 0x9842f043, 0xb5049368,
    0x98606a49, 0xb4dbdc42, 0x987df449, 0xb4b330b3, 0x989b8e40, 0xb48a90c0,
    0x98b93828, 0xb461fc70, 0x98d6f1fe, 0xb43973ca, 0x98f4bbbc, 0xb410f6d3,
    0x9912955f, 0xb3e88592, 0x99307ee0, 0xb3c0200c, 0x994e783d, 0xb397c649,
    0x996c816f, 0xb36f784f, 0x998a9a74, 0xb3473623, 0x99a8c345, 0xb31effcc,
    0x99c6fbde, 0xb2f6d550, 0x99e5443b, 0xb2ceb6b5, 0x9a039c57, 0xb2a6a402,
    0x9a22042d, 0xb27e9d3c, 0x9a407bb9, 0xb256a26a, 0x9a5f02f5, 0xb22eb392,
    0x9a7d99de, 0xb206d0ba, 0x9a9c406e, 0xb1def9e9, 0x9abaf6a1, 0xb1b72f23,
    0x9ad9bc71, 0xb18f7071, 0x9af891db, 0xb167bdd7, 0x9b1776da, 0xb140175b,
    0x9b366b68, 0xb1187d05, 0x9b556f81, 0xb0f0eeda, 0x9b748320, 0xb0c96ce0,
    0x9b93a641, 0xb0a1f71d, 0x9bb2d8de, 0xb07a8d97, 0x9bd21af3, 0xb0533055,
    0x9bf16c7a, 0xb02bdf5c, 0x9c10cd70, 0xb0049ab3, 0x9c303dcf, 0xafdd625f,
    0x9c4fbd93, 0xafb63667, 0x9c6f4cb6, 0xaf8f16d1, 0x9c8eeb34, 0xaf6803a2,
    0x9cae9907, 0xaf40fce1, 0x9cce562c, 0xaf1a0293, 0x9cee229c, 0xaef314c0,
    0x9d0dfe54, 0xaecc336c, 0x9d2de94d, 0xaea55e9e, 0x9d4de385, 0xae7e965b,
    0x9d6decf4, 0xae57daab, 0x9d8e0597, 0xae312b92, 0x9dae2d68, 0xae0a8916,
    0x9dce6463, 0xade3f33e, 0x9deeaa82, 0xadbd6a10, 0x9e0effc1, 0xad96ed92,
    0x9e2f641b, 0xad707dc8, 0x9e4fd78a, 0xad4a1aba, 0x9e705a09, 0xad23c46e,
    0x9e90eb94, 0xacfd7ae8, 0x9eb18c26, 0xacd73e30, 0x9ed23bb9, 0xacb10e4b,
    0x9ef2fa49, 0xac8aeb3e, 0x9f13c7d0, 0xac64d510, 0x9f34a449, 0xac3ecbc7,
    0x9f558fb0, 0xac18cf69, 0x9f7689ff, 0xabf2dffb, 0x9f979331, 0xabccfd83,
    0x9fb8ab41, 0xaba72807, 0x9fd9d22a, 0xab815f8d, 0x9ffb07e7, 0xab5ba41a,
    0xa01c4c73, 0xab35f5b5, 0xa03d9fc8, 0xab105464, 0xa05f01e1, 0xaaeac02c,
    0xa08072ba, 0xaac53912, 0xa0a1f24d, 0xaa9fbf1e, 0xa0c38095, 0xaa7a5253,
    0xa0e51d8c, 0xaa54f2ba, 0xa106c92f, 0xaa2fa056, 0xa1288376, 0xaa0a5b2e,
    0xa14a4c5e, 0xa9e52347, 0xa16c23e1, 0xa9bff8a8, 0xa18e09fa, 0xa99adb56,
    0xa1affea3, 0xa975cb57, 0xa1d201d7, 0xa950c8b0, 0xa1f41392, 0xa92bd367,
    0xa21633cd, 0xa906eb82, 0xa2386284, 0xa8e21106, 0xa25a9fb1, 0xa8bd43fa,
    0xa27ceb4f, 0xa8988463, 0xa29f4559, 0xa873d246, 0xa2c1adc9, 0xa84f2daa,
    0xa2e4249b, 0xa82a9693, 0xa306a9c8, 0xa8060d08, 0xa3293d4b, 0xa7e1910f,
    0xa34bdf20, 0xa7bd22ac, 0xa36e8f41, 0xa798c1e5, 0xa3914da8, 0xa7746ec0,
    0xa3b41a50, 0xa7502943, 0xa3d6f534, 0xa72bf174, 0xa3f9de4e, 0xa707c757,
    0xa41cd599, 0xa6e3aaf2, 0xa43fdb10, 0xa6bf9c4b, 0xa462eeac, 0xa69b9b68,
    0xa486106a, 0xa677a84e, 0xa4a94043, 0xa653c303, 0xa4cc7e32, 0xa62feb8b,
    0xa4efca31, 0xa60c21ee, 0xa513243b, 0xa5e8662f, 0xa5368c4b, 0xa5c4b855,
    0xa55a025b, 0xa5a11866, 0xa57d8666, 0xa57d8666, 0xa5a11866, 0xa55a025b,
    0xa5c4b855, 0xa5368c4b, 0xa5e8662f, 0xa513243b, 0xa60c21ee, 0xa4efca31,
    0xa62feb8b, 0xa4cc7e32, 0xa653c303, 0xa4a94043, 0xa677a84e, 0xa486106a,
    0xa69b9b68, 0xa462eeac, 0xa6bf9c4b, 0xa43fdb10, 0xa6e3aaf2, 0xa41cd599,
    0xa707c757, 0xa3f9de4e, 0xa72bf174, 0xa3d6f534, 0xa7502943, 0xa3b41a50,
    0xa7746ec0, 0xa3914da8, 0xa798c1e5, 0xa36e8f41, 0xa7bd22ac, 0xa34bdf20,
    0xa7e1910f, 0xa3293d4b, 0xa8060d08, 0xa306a9c8, 0xa82a9693, 0xa2e4249b,
    0xa84f2daa, 0xa2c1adc9, 0xa873d246, 0xa29f4559, 0xa8988463, 0xa27ceb4f,
    0xa8bd43fa, 0xa25a9fb1, 0xa8e21106, 0xa2386284, 0xa906eb82, 0xa21633cd,
    0xa92bd367, 0xa1f41392, 0xa950c8b0, 0xa1d201d7, 0xa975cb57, 0xa1affea3,
    0xa99adb56, 0xa18e09fa, 0xa9bff8a8, 0xa16c23e1, 0xa9e52347, 0xa14a4c5e,
    0xaa0a5b2e, 0xa1288376, 0xaa2fa056, 0xa106c92f, 0xaa54f2ba, 0xa0e51d8c,
    0xaa7a5253, 0xa0c38095, 0xaa9fbf1e, 0xa0a1f24d, 0xaac53912, 0xa08072ba,
    0xaaeac02c, 0xa05f01e1, 0xab105464, 0xa03d9fc8, 0xab35f5b5, 0xa01c4c73,
    0xab5ba41a, 0x9ffb07e7, 0xab815f8d, 0x9fd9d22a, 0xaba72807, 0x9fb8ab41,
    0xabccfd83, 0x9f979331, 0xabf2dffb, 0x9f7689ff, 0xac18cf69, 0x9f558fb0,
    0xac3ecbc7, 0x9f34a449, 0xac64d510, 0x9f13c7d0, 0xac8aeb3e, 0x9ef2fa49,
    0xacb10e4b, 0x9ed23bb9, 0xacd73e30, 0x9eb18c26, 0xacfd7ae8, 0x9e90eb94,
    0xad23c46e, 0x9e705a09, 0xad4a1aba, 0x9e4fd78a, 0xad707dc8, 0x9e2f641b,
    0xad96ed92, 0x9e0effc1, 0xadbd6a10, 0x9deeaa82, 0xade3f33e, 0x9dce6463,
    0xae0a8916, 0x9dae2d68, 0xae312b92, 0x9d8e0597, 0xae57daab, 0x9d6decf4,
    0xae7e965b, 0x9d4de385, 0xaea55e9e, 0x9d2de94d, 0xaecc336c, 0x9d0dfe54,
    0xaef314c0, 0x9cee229c, 0xaf1a0293, 0x9cce562c, 0xaf40fce1, 0x9cae9907,
    0xaf6803a2, 0x9c8eeb34, 0xaf8f16d1, 0x9c6f4cb6, 0xafb63667, 0x9c4fbd93,
    0xafdd625f, 0x9c303dcf, 0xb0049ab3, 0x9c10cd70, 0xb02bdf5c, 0x9bf16c7a,
    0xb0533055, 0x9bd21af3, 0xb07a8d97, 0x9bb2d8de, 0xb0a1f71d, 0x9b93a641,
    0xb0c96ce0, 0x9b748320, 0xb0f0eeda, 0x9b556f81, 0xb1187d05, 0x9b366b68,
    0xb140175b, 0x9b1776da, 0xb167bdd7, 0x9af891db, 0xb18f7071, 0x9ad9bc71,
    0xb1b72f23, 0x9abaf6a1, 0xb1def9e9, 0x9a9c406e, 0xb206d0ba, 0x9a7d99de,
    0xb22eb392, 0x9a5f02f5, 0xb256a26a, 0x9a407bb9, 0xb27e9d3c, 0x9a22042d,
    0xb2a6a402, 0x9a039c57, 0xb2ceb6b5, 0x99e5443b, 0xb2f6d550, 0x99c6fbde,
    0xb31effcc, 0x99a8c345, 0xb3473623, 0x998a9a74, 0xb36f784f, 0x996c816f,
    0xb397c649, 0x994e783d, 0xb3c0200c, 0x99307ee0, 0xb3e88592, 0x9912955f,
    0xb410f6d3, 0x98f4bbbc, 0xb43973ca, 0x98d6f1fe, 0xb461fc70, 0x98b93828,
    0xb48a90c0, 0x989b8e40, 0xb4b330b3, 0x987df449, 0xb4dbdc42, 0x98606a49,
    0xb5049368, 0x9842f043, 0xb52d561e, 0x9825863d, 0xb556245e, 0x98082c3b,
    0xb57efe22, 0x97eae242, 0xb5a7e362, 0x97cda855, 0xb5d0d41a, 0x97b07e7a,
    0xb5f9d043, 0x979364b5, 0xb622d7d6, 0x97765b0a, 0xb64beacd, 0x9759617f,
    0xb6750921, 0x973c7817, 0xb69e32cd, 0x971f9ed7, 0xb6c767ca, 0x9702d5c3,
    0xb6f0a812, 0x96e61ce0, 0xb719f39e, 0x96c97432, 0xb7434a67, 0x96acdbbe,
    0xb76cac69, 0x96905388, 0xb796199b, 0x9673db94, 0xb7bf91f8, 0x965773e7,
    0xb7e9157a, 0x963b1c86, 0xb812a41a, 0x961ed574, 0xb83c3dd1, 0x96029eb6,
    0xb865e299, 0x95e67850, 0xb88f926d, 0x95ca6247, 0xb8b94d44, 0x95ae5c9f,
    0xb8e31319, 0x9592675c, 0xb90ce3e6, 0x95768283, 0xb936bfa4, 0x955aae17,
    0xb960a64c, 0x953eea1e, 0xb98a97d8, 0x9523369c, 0xb9b49442, 0x95079394,
    0xb9de9b83, 0x94ec010b, 0xba08ad95, 0x94d07f05, 0xba32ca71, 0x94b50d87,
    0xba5cf210, 0x9499ac95, 0xba87246d, 0x947e5c33, 0xbab16180, 0x94631c65,
    0xbadba943, 0x9447ed2f, 0xbb05fbb0, 0x942cce96, 0xbb3058c0, 0x9411c09e,
    0xbb5ac06d, 0x93f6c34a, 0xbb8532b0, 0x93dbd6a0, 0xbbafaf82, 0x93c0faa3,
    0xbbda36dd, 0x93a62f57, 0xbc04c8ba, 0x938b74c1, 0xbc2f6513, 0x9370cae4,
    0xbc5a0be2, 0x935631c5, 0xbc84bd1f, 0x933ba968, 0xbcaf78c4, 0x932131d1,
    0xbcda3ecb, 0x9306cb04, 0xbd050f2c, 0x92ec7505, 0xbd2fe9e2, 0x92d22fd9,
    0xbd5acee5, 0x92b7fb82, 0xbd85be30, 0x929dd806, 0xbdb0b7bb, 0x9283c568,
    0xbddbbb7f, 0x9269c3ac, 0xbe06c977, 0x924fd2d7, 0xbe31e19b, 0x9235f2ec,
    0xbe5d03e6, 0x921c23ef, 0xbe88304f, 0x920265e4, 0xbeb366d1, 0x91e8b8d0,
    0xbedea765, 0x91cf1cb6, 0xbf09f205, 0x91b5919a, 0xbf3546a8, 0x919c1781,
    0xbf60a54a, 0x9182ae6d, 0xbf8c0de3, 0x91695663, 0xbfb7806c, 0x91500f67,
    0xbfe2fcdf, 0x9136d97d, 0xc00e8336, 0x911db4a9, 0xc03a1368, 0x9104a0ee,
    0xc065ad70, 0x90eb9e50, 0xc0915148, 0x90d2acd4, 0xc0bcfee7, 0x90b9cc7d,
    0xc0e8b648, 0x90a0fd4e, 0xc1147764, 0x90883f4d, 0xc1404233, 0x906f927c,
    0xc16c16b0, 0x9056f6df, 0xc197f4d4, 0x903e6c7b, 0xc1c3dc97, 0x9025f352,
    0xc1efcdf3, 0x900d8b69, 0xc21bc8e1, 0x8ff534c4, 0xc247cd5a, 0x8fdcef66,
    0xc273db58, 0x8fc4bb53, 0xc29ff2d4, 0x8fac988f, 0xc2cc13c7, 0x8f94871d,
    0xc2f83e2a, 0x8f7c8701, 0xc32471f7, 0x8f649840, 0xc350af26, 0x8f4cbadb,
    0xc37cf5b0, 0x8f34eed8, 0xc3a94590, 0x8f1d343a, 0xc3d59ebe, 0x8f058b04,
    0xc4020133, 0x8eedf33b, 0xc42e6ce8, 0x8ed66ce1, 0xc45ae1d7, 0x8ebef7fb,
    0xc4875ff9, 0x8ea7948c, 0xc4b3e746, 0x8e904298, 0xc4e077b8, 0x8e790222,
    0xc50d1149, 0x8e61d32e, 0xc539b3f1, 0x8e4ab5bf, 0xc5665fa9, 0x8e33a9da,
    0xc593146a, 0x8e1caf80, 0xc5bfd22e, 0x8e05c6b7, 0xc5ec98ee, 0x8deeef82,
    0xc61968a2, 0x8dd829e4, 0xc6464144, 0x8dc175e0, 0xc67322ce, 0x8daad37b,
    0xc6a00d37, 0x8d9442b8, 0xc6cd0079, 0x8d7dc399, 0xc6f9fc8d, 0x8d675623,
    0xc727016d, 0x8d50fa59, 0xc7540f11, 0x8d3ab03f, 0xc7812572, 0x8d2477d8,
    0xc7ae4489, 0x8d0e5127, 0xc7db6c50, 0x8cf83c30, 0xc8089cbf, 0x8ce238f6,
    0xc835d5d0, 0x8ccc477d, 0xc863177b, 0x8cb667c8, 0xc89061ba, 0x8ca099da,
    0xc8bdb485, 0x8c8addb7, 0xc8eb0fd6, 0x8c753362, 0xc91873a5, 0x8c5f9ade,
    0xc945dfec, 0x8c4a142f, 0xc97354a4, 0x8c349f58, 0xc9a0d1c5, 0x8c1f3c5d,
    0xc9ce5748, 0x8c09eb40, 0xc9fbe527, 0x8bf4ac05, 0xca297b5a, 0x8bdf7eb0,
    0xca5719db, 0x8bca6343, 0xca84c0a3, 0x8bb559c1, 0xcab26fa9, 0x8ba0622f,
    0xcae026e8, 0x8b8b7c8f, 0xcb0de658, 0x8b76a8e4, 0xcb3badf3, 0x8b61e733,
    0xcb697db0, 0x8b4d377c, 0xcb97558a, 0x8b3899c6, 0xcbc53579, 0x8b240e11,
    0xcbf31d75, 0x8b0f9462, 0xcc210d79, 0x8afb2cbb, 0xcc4f057c, 0x8ae6d720,
    0xcc7d0578, 0x8ad29394, 0xccab0d65, 0x8abe6219, 0xccd91d3d, 0x8aaa42b4,
    0xcd0734f9, 0x8a963567, 0xcd355491, 0x8a823a36, 0xcd637bfe, 0x8a6e5123,
    0xcd91ab39, 0x8a5a7a31, 0xcdbfe23a, 0x8a46b564, 0xcdee20fc, 0x8a3302be,
    0xce1c6777, 0x8a1f6243, 0xce4ab5a2, 0x8a0bd3f5, 0xce790b79, 0x89f857d8,
    0xcea768f2, 0x89e4edef, 0xced5ce08, 0x89d1963c, 0xcf043ab3, 0x89be50c3,
    0xcf32aeeb, 0x89ab1d87, 0xcf612aaa, 0x8997fc8a, 0xcf8fade9, 0x8984edcf,
    0xcfbe389f, 0x8971f15a, 0xcfeccac7, 0x895f072e, 0xd01b6459, 0x894c2f4c,
    0xd04a054e, 0x893969b9, 0xd078ad9e, 0x8926b677, 0xd0a75d42, 0x89141589,
    0xd0d61434, 0x890186f2, 0xd104d26b, 0x88ef0ab4, 0xd13397e2, 0x88dca0d3,
    0xd1626490, 0x88ca4951, 0xd191386e, 0x88b80432, 0xd1c01375, 0x88a5d177,
    0xd1eef59e, 0x8893b125, 0xd21ddee2, 0x8881a33d, 0xd24ccf39, 0x886fa7c2,
    0xd27bc69c, 0x885dbeb8, 0xd2aac504, 0x884be821, 0xd2d9ca6a, 0x883a23ff,
    0xd308d6c7, 0x88287256, 0xd337ea12, 0x8816d327, 0xd3670446, 0x88054677,
    0xd396255a, 0x87f3cc48, 0xd3c54d47, 0x87e2649b, 0xd3f47c06, 0x87d10f75,
    0xd423b191, 0x87bfccd7, 0xd452eddf, 0x87ae9cc5, 0xd48230e9, 0x879d7f41,
    0xd4b17aa8, 0x878c744d, 0xd4e0cb15, 0x877b7bec, 0xd5102228, 0x876a9621,
    0xd53f7fda, 0x8759c2ef, 0xd56ee424, 0x87490258, 0xd59e4eff, 0x8738545e,
    0xd5cdc062, 0x8727b905, 0xd5fd3848, 0x8717304e, 0xd62cb6a8, 0x8706ba3d,
    0xd65c3b7b, 0x86f656d3, 0xd68bc6ba, 0x86e60614, 0xd6bb585e, 0x86d5c802,
    0xd6eaf05f, 0x86c59c9f, 0xd71a8eb5, 0x86b583ee, 0xd74a335b, 0x86a57df2,
    0xd779de47, 0x86958aac, 0xd7a98f73, 0x8685aa20, 0xd7d946d8, 0x8675dc4f,
    0xd809046e, 0x8666213c, 0xd838c82d, 0x865678eb, 0xd868920f, 0x8646e35c,
    0xd898620c, 0x86376092, 0xd8c8381d, 0x8627f091, 0xd8f81439, 0x86189359,
    0xd927f65b, 0x860948ef, 0xd957de7a, 0x85fa1153, 0xd987cc90, 0x85eaec88,
    0xd9b7c094, 0x85dbda91, 0xd9e7ba7f, 0x85ccdb70, 0xda17ba4a, 0x85bdef28,
    0xda47bfee, 0x85af15b9, 0xda77cb63, 0x85a04f28, 0xdaa7dca1, 0x85919b76,
    0xdad7f3a2, 0x8582faa5, 0xdb08105e, 0x85746cb8, 0xdb3832cd, 0x8565f1b0,
    0xdb685ae9, 0x85578991, 0xdb9888a8, 0x8549345c, 0xdbc8bc06, 0x853af214,
    0xdbf8f4f8, 0x852cc2bb, 0xdc293379, 0x851ea652, 0xdc597781, 0x85109cdd,
    0xdc89c109, 0x8502a65c, 0xdcba1008, 0x84f4c2d4, 0xdcea6478, 0x84e6f244,
    0xdd1abe51, 0x84d934b1, 0xdd4b1d8c, 0x84cb8a1b, 0xdd7b8220, 0x84bdf286,
    0xddabec08, 0x84b06df2, 0xdddc5b3b, 0x84a2fc62, 0xde0ccfb1, 0x84959dd9,
    0xde3d4964, 0x84885258, 0xde6dc84b, 0x847b19e1, 0xde9e4c60, 0x846df477,
    0xdeced59b, 0x8460e21a, 0xdeff63f4, 0x8453e2cf, 0xdf2ff764, 0x8446f695,
    0xdf608fe4, 0x843a1d70, 0xdf912d6b, 0x842d5762, 0xdfc1cff3, 0x8420a46c,
    0xdff27773, 0x84140490, 0xe02323e5, 0x840777d0, 0xe053d541, 0x83fafe2e,
    0xe0848b7f, 0x83ee97ad, 0xe0b54698, 0x83e2444d, 0xe0e60685, 0x83d60412,
    0xe116cb3d, 0x83c9d6fc, 0xe14794ba, 0x83bdbd0e, 0xe17862f3, 0x83b1b649,
    0xe1a935e2, 0x83a5c2b0, 0xe1da0d7e, 0x8399e244, 0xe20ae9c1, 0x838e1507,
    0xe23bcaa2, 0x83825afb, 0xe26cb01b, 0x8376b422, 0xe29d9a23, 0x836b207d,
    0xe2ce88b3, 0x835fa00f, 0xe2ff7bc3, 0x835432d8, 0xe330734d, 0x8348d8dc,
    0xe3616f48, 0x833d921b, 0xe3926fad, 0x83325e97, 0xe3c37474, 0x83273e52,
    0xe3f47d96, 0x831c314e, 0xe4258b0a, 0x8311378d, 0xe4569ccb, 0x83065110,
    0xe487b2d0, 0x82fb7dd8, 0xe4b8cd11, 0x82f0bde8, 0xe4e9eb87, 0x82e61141,
    0xe51b0e2a, 0x82db77e5, 0xe54c34f3, 0x82d0f1d5, 0xe57d5fda, 0x82c67f14,
    0xe5ae8ed8, 0x82bc1fa2, 0xe5dfc1e5, 0x82b1d381, 0xe610f8f9, 0x82a79ab3,
    0xe642340d, 0x829d753a, 0xe6737319, 0x82936317, 0xe6a4b616, 0x8289644b,
    0xe6d5fcfc, 0x827f78d8, 0xe70747c4, 0x8275a0c0, 0xe7389665, 0x826bdc04,
    0xe769e8d8, 0x82622aa6, 0xe79b3f16, 0x82588ca7, 0xe7cc9917, 0x824f0208,
    0xe7fdf6d4, 0x82458acc, 0xe82f5844, 0x823c26f3, 0xe860bd61, 0x8232d67f,
    0xe8922622, 0x82299971, 0xe8c39280, 0x82206fcc, 0xe8f50273, 0x82175990,
    0xe92675f4, 0x820e56be, 0xe957ecfb, 0x82056758, 0xe9896781, 0x81fc8b60,
    0xe9bae57d, 0x81f3c2d7, 0xe9ec66e8, 0x81eb0dbe, 0xea1debbb, 0x81e26c16,
    0xea4f73ee, 0x81d9dde1, 0xea80ff7a, 0x81d16321, 0xeab28e56, 0x81c8fbd6,
    0xeae4207a, 0x81c0a801, 0xeb15b5e1, 0x81b867a5, 0xeb474e81, 0x81b03ac2,
    0xeb78ea52, 0x81a82159, 0xebaa894f, 0x81a01b6d, 0xebdc2b6e, 0x819828fd,
    0xec0dd0a8, 0x81904a0c, 0xec3f78f6, 0x81887e9a, 0xec71244f, 0x8180c6a9,
    0xeca2d2ad, 0x8179223a, 0xecd48407, 0x8171914e, 0xed063856, 0x816a13e6,
    0xed37ef91, 0x8162aa04, 0xed69a9b3, 0x815b53a8, 0xed9b66b2, 0x815410d4,
    0xedcd2687, 0x814ce188, 0xedfee92b, 0x8145c5c7, 0xee30ae96, 0x813ebd90,
    0xee6276bf, 0x8137c8e6, 0xee9441a0, 0x8130e7c9, 0xeec60f31, 0x812a1a3a,
    0xeef7df6a, 0x8123603a, 0xef29b243, 0x811cb9ca, 0xef5b87b5, 0x811626ec,
    0xef8d5fb8, 0x810fa7a0, 0xefbf3a45, 0x81093be8, 0xeff11753, 0x8102e3c4,
    0xf022f6da, 0x80fc9f35, 0xf054d8d5, 0x80f66e3c, 0xf086bd39, 0x80f050db,
    0xf0b8a401, 0x80ea4712, 0xf0ea8d24, 0x80e450e2, 0xf11c789a, 0x80de6e4c,
    0xf14e665c, 0x80d89f51, 0xf1805662, 0x80d2e3f2, 0xf1b248a5, 0x80cd3c2f,
    0xf1e43d1c, 0x80c7a80a, 0xf21633c0, 0x80c22784, 0xf2482c8a, 0x80bcba9d,
    0xf27a2771, 0x80b76156, 0xf2ac246e, 0x80b21baf, 0xf2de2379, 0x80ace9ab,
    0xf310248a, 0x80a7cb49, 0xf342279b, 0x80a2c08b, 0xf3742ca2, 0x809dc971,
    0xf3a63398, 0x8098e5fb, 0xf3d83c77, 0x8094162c, 0xf40a4735, 0x808f5a02,
    0xf43c53cb, 0x808ab180, 0xf46e6231, 0x80861ca6, 0xf4a07261, 0x80819b74,
    0xf4d28451, 0x807d2dec, 0xf50497fb, 0x8078d40d, 0xf536ad56, 0x80748dd9,
    0xf568c45b, 0x80705b50, 0xf59add02, 0x806c3c74, 0xf5ccf743, 0x80683143,
    0xf5ff1318, 0x806439c0, 0xf6313077, 0x806055eb, 0xf6634f59, 0x805c85c4,
    0xf6956fb7, 0x8058c94c, 0xf6c79188, 0x80552084, 0xf6f9b4c6, 0x80518b6b,
    0xf72bd967, 0x804e0a04, 0xf75dff66, 0x804a9c4d, 0xf79026b9, 0x80474248,
    0xf7c24f59, 0x8043fbf6, 0xf7f4793e, 0x8040c956, 0xf826a462, 0x803daa6a,
    0xf858d0bb, 0x803a9f31, 0xf88afe42, 0x8037a7ac, 0xf8bd2cef, 0x8034c3dd,
    0xf8ef5cbb, 0x8031f3c2, 0xf9218d9e, 0x802f375d, 0xf953bf91, 0x802c8ead,
    0xf985f28a, 0x8029f9b4, 0xf9b82684, 0x80277872, 0xf9ea5b75, 0x80250ae7,
    0xfa1c9157, 0x8022b114, 0xfa4ec821, 0x80206af8, 0xfa80ffcb, 0x801e3895,
    0xfab3384f, 0x801c19ea, 0xfae571a4, 0x801a0ef8, 0xfb17abc2, 0x801817bf,
    0xfb49e6a3, 0x80163440, 0xfb7c223d, 0x8014647b, 0xfbae5e89, 0x8012a86f,
    0xfbe09b80, 0x8011001f, 0xfc12d91a, 0x800f6b88, 0xfc45174e, 0x800deaad,
    0xfc775616, 0x800c7d8c, 0xfca9956a, 0x800b2427, 0xfcdbd541, 0x8009de7e,
    0xfd0e1594, 0x8008ac90, 0xfd40565c, 0x80078e5e, 0xfd729790, 0x800683e8,
    0xfda4d929, 0x80058d2f, 0xfdd71b1e, 0x8004aa32, 0xfe095d69, 0x8003daf1,
    0xfe3ba002, 0x80031f6d, 0xfe6de2e0, 0x800277a6, 0xfea025fd, 0x8001e39b,
    0xfed2694f, 0x8001634e, 0xff04acd0, 0x8000f6bd, 0xff36f078, 0x80009dea,
    0xff69343f, 0x800058d4, 0xff9b781d, 0x8000277a, 0xffcdbc0b, 0x800009df,

};

/*
 * @brief  Q15 Twiddle factors Table
 */

/**
 * \par
 * Example code for Q15 Twiddle factors Generation::
 * \par
 * <pre>for(i = 0; i< 3N/4; i++)
 * {
 *	twiddleCoefQ15[2*i]= cos(i * 2*PI/(float)N);
 *	twiddleCoefQ15[2*i+1]= sin(i * 2*PI/(float)N);
 * } </pre>
 * \par
 * where N = 4096	and PI = 3.14159265358979
 * \par
 * Cos and Sin values are interleaved fashion
 * \par
 * Convert Floating point to Q15(Fixed point 1.15):
 *	round(twiddleCoefQ15(i) * pow(2, 15))
 *
 */

const q15_t ALIGN4 twiddleCoefQ15[6144] = {

    0x7fff, 0x0,    0x7fff, 0x32,   0x7fff, 0x65,   0x7fff, 0x97,   0x7fff,
    0xc9,   0x7fff, 0xfb,   0x7fff, 0x12e,  0x7ffe, 0x160,  0x7ffe, 0x192,
    0x7ffd, 0x1c4,  0x7ffc, 0x1f7,  0x7ffb, 0x229,  0x7ffa, 0x25b,  0x7ff9,
    0x28d,  0x7ff8, 0x2c0,  0x7ff7, 0x2f2,  0x7ff6, 0x324,  0x7ff5, 0x356,
    0x7ff4, 0x389,  0x7ff2, 0x3bb,  0x7ff1, 0x3ed,  0x7fef, 0x41f,  0x7fed,
    0x452,  0x7fec, 0x484,  0x7fea, 0x4b6,  0x7fe8, 0x4e8,  0x7fe6, 0x51b,
    0x7fe4, 0x54d,  0x7fe2, 0x57f,  0x7fe0, 0x5b1,  0x7fdd, 0x5e3,  0x7fdb,
    0x616,  0x7fd9, 0x648,  0x7fd6, 0x67a,  0x7fd3, 0x6ac,  0x7fd1, 0x6de,
    0x7fce, 0x711,  0x7fcb, 0x743,  0x7fc8, 0x775,  0x7fc5, 0x7a7,  0x7fc2,
    0x7d9,  0x7fbf, 0x80c,  0x7fbc, 0x83e,  0x7fb9, 0x870,  0x7fb5, 0x8a2,
    0x7fb2, 0x8d4,  0x7fae, 0x906,  0x7fab, 0x938,  0x7fa7, 0x96b,  0x7fa3,
    0x99d,  0x7fa0, 0x9cf,  0x7f9c, 0xa01,  0x7f98, 0xa33,  0x7f94, 0xa65,
    0x7f90, 0xa97,  0x7f8b, 0xac9,  0x7f87, 0xafb,  0x7f83, 0xb2d,  0x7f7e,
    0xb60,  0x7f7a, 0xb92,  0x7f75, 0xbc4,  0x7f71, 0xbf6,  0x7f6c, 0xc28,
    0x7f67, 0xc5a,  0x7f62, 0xc8c,  0x7f5d, 0xcbe,  0x7f58, 0xcf0,  0x7f53,
    0xd22,  0x7f4e, 0xd54,  0x7f49, 0xd86,  0x7f43, 0xdb8,  0x7f3e, 0xdea,
    0x7f38, 0xe1c,  0x7f33, 0xe4e,  0x7f2d, 0xe80,  0x7f27, 0xeb2,  0x7f22,
    0xee4,  0x7f1c, 0xf15,  0x7f16, 0xf47,  0x7f10, 0xf79,  0x7f0a, 0xfab,
    0x7f03, 0xfdd,  0x7efd, 0x100f, 0x7ef7, 0x1041, 0x7ef0, 0x1073, 0x7eea,
    0x10a4, 0x7ee3, 0x10d6, 0x7edd, 0x1108, 0x7ed6, 0x113a, 0x7ecf, 0x116c,
    0x7ec8, 0x119e, 0x7ec1, 0x11cf, 0x7eba, 0x1201, 0x7eb3, 0x1233, 0x7eac,
    0x1265, 0x7ea5, 0x1296, 0x7e9d, 0x12c8, 0x7e96, 0x12fa, 0x7e8e, 0x132b,
    0x7e87, 0x135d, 0x7e7f, 0x138f, 0x7e78, 0x13c1, 0x7e70, 0x13f2, 0x7e68,
    0x1424, 0x7e60, 0x1455, 0x7e58, 0x1487, 0x7e50, 0x14b9, 0x7e48, 0x14ea,
    0x7e3f, 0x151c, 0x7e37, 0x154d, 0x7e2f, 0x157f, 0x7e26, 0x15b1, 0x7e1e,
    0x15e2, 0x7e15, 0x1614, 0x7e0c, 0x1645, 0x7e03, 0x1677, 0x7dfb, 0x16a8,
    0x7df2, 0x16da, 0x7de9, 0x170b, 0x7de0, 0x173c, 0x7dd6, 0x176e, 0x7dcd,
    0x179f, 0x7dc4, 0x17d1, 0x7dba, 0x1802, 0x7db1, 0x1833, 0x7da7, 0x1865,
    0x7d9e, 0x1896, 0x7d94, 0x18c7, 0x7d8a, 0x18f9, 0x7d81, 0x192a, 0x7d77,
    0x195b, 0x7d6d, 0x198d, 0x7d63, 0x19be, 0x7d58, 0x19ef, 0x7d4e, 0x1a20,
    0x7d44, 0x1a51, 0x7d3a, 0x1a83, 0x7d2f, 0x1ab4, 0x7d25, 0x1ae5, 0x7d1a,
    0x1b16, 0x7d0f, 0x1b47, 0x7d05, 0x1b78, 0x7cfa, 0x1ba9, 0x7cef, 0x1bda,
    0x7ce4, 0x1c0c, 0x7cd9, 0x1c3d, 0x7cce, 0x1c6e, 0x7cc2, 0x1c9f, 0x7cb7,
    0x1cd0, 0x7cac, 0x1d01, 0x7ca0, 0x1d31, 0x7c95, 0x1d62, 0x7c89, 0x1d93,
    0x7c7e, 0x1dc4, 0x7c72, 0x1df5, 0x7c66, 0x1e26, 0x7c5a, 0x1e57, 0x7c4e,
    0x1e88, 0x7c42, 0x1eb8, 0x7c36, 0x1ee9, 0x7c2a, 0x1f1a, 0x7c1e, 0x1f4b,
    0x7c11, 0x1f7b, 0x7c05, 0x1fac, 0x7bf9, 0x1fdd, 0x7bec, 0x200e, 0x7bdf,
    0x203e, 0x7bd3, 0x206f, 0x7bc6, 0x209f, 0x7bb9, 0x20d0, 0x7bac, 0x2101,
    0x7b9f, 0x2131, 0x7b92, 0x2162, 0x7b85, 0x2192, 0x7b78, 0x21c3, 0x7b6a,
    0x21f3, 0x7b5d, 0x2224, 0x7b50, 0x2254, 0x7b42, 0x2284, 0x7b34, 0x22b5,
    0x7b27, 0x22e5, 0x7b19, 0x2316, 0x7b0b, 0x2346, 0x7afd, 0x2376, 0x7aef,
    0x23a7, 0x7ae1, 0x23d7, 0x7ad3, 0x2407, 0x7ac5, 0x2437, 0x7ab7, 0x2467,
    0x7aa8, 0x2498, 0x7a9a, 0x24c8, 0x7a8c, 0x24f8, 0x7a7d, 0x2528, 0x7a6e,
    0x2558, 0x7a60, 0x2588, 0x7a51, 0x25b8, 0x7a42, 0x25e8, 0x7a33, 0x2618,
    0x7a24, 0x2648, 0x7a15, 0x2678, 0x7a06, 0x26a8, 0x79f7, 0x26d8, 0x79e7,
    0x2708, 0x79d8, 0x2738, 0x79c9, 0x2768, 0x79b9, 0x2797, 0x79aa, 0x27c7,
    0x799a, 0x27f7, 0x798a, 0x2827, 0x797a, 0x2856, 0x796a, 0x2886, 0x795b,
    0x28b6, 0x794a, 0x28e5, 0x793a, 0x2915, 0x792a, 0x2945, 0x791a, 0x2974,
    0x790a, 0x29a4, 0x78f9, 0x29d3, 0x78e9, 0x2a03, 0x78d8, 0x2a32, 0x78c8,
    0x2a62, 0x78b7, 0x2a91, 0x78a6, 0x2ac1, 0x7895, 0x2af0, 0x7885, 0x2b1f,
    0x7874, 0x2b4f, 0x7863, 0x2b7e, 0x7851, 0x2bad, 0x7840, 0x2bdc, 0x782f,
    0x2c0c, 0x781e, 0x2c3b, 0x780c, 0x2c6a, 0x77fb, 0x2c99, 0x77e9, 0x2cc8,
    0x77d8, 0x2cf7, 0x77c6, 0x2d26, 0x77b4, 0x2d55, 0x77a2, 0x2d84, 0x7790,
    0x2db3, 0x777e, 0x2de2, 0x776c, 0x2e11, 0x775a, 0x2e40, 0x7748, 0x2e6f,
    0x7736, 0x2e9e, 0x7723, 0x2ecc, 0x7711, 0x2efb, 0x76fe, 0x2f2a, 0x76ec,
    0x2f59, 0x76d9, 0x2f87, 0x76c7, 0x2fb6, 0x76b4, 0x2fe5, 0x76a1, 0x3013,
    0x768e, 0x3042, 0x767b, 0x3070, 0x7668, 0x309f, 0x7655, 0x30cd, 0x7642,
    0x30fc, 0x762e, 0x312a, 0x761b, 0x3159, 0x7608, 0x3187, 0x75f4, 0x31b5,
    0x75e1, 0x31e4, 0x75cd, 0x3212, 0x75b9, 0x3240, 0x75a6, 0x326e, 0x7592,
    0x329d, 0x757e, 0x32cb, 0x756a, 0x32f9, 0x7556, 0x3327, 0x7542, 0x3355,
    0x752d, 0x3383, 0x7519, 0x33b1, 0x7505, 0x33df, 0x74f0, 0x340d, 0x74dc,
    0x343b, 0x74c7, 0x3469, 0x74b3, 0x3497, 0x749e, 0x34c4, 0x7489, 0x34f2,
    0x7475, 0x3520, 0x7460, 0x354e, 0x744b, 0x357b, 0x7436, 0x35a9, 0x7421,
    0x35d7, 0x740b, 0x3604, 0x73f6, 0x3632, 0x73e1, 0x365f, 0x73cb, 0x368d,
    0x73b6, 0x36ba, 0x73a0, 0x36e8, 0x738b, 0x3715, 0x7375, 0x3742, 0x735f,
    0x3770, 0x734a, 0x379d, 0x7334, 0x37ca, 0x731e, 0x37f7, 0x7308, 0x3825,
    0x72f2, 0x3852, 0x72dc, 0x387f, 0x72c5, 0x38ac, 0x72af, 0x38d9, 0x7299,
    0x3906, 0x7282, 0x3933, 0x726c, 0x3960, 0x7255, 0x398d, 0x723f, 0x39ba,
    0x7228, 0x39e7, 0x7211, 0x3a13, 0x71fa, 0x3a40, 0x71e3, 0x3a6d, 0x71cc,
    0x3a9a, 0x71b5, 0x3ac6, 0x719e, 0x3af3, 0x7187, 0x3b20, 0x7170, 0x3b4c,
    0x7158, 0x3b79, 0x7141, 0x3ba5, 0x712a, 0x3bd2, 0x7112, 0x3bfe, 0x70fa,
    0x3c2a, 0x70e3, 0x3c57, 0x70cb, 0x3c83, 0x70b3, 0x3caf, 0x709b, 0x3cdc,
    0x7083, 0x3d08, 0x706b, 0x3d34, 0x7053, 0x3d60, 0x703b, 0x3d8c, 0x7023,
    0x3db8, 0x700b, 0x3de4, 0x6ff2, 0x3e10, 0x6fda, 0x3e3c, 0x6fc2, 0x3e68,
    0x6fa9, 0x3e94, 0x6f90, 0x3ec0, 0x6f78, 0x3eec, 0x6f5f, 0x3f17, 0x6f46,
    0x3f43, 0x6f2d, 0x3f6f, 0x6f14, 0x3f9a, 0x6efb, 0x3fc6, 0x6ee2, 0x3ff1,
    0x6ec9, 0x401d, 0x6eb0, 0x4048, 0x6e97, 0x4074, 0x6e7d, 0x409f, 0x6e64,
    0x40cb, 0x6e4a, 0x40f6, 0x6e31, 0x4121, 0x6e17, 0x414d, 0x6dfe, 0x4178,
    0x6de4, 0x41a3, 0x6dca, 0x41ce, 0x6db0, 0x41f9, 0x6d96, 0x4224, 0x6d7c,
    0x424f, 0x6d62, 0x427a, 0x6d48, 0x42a5, 0x6d2e, 0x42d0, 0x6d14, 0x42fb,
    0x6cf9, 0x4326, 0x6cdf, 0x4351, 0x6cc4, 0x437b, 0x6caa, 0x43a6, 0x6c8f,
    0x43d1, 0x6c75, 0x43fb, 0x6c5a, 0x4426, 0x6c3f, 0x4450, 0x6c24, 0x447b,
    0x6c09, 0x44a5, 0x6bee, 0x44d0, 0x6bd3, 0x44fa, 0x6bb8, 0x4524, 0x6b9d,
    0x454f, 0x6b82, 0x4579, 0x6b66, 0x45a3, 0x6b4b, 0x45cd, 0x6b30, 0x45f7,
    0x6b14, 0x4621, 0x6af8, 0x464b, 0x6add, 0x4675, 0x6ac1, 0x469f, 0x6aa5,
    0x46c9, 0x6a89, 0x46f3, 0x6a6e, 0x471d, 0x6a52, 0x4747, 0x6a36, 0x4770,
    0x6a1a, 0x479a, 0x69fd, 0x47c4, 0x69e1, 0x47ed, 0x69c5, 0x4817, 0x69a9,
    0x4840, 0x698c, 0x486a, 0x6970, 0x4893, 0x6953, 0x48bd, 0x6937, 0x48e6,
    0x691a, 0x490f, 0x68fd, 0x4939, 0x68e0, 0x4962, 0x68c4, 0x498b, 0x68a7,
    0x49b4, 0x688a, 0x49dd, 0x686d, 0x4a06, 0x6850, 0x4a2f, 0x6832, 0x4a58,
    0x6815, 0x4a81, 0x67f8, 0x4aaa, 0x67da, 0x4ad3, 0x67bd, 0x4afb, 0x67a0,
    0x4b24, 0x6782, 0x4b4d, 0x6764, 0x4b75, 0x6747, 0x4b9e, 0x6729, 0x4bc7,
    0x670b, 0x4bef, 0x66ed, 0x4c17, 0x66d0, 0x4c40, 0x66b2, 0x4c68, 0x6693,
    0x4c91, 0x6675, 0x4cb9, 0x6657, 0x4ce1, 0x6639, 0x4d09, 0x661b, 0x4d31,
    0x65fc, 0x4d59, 0x65de, 0x4d81, 0x65c0, 0x4da9, 0x65a1, 0x4dd1, 0x6582,
    0x4df9, 0x6564, 0x4e21, 0x6545, 0x4e49, 0x6526, 0x4e71, 0x6507, 0x4e98,
    0x64e9, 0x4ec0, 0x64ca, 0x4ee8, 0x64ab, 0x4f0f, 0x648b, 0x4f37, 0x646c,
    0x4f5e, 0x644d, 0x4f85, 0x642e, 0x4fad, 0x640f, 0x4fd4, 0x63ef, 0x4ffb,
    0x63d0, 0x5023, 0x63b0, 0x504a, 0x6391, 0x5071, 0x6371, 0x5098, 0x6351,
    0x50bf, 0x6332, 0x50e6, 0x6312, 0x510d, 0x62f2, 0x5134, 0x62d2, 0x515b,
    0x62b2, 0x5181, 0x6292, 0x51a8, 0x6272, 0x51cf, 0x6252, 0x51f5, 0x6232,
    0x521c, 0x6211, 0x5243, 0x61f1, 0x5269, 0x61d1, 0x5290, 0x61b0, 0x52b6,
    0x6190, 0x52dc, 0x616f, 0x5303, 0x614e, 0x5329, 0x612e, 0x534f, 0x610d,
    0x5375, 0x60ec, 0x539b, 0x60cb, 0x53c1, 0x60aa, 0x53e7, 0x6089, 0x540d,
    0x6068, 0x5433, 0x6047, 0x5459, 0x6026, 0x547f, 0x6005, 0x54a4, 0x5fe4,
    0x54ca, 0x5fc2, 0x54f0, 0x5fa1, 0x5515, 0x5f80, 0x553b, 0x5f5e, 0x5560,
    0x5f3c, 0x5586, 0x5f1b, 0x55ab, 0x5ef9, 0x55d0, 0x5ed7, 0x55f6, 0x5eb6,
    0x561b, 0x5e94, 0x5640, 0x5e72, 0x5665, 0x5e50, 0x568a, 0x5e2e, 0x56af,
    0x5e0c, 0x56d4, 0x5dea, 0x56f9, 0x5dc8, 0x571e, 0x5da5, 0x5743, 0x5d83,
    0x5767, 0x5d61, 0x578c, 0x5d3e, 0x57b1, 0x5d1c, 0x57d5, 0x5cf9, 0x57fa,
    0x5cd7, 0x581e, 0x5cb4, 0x5843, 0x5c91, 0x5867, 0x5c6f, 0x588c, 0x5c4c,
    0x58b0, 0x5c29, 0x58d4, 0x5c06, 0x58f8, 0x5be3, 0x591c, 0x5bc0, 0x5940,
    0x5b9d, 0x5964, 0x5b7a, 0x5988, 0x5b57, 0x59ac, 0x5b34, 0x59d0, 0x5b10,
    0x59f4, 0x5aed, 0x5a18, 0x5ac9, 0x5a3b, 0x5aa6, 0x5a5f, 0x5a82, 0x5a82,
    0x5a5f, 0x5aa6, 0x5a3b, 0x5ac9, 0x5a18, 0x5aed, 0x59f4, 0x5b10, 0x59d0,
    0x5b34, 0x59ac, 0x5b57, 0x5988, 0x5b7a, 0x5964, 0x5b9d, 0x5940, 0x5bc0,
    0x591c, 0x5be3, 0x58f8, 0x5c06, 0x58d4, 0x5c29, 0x58b0, 0x5c4c, 0x588c,
    0x5c6f, 0x5867, 0x5c91, 0x5843, 0x5cb4, 0x581e, 0x5cd7, 0x57fa, 0x5cf9,
    0x57d5, 0x5d1c, 0x57b1, 0x5d3e, 0x578c, 0x5d61, 0x5767, 0x5d83, 0x5743,
    0x5da5, 0x571e, 0x5dc8, 0x56f9, 0x5dea, 0x56d4, 0x5e0c, 0x56af, 0x5e2e,
    0x568a, 0x5e50, 0x5665, 0x5e72, 0x5640, 0x5e94, 0x561b, 0x5eb6, 0x55f6,
    0x5ed7, 0x55d0, 0x5ef9, 0x55ab, 0x5f1b, 0x5586, 0x5f3c, 0x5560, 0x5f5e,
    0x553b, 0x5f80, 0x5515, 0x5fa1, 0x54f0, 0x5fc2, 0x54ca, 0x5fe4, 0x54a4,
    0x6005, 0x547f, 0x6026, 0x5459, 0x6047, 0x5433, 0x6068, 0x540d, 0x6089,
    0x53e7, 0x60aa, 0x53c1, 0x60cb, 0x539b, 0x60ec, 0x5375, 0x610d, 0x534f,
    0x612e, 0x5329, 0x614e, 0x5303, 0x616f, 0x52dc, 0x6190, 0x52b6, 0x61b0,
    0x5290, 0x61d1, 0x5269, 0x61f1, 0x5243, 0x6211, 0x521c, 0x6232, 0x51f5,
    0x6252, 0x51cf, 0x6272, 0x51a8, 0x6292, 0x5181, 0x62b2, 0x515b, 0x62d2,
    0x5134, 0x62f2, 0x510d, 0x6312, 0x50e6, 0x6332, 0x50bf, 0x6351, 0x5098,
    0x6371, 0x5071, 0x6391, 0x504a, 0x63b0, 0x5023, 0x63d0, 0x4ffb, 0x63ef,
    0x4fd4, 0x640f, 0x4fad, 0x642e, 0x4f85, 0x644d, 0x4f5e, 0x646c, 0x4f37,
    0x648b, 0x4f0f, 0x64ab, 0x4ee8, 0x64ca, 0x4ec0, 0x64e9, 0x4e98, 0x6507,
    0x4e71, 0x6526, 0x4e49, 0x6545, 0x4e21, 0x6564, 0x4df9, 0x6582, 0x4dd1,
    0x65a1, 0x4da9, 0x65c0, 0x4d81, 0x65de, 0x4d59, 0x65fc, 0x4d31, 0x661b,
    0x4d09, 0x6639, 0x4ce1, 0x6657, 0x4cb9, 0x6675, 0x4c91, 0x6693, 0x4c68,
    0x66b2, 0x4c40, 0x66d0, 0x4c17, 0x66ed, 0x4bef, 0x670b, 0x4bc7, 0x6729,
    0x4b9e, 0x6747, 0x4b75, 0x6764, 0x4b4d, 0x6782, 0x4b24, 0x67a0, 0x4afb,
    0x67bd, 0x4ad3, 0x67da, 0x4aaa, 0x67f8, 0x4a81, 0x6815, 0x4a58, 0x6832,
    0x4a2f, 0x6850, 0x4a06, 0x686d, 0x49dd, 0x688a, 0x49b4, 0x68a7, 0x498b,
    0x68c4, 0x4962, 0x68e0, 0x4939, 0x68fd, 0x490f, 0x691a, 0x48e6, 0x6937,
    0x48bd, 0x6953, 0x4893, 0x6970, 0x486a, 0x698c, 0x4840, 0x69a9, 0x4817,
    0x69c5, 0x47ed, 0x69e1, 0x47c4, 0x69fd, 0x479a, 0x6a1a, 0x4770, 0x6a36,
    0x4747, 0x6a52, 0x471d, 0x6a6e, 0x46f3, 0x6a89, 0x46c9, 0x6aa5, 0x469f,
    0x6ac1, 0x4675, 0x6add, 0x464b, 0x6af8, 0x4621, 0x6b14, 0x45f7, 0x6b30,
    0x45cd, 0x6b4b, 0x45a3, 0x6b66, 0x4579, 0x6b82, 0x454f, 0x6b9d, 0x4524,
    0x6bb8, 0x44fa, 0x6bd3, 0x44d0, 0x6bee, 0x44a5, 0x6c09, 0x447b, 0x6c24,
    0x4450, 0x6c3f, 0x4426, 0x6c5a, 0x43fb, 0x6c75, 0x43d1, 0x6c8f, 0x43a6,
    0x6caa, 0x437b, 0x6cc4, 0x4351, 0x6cdf, 0x4326, 0x6cf9, 0x42fb, 0x6d14,
    0x42d0, 0x6d2e, 0x42a5, 0x6d48, 0x427a, 0x6d62, 0x424f, 0x6d7c, 0x4224,
    0x6d96, 0x41f9, 0x6db0, 0x41ce, 0x6dca, 0x41a3, 0x6de4, 0x4178, 0x6dfe,
    0x414d, 0x6e17, 0x4121, 0x6e31, 0x40f6, 0x6e4a, 0x40cb, 0x6e64, 0x409f,
    0x6e7d, 0x4074, 0x6e97, 0x4048, 0x6eb0, 0x401d, 0x6ec9, 0x3ff1, 0x6ee2,
    0x3fc6, 0x6efb, 0x3f9a, 0x6f14, 0x3f6f, 0x6f2d, 0x3f43, 0x6f46, 0x3f17,
    0x6f5f, 0x3eec, 0x6f78, 0x3ec0, 0x6f90, 0x3e94, 0x6fa9, 0x3e68, 0x6fc2,
    0x3e3c, 0x6fda, 0x3e10, 0x6ff2, 0x3de4, 0x700b, 0x3db8, 0x7023, 0x3d8c,
    0x703b, 0x3d60, 0x7053, 0x3d34, 0x706b, 0x3d08, 0x7083, 0x3cdc, 0x709b,
    0x3caf, 0x70b3, 0x3c83, 0x70cb, 0x3c57, 0x70e3, 0x3c2a, 0x70fa, 0x3bfe,
    0x7112, 0x3bd2, 0x712a, 0x3ba5, 0x7141, 0x3b79, 0x7158, 0x3b4c, 0x7170,
    0x3b20, 0x7187, 0x3af3, 0x719e, 0x3ac6, 0x71b5, 0x3a9a, 0x71cc, 0x3a6d,
    0x71e3, 0x3a40, 0x71fa, 0x3a13, 0x7211, 0x39e7, 0x7228, 0x39ba, 0x723f,
    0x398d, 0x7255, 0x3960, 0x726c, 0x3933, 0x7282, 0x3906, 0x7299, 0x38d9,
    0x72af, 0x38ac, 0x72c5, 0x387f, 0x72dc, 0x3852, 0x72f2, 0x3825, 0x7308,
    0x37f7, 0x731e, 0x37ca, 0x7334, 0x379d, 0x734a, 0x3770, 0x735f, 0x3742,
    0x7375, 0x3715, 0x738b, 0x36e8, 0x73a0, 0x36ba, 0x73b6, 0x368d, 0x73cb,
    0x365f, 0x73e1, 0x3632, 0x73f6, 0x3604, 0x740b, 0x35d7, 0x7421, 0x35a9,
    0x7436, 0x357b, 0x744b, 0x354e, 0x7460, 0x3520, 0x7475, 0x34f2, 0x7489,
    0x34c4, 0x749e, 0x3497, 0x74b3, 0x3469, 0x74c7, 0x343b, 0x74dc, 0x340d,
    0x74f0, 0x33df, 0x7505, 0x33b1, 0x7519, 0x3383, 0x752d, 0x3355, 0x7542,
    0x3327, 0x7556, 0x32f9, 0x756a, 0x32cb, 0x757e, 0x329d, 0x7592, 0x326e,
    0x75a6, 0x3240, 0x75b9, 0x3212, 0x75cd, 0x31e4, 0x75e1, 0x31b5, 0x75f4,
    0x3187, 0x7608, 0x3159, 0x761b, 0x312a, 0x762e, 0x30fc, 0x7642, 0x30cd,
    0x7655, 0x309f, 0x7668, 0x3070, 0x767b, 0x3042, 0x768e, 0x3013, 0x76a1,
    0x2fe5, 0x76b4, 0x2fb6, 0x76c7, 0x2f87, 0x76d9, 0x2f59, 0x76ec, 0x2f2a,
    0x76fe, 0x2efb, 0x7711, 0x2ecc, 0x7723, 0x2e9e, 0x7736, 0x2e6f, 0x7748,
    0x2e40, 0x775a, 0x2e11, 0x776c, 0x2de2, 0x777e, 0x2db3, 0x7790, 0x2d84,
    0x77a2, 0x2d55, 0x77b4, 0x2d26, 0x77c6, 0x2cf7, 0x77d8, 0x2cc8, 0x77e9,
    0x2c99, 0x77fb, 0x2c6a, 0x780c, 0x2c3b, 0x781e, 0x2c0c, 0x782f, 0x2bdc,
    0x7840, 0x2bad, 0x7851, 0x2b7e, 0x7863, 0x2b4f, 0x7874, 0x2b1f, 0x7885,
    0x2af0, 0x7895, 0x2ac1, 0x78a6, 0x2a91, 0x78b7, 0x2a62, 0x78c8, 0x2a32,
    0x78d8, 0x2a03, 0x78e9, 0x29d3, 0x78f9, 0x29a4, 0x790a, 0x2974, 0x791a,
    0x2945, 0x792a, 0x2915, 0x793a, 0x28e5, 0x794a, 0x28b6, 0x795b, 0x2886,
    0x796a, 0x2856, 0x797a, 0x2827, 0x798a, 0x27f7, 0x799a, 0x27c7, 0x79aa,
    0x2797, 0x79b9, 0x2768, 0x79c9, 0x2738, 0x79d8, 0x2708, 0x79e7, 0x26d8,
    0x79f7, 0x26a8, 0x7a06, 0x2678, 0x7a15, 0x2648, 0x7a24, 0x2618, 0x7a33,
    0x25e8, 0x7a42, 0x25b8, 0x7a51, 0x2588, 0x7a60, 0x2558, 0x7a6e, 0x2528,
    0x7a7d, 0x24f8, 0x7a8c, 0x24c8, 0x7a9a, 0x2498, 0x7aa8, 0x2467, 0x7ab7,
    0x2437, 0x7ac5, 0x2407, 0x7ad3, 0x23d7, 0x7ae1, 0x23a7, 0x7aef, 0x2376,
    0x7afd, 0x2346, 0x7b0b, 0x2316, 0x7b19, 0x22e5, 0x7b27, 0x22b5, 0x7b34,
    0x2284, 0x7b42, 0x2254, 0x7b50, 0x2224, 0x7b5d, 0x21f3, 0x7b6a, 0x21c3,
    0x7b78, 0x2192, 0x7b85, 0x2162, 0x7b92, 0x2131, 0x7b9f, 0x2101, 0x7bac,
    0x20d0, 0x7bb9, 0x209f, 0x7bc6, 0x206f, 0x7bd3, 0x203e, 0x7bdf, 0x200e,
    0x7bec, 0x1fdd, 0x7bf9, 0x1fac, 0x7c05, 0x1f7b, 0x7c11, 0x1f4b, 0x7c1e,
    0x1f1a, 0x7c2a, 0x1ee9, 0x7c36, 0x1eb8, 0x7c42, 0x1e88, 0x7c4e, 0x1e57,
    0x7c5a, 0x1e26, 0x7c66, 0x1df5, 0x7c72, 0x1dc4, 0x7c7e, 0x1d93, 0x7c89,
    0x1d62, 0x7c95, 0x1d31, 0x7ca0, 0x1d01, 0x7cac, 0x1cd0, 0x7cb7, 0x1c9f,
    0x7cc2, 0x1c6e, 0x7cce, 0x1c3d, 0x7cd9, 0x1c0c, 0x7ce4, 0x1bda, 0x7cef,
    0x1ba9, 0x7cfa, 0x1b78, 0x7d05, 0x1b47, 0x7d0f, 0x1b16, 0x7d1a, 0x1ae5,
    0x7d25, 0x1ab4, 0x7d2f, 0x1a83, 0x7d3a, 0x1a51, 0x7d44, 0x1a20, 0x7d4e,
    0x19ef, 0x7d58, 0x19be, 0x7d63, 0x198d, 0x7d6d, 0x195b, 0x7d77, 0x192a,
    0x7d81, 0x18f9, 0x7d8a, 0x18c7, 0x7d94, 0x1896, 0x7d9e, 0x1865, 0x7da7,
    0x1833, 0x7db1, 0x1802, 0x7dba, 0x17d1, 0x7dc4, 0x179f, 0x7dcd, 0x176e,
    0x7dd6, 0x173c, 0x7de0, 0x170b, 0x7de9, 0x16da, 0x7df2, 0x16a8, 0x7dfb,
    0x1677, 0x7e03, 0x1645, 0x7e0c, 0x1614, 0x7e15, 0x15e2, 0x7e1e, 0x15b1,
    0x7e26, 0x157f, 0x7e2f, 0x154d, 0x7e37, 0x151c, 0x7e3f, 0x14ea, 0x7e48,
    0x14b9, 0x7e50, 0x1487, 0x7e58, 0x1455, 0x7e60, 0x1424, 0x7e68, 0x13f2,
    0x7e70, 0x13c1, 0x7e78, 0x138f, 0x7e7f, 0x135d, 0x7e87, 0x132b, 0x7e8e,
    0x12fa, 0x7e96, 0x12c8, 0x7e9d, 0x1296, 0x7ea5, 0x1265, 0x7eac, 0x1233,
    0x7eb3, 0x1201, 0x7eba, 0x11cf, 0x7ec1, 0x119e, 0x7ec8, 0x116c, 0x7ecf,
    0x113a, 0x7ed6, 0x1108, 0x7edd, 0x10d6, 0x7ee3, 0x10a4, 0x7eea, 0x1073,
    0x7ef0, 0x1041, 0x7ef7, 0x100f, 0x7efd, 0xfdd,  0x7f03, 0xfab,  0x7f0a,
    0xf79,  0x7f10, 0xf47,  0x7f16, 0xf15,  0x7f1c, 0xee4,  0x7f22, 0xeb2,
    0x7f27, 0xe80,  0x7f2d, 0xe4e,  0x7f33, 0xe1c,  0x7f38, 0xdea,  0x7f3e,
    0xdb8,  0x7f43, 0xd86,  0x7f49, 0xd54,  0x7f4e, 0xd22,  0x7f53, 0xcf0,
    0x7f58, 0xcbe,  0x7f5d, 0xc8c,  0x7f62, 0xc5a,  0x7f67, 0xc28,  0x7f6c,
    0xbf6,  0x7f71, 0xbc4,  0x7f75, 0xb92,  0x7f7a, 0xb60,  0x7f7e, 0xb2d,
    0x7f83, 0xafb,  0x7f87, 0xac9,  0x7f8b, 0xa97,  0x7f90, 0xa65,  0x7f94,
    0xa33,  0x7f98, 0xa01,  0x7f9c, 0x9cf,  0x7fa0, 0x99d,  0x7fa3, 0x96b,
    0x7fa7, 0x938,  0x7fab, 0x906,  0x7fae, 0x8d4,  0x7fb2, 0x8a2,  0x7fb5,
    0x870,  0x7fb9, 0x83e,  0x7fbc, 0x80c,  0x7fbf, 0x7d9,  0x7fc2, 0x7a7,
    0x7fc5, 0x775,  0x7fc8, 0x743,  0x7fcb, 0x711,  0x7fce, 0x6de,  0x7fd1,
    0x6ac,  0x7fd3, 0x67a,  0x7fd6, 0x648,  0x7fd9, 0x616,  0x7fdb, 0x5e3,
    0x7fdd, 0x5b1,  0x7fe0, 0x57f,  0x7fe2, 0x54d,  0x7fe4, 0x51b,  0x7fe6,
    0x4e8,  0x7fe8, 0x4b6,  0x7fea, 0x484,  0x7fec, 0x452,  0x7fed, 0x41f,
    0x7fef, 0x3ed,  0x7ff1, 0x3bb,  0x7ff2, 0x389,  0x7ff4, 0x356,  0x7ff5,
    0x324,  0x7ff6, 0x2f2,  0x7ff7, 0x2c0,  0x7ff8, 0x28d,  0x7ff9, 0x25b,
    0x7ffa, 0x229,  0x7ffb, 0x1f7,  0x7ffc, 0x1c4,  0x7ffd, 0x192,  0x7ffe,
    0x160,  0x7ffe, 0x12e,  0x7fff, 0xfb,   0x7fff, 0xc9,   0x7fff, 0x97,
    0x7fff, 0x65,   0x7fff, 0x32,   0x7fff, 0x0,    0x7fff, 0xffce, 0x7fff,
    0xff9b, 0x7fff, 0xff69, 0x7fff, 0xff37, 0x7fff, 0xff05, 0x7fff, 0xfed2,
    0x7fff, 0xfea0, 0x7ffe, 0xfe6e, 0x7ffe, 0xfe3c, 0x7ffd, 0xfe09, 0x7ffc,
    0xfdd7, 0x7ffb, 0xfda5, 0x7ffa, 0xfd73, 0x7ff9, 0xfd40, 0x7ff8, 0xfd0e,
    0x7ff7, 0xfcdc, 0x7ff6, 0xfcaa, 0x7ff5, 0xfc77, 0x7ff4, 0xfc45, 0x7ff2,
    0xfc13, 0x7ff1, 0xfbe1, 0x7fef, 0xfbae, 0x7fed, 0xfb7c, 0x7fec, 0xfb4a,
    0x7fea, 0xfb18, 0x7fe8, 0xfae5, 0x7fe6, 0xfab3, 0x7fe4, 0xfa81, 0x7fe2,
    0xfa4f, 0x7fe0, 0xfa1d, 0x7fdd, 0xf9ea, 0x7fdb, 0xf9b8, 0x7fd9, 0xf986,
    0x7fd6, 0xf954, 0x7fd3, 0xf922, 0x7fd1, 0xf8ef, 0x7fce, 0xf8bd, 0x7fcb,
    0xf88b, 0x7fc8, 0xf859, 0x7fc5, 0xf827, 0x7fc2, 0xf7f4, 0x7fbf, 0xf7c2,
    0x7fbc, 0xf790, 0x7fb9, 0xf75e, 0x7fb5, 0xf72c, 0x7fb2, 0xf6fa, 0x7fae,
    0xf6c8, 0x7fab, 0xf695, 0x7fa7, 0xf663, 0x7fa3, 0xf631, 0x7fa0, 0xf5ff,
    0x7f9c, 0xf5cd, 0x7f98, 0xf59b, 0x7f94, 0xf569, 0x7f90, 0xf537, 0x7f8b,
    0xf505, 0x7f87, 0xf4d3, 0x7f83, 0xf4a0, 0x7f7e, 0xf46e, 0x7f7a, 0xf43c,
    0x7f75, 0xf40a, 0x7f71, 0xf3d8, 0x7f6c, 0xf3a6, 0x7f67, 0xf374, 0x7f62,
    0xf342, 0x7f5d, 0xf310, 0x7f58, 0xf2de, 0x7f53, 0xf2ac, 0x7f4e, 0xf27a,
    0x7f49, 0xf248, 0x7f43, 0xf216, 0x7f3e, 0xf1e4, 0x7f38, 0xf1b2, 0x7f33,
    0xf180, 0x7f2d, 0xf14e, 0x7f27, 0xf11c, 0x7f22, 0xf0eb, 0x7f1c, 0xf0b9,
    0x7f16, 0xf087, 0x7f10, 0xf055, 0x7f0a, 0xf023, 0x7f03, 0xeff1, 0x7efd,
    0xefbf, 0x7ef7, 0xef8d, 0x7ef0, 0xef5c, 0x7eea, 0xef2a, 0x7ee3, 0xeef8,
    0x7edd, 0xeec6, 0x7ed6, 0xee94, 0x7ecf, 0xee62, 0x7ec8, 0xee31, 0x7ec1,
    0xedff, 0x7eba, 0xedcd, 0x7eb3, 0xed9b, 0x7eac, 0xed6a, 0x7ea5, 0xed38,
    0x7e9d, 0xed06, 0x7e96, 0xecd5, 0x7e8e, 0xeca3, 0x7e87, 0xec71, 0x7e7f,
    0xec3f, 0x7e78, 0xec0e, 0x7e70, 0xebdc, 0x7e68, 0xebab, 0x7e60, 0xeb79,
    0x7e58, 0xeb47, 0x7e50, 0xeb16, 0x7e48, 0xeae4, 0x7e3f, 0xeab3, 0x7e37,
    0xea81, 0x7e2f, 0xea4f, 0x7e26, 0xea1e, 0x7e1e, 0xe9ec, 0x7e15, 0xe9bb,
    0x7e0c, 0xe989, 0x7e03, 0xe958, 0x7dfb, 0xe926, 0x7df2, 0xe8f5, 0x7de9,
    0xe8c4, 0x7de0, 0xe892, 0x7dd6, 0xe861, 0x7dcd, 0xe82f, 0x7dc4, 0xe7fe,
    0x7dba, 0xe7cd, 0x7db1, 0xe79b, 0x7da7, 0xe76a, 0x7d9e, 0xe739, 0x7d94,
    0xe707, 0x7d8a, 0xe6d6, 0x7d81, 0xe6a5, 0x7d77, 0xe673, 0x7d6d, 0xe642,
    0x7d63, 0xe611, 0x7d58, 0xe5e0, 0x7d4e, 0xe5af, 0x7d44, 0xe57d, 0x7d3a,
    0xe54c, 0x7d2f, 0xe51b, 0x7d25, 0xe4ea, 0x7d1a, 0xe4b9, 0x7d0f, 0xe488,
    0x7d05, 0xe457, 0x7cfa, 0xe426, 0x7cef, 0xe3f4, 0x7ce4, 0xe3c3, 0x7cd9,
    0xe392, 0x7cce, 0xe361, 0x7cc2, 0xe330, 0x7cb7, 0xe2ff, 0x7cac, 0xe2cf,
    0x7ca0, 0xe29e, 0x7c95, 0xe26d, 0x7c89, 0xe23c, 0x7c7e, 0xe20b, 0x7c72,
    0xe1da, 0x7c66, 0xe1a9, 0x7c5a, 0xe178, 0x7c4e, 0xe148, 0x7c42, 0xe117,
    0x7c36, 0xe0e6, 0x7c2a, 0xe0b5, 0x7c1e, 0xe085, 0x7c11, 0xe054, 0x7c05,
    0xe023, 0x7bf9, 0xdff2, 0x7bec, 0xdfc2, 0x7bdf, 0xdf91, 0x7bd3, 0xdf61,
    0x7bc6, 0xdf30, 0x7bb9, 0xdeff, 0x7bac, 0xdecf, 0x7b9f, 0xde9e, 0x7b92,
    0xde6e, 0x7b85, 0xde3d, 0x7b78, 0xde0d, 0x7b6a, 0xdddc, 0x7b5d, 0xddac,
    0x7b50, 0xdd7c, 0x7b42, 0xdd4b, 0x7b34, 0xdd1b, 0x7b27, 0xdcea, 0x7b19,
    0xdcba, 0x7b0b, 0xdc8a, 0x7afd, 0xdc59, 0x7aef, 0xdc29, 0x7ae1, 0xdbf9,
    0x7ad3, 0xdbc9, 0x7ac5, 0xdb99, 0x7ab7, 0xdb68, 0x7aa8, 0xdb38, 0x7a9a,
    0xdb08, 0x7a8c, 0xdad8, 0x7a7d, 0xdaa8, 0x7a6e, 0xda78, 0x7a60, 0xda48,
    0x7a51, 0xda18, 0x7a42, 0xd9e8, 0x7a33, 0xd9b8, 0x7a24, 0xd988, 0x7a15,
    0xd958, 0x7a06, 0xd928, 0x79f7, 0xd8f8, 0x79e7, 0xd8c8, 0x79d8, 0xd898,
    0x79c9, 0xd869, 0x79b9, 0xd839, 0x79aa, 0xd809, 0x799a, 0xd7d9, 0x798a,
    0xd7aa, 0x797a, 0xd77a, 0x796a, 0xd74a, 0x795b, 0xd71b, 0x794a, 0xd6eb,
    0x793a, 0xd6bb, 0x792a, 0xd68c, 0x791a, 0xd65c, 0x790a, 0xd62d, 0x78f9,
    0xd5fd, 0x78e9, 0xd5ce, 0x78d8, 0xd59e, 0x78c8, 0xd56f, 0x78b7, 0xd53f,
    0x78a6, 0xd510, 0x7895, 0xd4e1, 0x7885, 0xd4b1, 0x7874, 0xd482, 0x7863,
    0xd453, 0x7851, 0xd424, 0x7840, 0xd3f4, 0x782f, 0xd3c5, 0x781e, 0xd396,
    0x780c, 0xd367, 0x77fb, 0xd338, 0x77e9, 0xd309, 0x77d8, 0xd2da, 0x77c6,
    0xd2ab, 0x77b4, 0xd27c, 0x77a2, 0xd24d, 0x7790, 0xd21e, 0x777e, 0xd1ef,
    0x776c, 0xd1c0, 0x775a, 0xd191, 0x7748, 0xd162, 0x7736, 0xd134, 0x7723,
    0xd105, 0x7711, 0xd0d6, 0x76fe, 0xd0a7, 0x76ec, 0xd079, 0x76d9, 0xd04a,
    0x76c7, 0xd01b, 0x76b4, 0xcfed, 0x76a1, 0xcfbe, 0x768e, 0xcf90, 0x767b,
    0xcf61, 0x7668, 0xcf33, 0x7655, 0xcf04, 0x7642, 0xced6, 0x762e, 0xcea7,
    0x761b, 0xce79, 0x7608, 0xce4b, 0x75f4, 0xce1c, 0x75e1, 0xcdee, 0x75cd,
    0xcdc0, 0x75b9, 0xcd92, 0x75a6, 0xcd63, 0x7592, 0xcd35, 0x757e, 0xcd07,
    0x756a, 0xccd9, 0x7556, 0xccab, 0x7542, 0xcc7d, 0x752d, 0xcc4f, 0x7519,
    0xcc21, 0x7505, 0xcbf3, 0x74f0, 0xcbc5, 0x74dc, 0xcb97, 0x74c7, 0xcb69,
    0x74b3, 0xcb3c, 0x749e, 0xcb0e, 0x7489, 0xcae0, 0x7475, 0xcab2, 0x7460,
    0xca85, 0x744b, 0xca57, 0x7436, 0xca29, 0x7421, 0xc9fc, 0x740b, 0xc9ce,
    0x73f6, 0xc9a1, 0x73e1, 0xc973, 0x73cb, 0xc946, 0x73b6, 0xc918, 0x73a0,
    0xc8eb, 0x738b, 0xc8be, 0x7375, 0xc890, 0x735f, 0xc863, 0x734a, 0xc836,
    0x7334, 0xc809, 0x731e, 0xc7db, 0x7308, 0xc7ae, 0x72f2, 0xc781, 0x72dc,
    0xc754, 0x72c5, 0xc727, 0x72af, 0xc6fa, 0x7299, 0xc6cd, 0x7282, 0xc6a0,
    0x726c, 0xc673, 0x7255, 0xc646, 0x723f, 0xc619, 0x7228, 0xc5ed, 0x7211,
    0xc5c0, 0x71fa, 0xc593, 0x71e3, 0xc566, 0x71cc, 0xc53a, 0x71b5, 0xc50d,
    0x719e, 0xc4e0, 0x7187, 0xc4b4, 0x7170, 0xc487, 0x7158, 0xc45b, 0x7141,
    0xc42e, 0x712a, 0xc402, 0x7112, 0xc3d6, 0x70fa, 0xc3a9, 0x70e3, 0xc37d,
    0x70cb, 0xc351, 0x70b3, 0xc324, 0x709b, 0xc2f8, 0x7083, 0xc2cc, 0x706b,
    0xc2a0, 0x7053, 0xc274, 0x703b, 0xc248, 0x7023, 0xc21c, 0x700b, 0xc1f0,
    0x6ff2, 0xc1c4, 0x6fda, 0xc198, 0x6fc2, 0xc16c, 0x6fa9, 0xc140, 0x6f90,
    0xc114, 0x6f78, 0xc0e9, 0x6f5f, 0xc0bd, 0x6f46, 0xc091, 0x6f2d, 0xc066,
    0x6f14, 0xc03a, 0x6efb, 0xc00f, 0x6ee2, 0xbfe3, 0x6ec9, 0xbfb8, 0x6eb0,
    0xbf8c, 0x6e97, 0xbf61, 0x6e7d, 0xbf35, 0x6e64, 0xbf0a, 0x6e4a, 0xbedf,
    0x6e31, 0xbeb3, 0x6e17, 0xbe88, 0x6dfe, 0xbe5d, 0x6de4, 0xbe32, 0x6dca,
    0xbe07, 0x6db0, 0xbddc, 0x6d96, 0xbdb1, 0x6d7c, 0xbd86, 0x6d62, 0xbd5b,
    0x6d48, 0xbd30, 0x6d2e, 0xbd05, 0x6d14, 0xbcda, 0x6cf9, 0xbcaf, 0x6cdf,
    0xbc85, 0x6cc4, 0xbc5a, 0x6caa, 0xbc2f, 0x6c8f, 0xbc05, 0x6c75, 0xbbda,
    0x6c5a, 0xbbb0, 0x6c3f, 0xbb85, 0x6c24, 0xbb5b, 0x6c09, 0xbb30, 0x6bee,
    0xbb06, 0x6bd3, 0xbadc, 0x6bb8, 0xbab1, 0x6b9d, 0xba87, 0x6b82, 0xba5d,
    0x6b66, 0xba33, 0x6b4b, 0xba09, 0x6b30, 0xb9df, 0x6b14, 0xb9b5, 0x6af8,
    0xb98b, 0x6add, 0xb961, 0x6ac1, 0xb937, 0x6aa5, 0xb90d, 0x6a89, 0xb8e3,
    0x6a6e, 0xb8b9, 0x6a52, 0xb890, 0x6a36, 0xb866, 0x6a1a, 0xb83c, 0x69fd,
    0xb813, 0x69e1, 0xb7e9, 0x69c5, 0xb7c0, 0x69a9, 0xb796, 0x698c, 0xb76d,
    0x6970, 0xb743, 0x6953, 0xb71a, 0x6937, 0xb6f1, 0x691a, 0xb6c7, 0x68fd,
    0xb69e, 0x68e0, 0xb675, 0x68c4, 0xb64c, 0x68a7, 0xb623, 0x688a, 0xb5fa,
    0x686d, 0xb5d1, 0x6850, 0xb5a8, 0x6832, 0xb57f, 0x6815, 0xb556, 0x67f8,
    0xb52d, 0x67da, 0xb505, 0x67bd, 0xb4dc, 0x67a0, 0xb4b3, 0x6782, 0xb48b,
    0x6764, 0xb462, 0x6747, 0xb439, 0x6729, 0xb411, 0x670b, 0xb3e9, 0x66ed,
    0xb3c0, 0x66d0, 0xb398, 0x66b2, 0xb36f, 0x6693, 0xb347, 0x6675, 0xb31f,
    0x6657, 0xb2f7, 0x6639, 0xb2cf, 0x661b, 0xb2a7, 0x65fc, 0xb27f, 0x65de,
    0xb257, 0x65c0, 0xb22f, 0x65a1, 0xb207, 0x6582, 0xb1df, 0x6564, 0xb1b7,
    0x6545, 0xb18f, 0x6526, 0xb168, 0x6507, 0xb140, 0x64e9, 0xb118, 0x64ca,
    0xb0f1, 0x64ab, 0xb0c9, 0x648b, 0xb0a2, 0x646c, 0xb07b, 0x644d, 0xb053,
    0x642e, 0xb02c, 0x640f, 0xb005, 0x63ef, 0xafdd, 0x63d0, 0xafb6, 0x63b0,
    0xaf8f, 0x6391, 0xaf68, 0x6371, 0xaf41, 0x6351, 0xaf1a, 0x6332, 0xaef3,
    0x6312, 0xaecc, 0x62f2, 0xaea5, 0x62d2, 0xae7f, 0x62b2, 0xae58, 0x6292,
    0xae31, 0x6272, 0xae0b, 0x6252, 0xade4, 0x6232, 0xadbd, 0x6211, 0xad97,
    0x61f1, 0xad70, 0x61d1, 0xad4a, 0x61b0, 0xad24, 0x6190, 0xacfd, 0x616f,
    0xacd7, 0x614e, 0xacb1, 0x612e, 0xac8b, 0x610d, 0xac65, 0x60ec, 0xac3f,
    0x60cb, 0xac19, 0x60aa, 0xabf3, 0x6089, 0xabcd, 0x6068, 0xaba7, 0x6047,
    0xab81, 0x6026, 0xab5c, 0x6005, 0xab36, 0x5fe4, 0xab10, 0x5fc2, 0xaaeb,
    0x5fa1, 0xaac5, 0x5f80, 0xaaa0, 0x5f5e, 0xaa7a, 0x5f3c, 0xaa55, 0x5f1b,
    0xaa30, 0x5ef9, 0xaa0a, 0x5ed7, 0xa9e5, 0x5eb6, 0xa9c0, 0x5e94, 0xa99b,
    0x5e72, 0xa976, 0x5e50, 0xa951, 0x5e2e, 0xa92c, 0x5e0c, 0xa907, 0x5dea,
    0xa8e2, 0x5dc8, 0xa8bd, 0x5da5, 0xa899, 0x5d83, 0xa874, 0x5d61, 0xa84f,
    0x5d3e, 0xa82b, 0x5d1c, 0xa806, 0x5cf9, 0xa7e2, 0x5cd7, 0xa7bd, 0x5cb4,
    0xa799, 0x5c91, 0xa774, 0x5c6f, 0xa750, 0x5c4c, 0xa72c, 0x5c29, 0xa708,
    0x5c06, 0xa6e4, 0x5be3, 0xa6c0, 0x5bc0, 0xa69c, 0x5b9d, 0xa678, 0x5b7a,
    0xa654, 0x5b57, 0xa630, 0x5b34, 0xa60c, 0x5b10, 0xa5e8, 0x5aed, 0xa5c5,
    0x5ac9, 0xa5a1, 0x5aa6, 0xa57e, 0x5a82, 0xa55a, 0x5a5f, 0xa537, 0x5a3b,
    0xa513, 0x5a18, 0xa4f0, 0x59f4, 0xa4cc, 0x59d0, 0xa4a9, 0x59ac, 0xa486,
    0x5988, 0xa463, 0x5964, 0xa440, 0x5940, 0xa41d, 0x591c, 0xa3fa, 0x58f8,
    0xa3d7, 0x58d4, 0xa3b4, 0x58b0, 0xa391, 0x588c, 0xa36f, 0x5867, 0xa34c,
    0x5843, 0xa329, 0x581e, 0xa307, 0x57fa, 0xa2e4, 0x57d5, 0xa2c2, 0x57b1,
    0xa29f, 0x578c, 0xa27d, 0x5767, 0xa25b, 0x5743, 0xa238, 0x571e, 0xa216,
    0x56f9, 0xa1f4, 0x56d4, 0xa1d2, 0x56af, 0xa1b0, 0x568a, 0xa18e, 0x5665,
    0xa16c, 0x5640, 0xa14a, 0x561b, 0xa129, 0x55f6, 0xa107, 0x55d0, 0xa0e5,
    0x55ab, 0xa0c4, 0x5586, 0xa0a2, 0x5560, 0xa080, 0x553b, 0xa05f, 0x5515,
    0xa03e, 0x54f0, 0xa01c, 0x54ca, 0x9ffb, 0x54a4, 0x9fda, 0x547f, 0x9fb9,
    0x5459, 0x9f98, 0x5433, 0x9f77, 0x540d, 0x9f56, 0x53e7, 0x9f35, 0x53c1,
    0x9f14, 0x539b, 0x9ef3, 0x5375, 0x9ed2, 0x534f, 0x9eb2, 0x5329, 0x9e91,
    0x5303, 0x9e70, 0x52dc, 0x9e50, 0x52b6, 0x9e2f, 0x5290, 0x9e0f, 0x5269,
    0x9def, 0x5243, 0x9dce, 0x521c, 0x9dae, 0x51f5, 0x9d8e, 0x51cf, 0x9d6e,
    0x51a8, 0x9d4e, 0x5181, 0x9d2e, 0x515b, 0x9d0e, 0x5134, 0x9cee, 0x510d,
    0x9cce, 0x50e6, 0x9caf, 0x50bf, 0x9c8f, 0x5098, 0x9c6f, 0x5071, 0x9c50,
    0x504a, 0x9c30, 0x5023, 0x9c11, 0x4ffb, 0x9bf1, 0x4fd4, 0x9bd2, 0x4fad,
    0x9bb3, 0x4f85, 0x9b94, 0x4f5e, 0x9b75, 0x4f37, 0x9b55, 0x4f0f, 0x9b36,
    0x4ee8, 0x9b17, 0x4ec0, 0x9af9, 0x4e98, 0x9ada, 0x4e71, 0x9abb, 0x4e49,
    0x9a9c, 0x4e21, 0x9a7e, 0x4df9, 0x9a5f, 0x4dd1, 0x9a40, 0x4da9, 0x9a22,
    0x4d81, 0x9a04, 0x4d59, 0x99e5, 0x4d31, 0x99c7, 0x4d09, 0x99a9, 0x4ce1,
    0x998b, 0x4cb9, 0x996d, 0x4c91, 0x994e, 0x4c68, 0x9930, 0x4c40, 0x9913,
    0x4c17, 0x98f5, 0x4bef, 0x98d7, 0x4bc7, 0x98b9, 0x4b9e, 0x989c, 0x4b75,
    0x987e, 0x4b4d, 0x9860, 0x4b24, 0x9843, 0x4afb, 0x9826, 0x4ad3, 0x9808,
    0x4aaa, 0x97eb, 0x4a81, 0x97ce, 0x4a58, 0x97b0, 0x4a2f, 0x9793, 0x4a06,
    0x9776, 0x49dd, 0x9759, 0x49b4, 0x973c, 0x498b, 0x9720, 0x4962, 0x9703,
    0x4939, 0x96e6, 0x490f, 0x96c9, 0x48e6, 0x96ad, 0x48bd, 0x9690, 0x4893,
    0x9674, 0x486a, 0x9657, 0x4840, 0x963b, 0x4817, 0x961f, 0x47ed, 0x9603,
    0x47c4, 0x95e6, 0x479a, 0x95ca, 0x4770, 0x95ae, 0x4747, 0x9592, 0x471d,
    0x9577, 0x46f3, 0x955b, 0x46c9, 0x953f, 0x469f, 0x9523, 0x4675, 0x9508,
    0x464b, 0x94ec, 0x4621, 0x94d0, 0x45f7, 0x94b5, 0x45cd, 0x949a, 0x45a3,
    0x947e, 0x4579, 0x9463, 0x454f, 0x9448, 0x4524, 0x942d, 0x44fa, 0x9412,
    0x44d0, 0x93f7, 0x44a5, 0x93dc, 0x447b, 0x93c1, 0x4450, 0x93a6, 0x4426,
    0x938b, 0x43fb, 0x9371, 0x43d1, 0x9356, 0x43a6, 0x933c, 0x437b, 0x9321,
    0x4351, 0x9307, 0x4326, 0x92ec, 0x42fb, 0x92d2, 0x42d0, 0x92b8, 0x42a5,
    0x929e, 0x427a, 0x9284, 0x424f, 0x926a, 0x4224, 0x9250, 0x41f9, 0x9236,
    0x41ce, 0x921c, 0x41a3, 0x9202, 0x4178, 0x91e9, 0x414d, 0x91cf, 0x4121,
    0x91b6, 0x40f6, 0x919c, 0x40cb, 0x9183, 0x409f, 0x9169, 0x4074, 0x9150,
    0x4048, 0x9137, 0x401d, 0x911e, 0x3ff1, 0x9105, 0x3fc6, 0x90ec, 0x3f9a,
    0x90d3, 0x3f6f, 0x90ba, 0x3f43, 0x90a1, 0x3f17, 0x9088, 0x3eec, 0x9070,
    0x3ec0, 0x9057, 0x3e94, 0x903e, 0x3e68, 0x9026, 0x3e3c, 0x900e, 0x3e10,
    0x8ff5, 0x3de4, 0x8fdd, 0x3db8, 0x8fc5, 0x3d8c, 0x8fad, 0x3d60, 0x8f95,
    0x3d34, 0x8f7d, 0x3d08, 0x8f65, 0x3cdc, 0x8f4d, 0x3caf, 0x8f35, 0x3c83,
    0x8f1d, 0x3c57, 0x8f06, 0x3c2a, 0x8eee, 0x3bfe, 0x8ed6, 0x3bd2, 0x8ebf,
    0x3ba5, 0x8ea8, 0x3b79, 0x8e90, 0x3b4c, 0x8e79, 0x3b20, 0x8e62, 0x3af3,
    0x8e4b, 0x3ac6, 0x8e34, 0x3a9a, 0x8e1d, 0x3a6d, 0x8e06, 0x3a40, 0x8def,
    0x3a13, 0x8dd8, 0x39e7, 0x8dc1, 0x39ba, 0x8dab, 0x398d, 0x8d94, 0x3960,
    0x8d7e, 0x3933, 0x8d67, 0x3906, 0x8d51, 0x38d9, 0x8d3b, 0x38ac, 0x8d24,
    0x387f, 0x8d0e, 0x3852, 0x8cf8, 0x3825, 0x8ce2, 0x37f7, 0x8ccc, 0x37ca,
    0x8cb6, 0x379d, 0x8ca1, 0x3770, 0x8c8b, 0x3742, 0x8c75, 0x3715, 0x8c60,
    0x36e8, 0x8c4a, 0x36ba, 0x8c35, 0x368d, 0x8c1f, 0x365f, 0x8c0a, 0x3632,
    0x8bf5, 0x3604, 0x8bdf, 0x35d7, 0x8bca, 0x35a9, 0x8bb5, 0x357b, 0x8ba0,
    0x354e, 0x8b8b, 0x3520, 0x8b77, 0x34f2, 0x8b62, 0x34c4, 0x8b4d, 0x3497,
    0x8b39, 0x3469, 0x8b24, 0x343b, 0x8b10, 0x340d, 0x8afb, 0x33df, 0x8ae7,
    0x33b1, 0x8ad3, 0x3383, 0x8abe, 0x3355, 0x8aaa, 0x3327, 0x8a96, 0x32f9,
    0x8a82, 0x32cb, 0x8a6e, 0x329d, 0x8a5a, 0x326e, 0x8a47, 0x3240, 0x8a33,
    0x3212, 0x8a1f, 0x31e4, 0x8a0c, 0x31b5, 0x89f8, 0x3187, 0x89e5, 0x3159,
    0x89d2, 0x312a, 0x89be, 0x30fc, 0x89ab, 0x30cd, 0x8998, 0x309f, 0x8985,
    0x3070, 0x8972, 0x3042, 0x895f, 0x3013, 0x894c, 0x2fe5, 0x8939, 0x2fb6,
    0x8927, 0x2f87, 0x8914, 0x2f59, 0x8902, 0x2f2a, 0x88ef, 0x2efb, 0x88dd,
    0x2ecc, 0x88ca, 0x2e9e, 0x88b8, 0x2e6f, 0x88a6, 0x2e40, 0x8894, 0x2e11,
    0x8882, 0x2de2, 0x8870, 0x2db3, 0x885e, 0x2d84, 0x884c, 0x2d55, 0x883a,
    0x2d26, 0x8828, 0x2cf7, 0x8817, 0x2cc8, 0x8805, 0x2c99, 0x87f4, 0x2c6a,
    0x87e2, 0x2c3b, 0x87d1, 0x2c0c, 0x87c0, 0x2bdc, 0x87af, 0x2bad, 0x879d,
    0x2b7e, 0x878c, 0x2b4f, 0x877b, 0x2b1f, 0x876b, 0x2af0, 0x875a, 0x2ac1,
    0x8749, 0x2a91, 0x8738, 0x2a62, 0x8728, 0x2a32, 0x8717, 0x2a03, 0x8707,
    0x29d3, 0x86f6, 0x29a4, 0x86e6, 0x2974, 0x86d6, 0x2945, 0x86c6, 0x2915,
    0x86b6, 0x28e5, 0x86a5, 0x28b6, 0x8696, 0x2886, 0x8686, 0x2856, 0x8676,
    0x2827, 0x8666, 0x27f7, 0x8656, 0x27c7, 0x8647, 0x2797, 0x8637, 0x2768,
    0x8628, 0x2738, 0x8619, 0x2708, 0x8609, 0x26d8, 0x85fa, 0x26a8, 0x85eb,
    0x2678, 0x85dc, 0x2648, 0x85cd, 0x2618, 0x85be, 0x25e8, 0x85af, 0x25b8,
    0x85a0, 0x2588, 0x8592, 0x2558, 0x8583, 0x2528, 0x8574, 0x24f8, 0x8566,
    0x24c8, 0x8558, 0x2498, 0x8549, 0x2467, 0x853b, 0x2437, 0x852d, 0x2407,
    0x851f, 0x23d7, 0x8511, 0x23a7, 0x8503, 0x2376, 0x84f5, 0x2346, 0x84e7,
    0x2316, 0x84d9, 0x22e5, 0x84cc, 0x22b5, 0x84be, 0x2284, 0x84b0, 0x2254,
    0x84a3, 0x2224, 0x8496, 0x21f3, 0x8488, 0x21c3, 0x847b, 0x2192, 0x846e,
    0x2162, 0x8461, 0x2131, 0x8454, 0x2101, 0x8447, 0x20d0, 0x843a, 0x209f,
    0x842d, 0x206f, 0x8421, 0x203e, 0x8414, 0x200e, 0x8407, 0x1fdd, 0x83fb,
    0x1fac, 0x83ef, 0x1f7b, 0x83e2, 0x1f4b, 0x83d6, 0x1f1a, 0x83ca, 0x1ee9,
    0x83be, 0x1eb8, 0x83b2, 0x1e88, 0x83a6, 0x1e57, 0x839a, 0x1e26, 0x838e,
    0x1df5, 0x8382, 0x1dc4, 0x8377, 0x1d93, 0x836b, 0x1d62, 0x8360, 0x1d31,
    0x8354, 0x1d01, 0x8349, 0x1cd0, 0x833e, 0x1c9f, 0x8332, 0x1c6e, 0x8327,
    0x1c3d, 0x831c, 0x1c0c, 0x8311, 0x1bda, 0x8306, 0x1ba9, 0x82fb, 0x1b78,
    0x82f1, 0x1b47, 0x82e6, 0x1b16, 0x82db, 0x1ae5, 0x82d1, 0x1ab4, 0x82c6,
    0x1a83, 0x82bc, 0x1a51, 0x82b2, 0x1a20, 0x82a8, 0x19ef, 0x829d, 0x19be,
    0x8293, 0x198d, 0x8289, 0x195b, 0x827f, 0x192a, 0x8276, 0x18f9, 0x826c,
    0x18c7, 0x8262, 0x1896, 0x8259, 0x1865, 0x824f, 0x1833, 0x8246, 0x1802,
    0x823c, 0x17d1, 0x8233, 0x179f, 0x822a, 0x176e, 0x8220, 0x173c, 0x8217,
    0x170b, 0x820e, 0x16da, 0x8205, 0x16a8, 0x81fd, 0x1677, 0x81f4, 0x1645,
    0x81eb, 0x1614, 0x81e2, 0x15e2, 0x81da, 0x15b1, 0x81d1, 0x157f, 0x81c9,
    0x154d, 0x81c1, 0x151c, 0x81b8, 0x14ea, 0x81b0, 0x14b9, 0x81a8, 0x1487,
    0x81a0, 0x1455, 0x8198, 0x1424, 0x8190, 0x13f2, 0x8188, 0x13c1, 0x8181,
    0x138f, 0x8179, 0x135d, 0x8172, 0x132b, 0x816a, 0x12fa, 0x8163, 0x12c8,
    0x815b, 0x1296, 0x8154, 0x1265, 0x814d, 0x1233, 0x8146, 0x1201, 0x813f,
    0x11cf, 0x8138, 0x119e, 0x8131, 0x116c, 0x812a, 0x113a, 0x8123, 0x1108,
    0x811d, 0x10d6, 0x8116, 0x10a4, 0x8110, 0x1073, 0x8109, 0x1041, 0x8103,
    0x100f, 0x80fd, 0xfdd,  0x80f6, 0xfab,  0x80f0, 0xf79,  0x80ea, 0xf47,
    0x80e4, 0xf15,  0x80de, 0xee4,  0x80d9, 0xeb2,  0x80d3, 0xe80,  0x80cd,
    0xe4e,  0x80c8, 0xe1c,  0x80c2, 0xdea,  0x80bd, 0xdb8,  0x80b7, 0xd86,
    0x80b2, 0xd54,  0x80ad, 0xd22,  0x80a8, 0xcf0,  0x80a3, 0xcbe,  0x809e,
    0xc8c,  0x8099, 0xc5a,  0x8094, 0xc28,  0x808f, 0xbf6,  0x808b, 0xbc4,
    0x8086, 0xb92,  0x8082, 0xb60,  0x807d, 0xb2d,  0x8079, 0xafb,  0x8075,
    0xac9,  0x8070, 0xa97,  0x806c, 0xa65,  0x8068, 0xa33,  0x8064, 0xa01,
    0x8060, 0x9cf,  0x805d, 0x99d,  0x8059, 0x96b,  0x8055, 0x938,  0x8052,
    0x906,  0x804e, 0x8d4,  0x804b, 0x8a2,  0x8047, 0x870,  0x8044, 0x83e,
    0x8041, 0x80c,  0x803e, 0x7d9,  0x803b, 0x7a7,  0x8038, 0x775,  0x8035,
    0x743,  0x8032, 0x711,  0x802f, 0x6de,  0x802d, 0x6ac,  0x802a, 0x67a,
    0x8027, 0x648,  0x8025, 0x616,  0x8023, 0x5e3,  0x8020, 0x5b1,  0x801e,
    0x57f,  0x801c, 0x54d,  0x801a, 0x51b,  0x8018, 0x4e8,  0x8016, 0x4b6,
    0x8014, 0x484,  0x8013, 0x452,  0x8011, 0x41f,  0x800f, 0x3ed,  0x800e,
    0x3bb,  0x800c, 0x389,  0x800b, 0x356,  0x800a, 0x324,  0x8009, 0x2f2,
    0x8008, 0x2c0,  0x8007, 0x28d,  0x8006, 0x25b,  0x8005, 0x229,  0x8004,
    0x1f7,  0x8003, 0x1c4,  0x8002, 0x192,  0x8002, 0x160,  0x8001, 0x12e,
    0x8001, 0xfb,   0x8001, 0xc9,   0x8000, 0x97,   0x8000, 0x65,   0x8000,
    0x32,   0x8000, 0x0,    0x8000, 0xffce, 0x8000, 0xff9b, 0x8000, 0xff69,
    0x8001, 0xff37, 0x8001, 0xff05, 0x8001, 0xfed2, 0x8002, 0xfea0, 0x8002,
    0xfe6e, 0x8003, 0xfe3c, 0x8004, 0xfe09, 0x8005, 0xfdd7, 0x8006, 0xfda5,
    0x8007, 0xfd73, 0x8008, 0xfd40, 0x8009, 0xfd0e, 0x800a, 0xfcdc, 0x800b,
    0xfcaa, 0x800c, 0xfc77, 0x800e, 0xfc45, 0x800f, 0xfc13, 0x8011, 0xfbe1,
    0x8013, 0xfbae, 0x8014, 0xfb7c, 0x8016, 0xfb4a, 0x8018, 0xfb18, 0x801a,
    0xfae5, 0x801c, 0xfab3, 0x801e, 0xfa81, 0x8020, 0xfa4f, 0x8023, 0xfa1d,
    0x8025, 0xf9ea, 0x8027, 0xf9b8, 0x802a, 0xf986, 0x802d, 0xf954, 0x802f,
    0xf922, 0x8032, 0xf8ef, 0x8035, 0xf8bd, 0x8038, 0xf88b, 0x803b, 0xf859,
    0x803e, 0xf827, 0x8041, 0xf7f4, 0x8044, 0xf7c2, 0x8047, 0xf790, 0x804b,
    0xf75e, 0x804e, 0xf72c, 0x8052, 0xf6fa, 0x8055, 0xf6c8, 0x8059, 0xf695,
    0x805d, 0xf663, 0x8060, 0xf631, 0x8064, 0xf5ff, 0x8068, 0xf5cd, 0x806c,
    0xf59b, 0x8070, 0xf569, 0x8075, 0xf537, 0x8079, 0xf505, 0x807d, 0xf4d3,
    0x8082, 0xf4a0, 0x8086, 0xf46e, 0x808b, 0xf43c, 0x808f, 0xf40a, 0x8094,
    0xf3d8, 0x8099, 0xf3a6, 0x809e, 0xf374, 0x80a3, 0xf342, 0x80a8, 0xf310,
    0x80ad, 0xf2de, 0x80b2, 0xf2ac, 0x80b7, 0xf27a, 0x80bd, 0xf248, 0x80c2,
    0xf216, 0x80c8, 0xf1e4, 0x80cd, 0xf1b2, 0x80d3, 0xf180, 0x80d9, 0xf14e,
    0x80de, 0xf11c, 0x80e4, 0xf0eb, 0x80ea, 0xf0b9, 0x80f0, 0xf087, 0x80f6,
    0xf055, 0x80fd, 0xf023, 0x8103, 0xeff1, 0x8109, 0xefbf, 0x8110, 0xef8d,
    0x8116, 0xef5c, 0x811d, 0xef2a, 0x8123, 0xeef8, 0x812a, 0xeec6, 0x8131,
    0xee94, 0x8138, 0xee62, 0x813f, 0xee31, 0x8146, 0xedff, 0x814d, 0xedcd,
    0x8154, 0xed9b, 0x815b, 0xed6a, 0x8163, 0xed38, 0x816a, 0xed06, 0x8172,
    0xecd5, 0x8179, 0xeca3, 0x8181, 0xec71, 0x8188, 0xec3f, 0x8190, 0xec0e,
    0x8198, 0xebdc, 0x81a0, 0xebab, 0x81a8, 0xeb79, 0x81b0, 0xeb47, 0x81b8,
    0xeb16, 0x81c1, 0xeae4, 0x81c9, 0xeab3, 0x81d1, 0xea81, 0x81da, 0xea4f,
    0x81e2, 0xea1e, 0x81eb, 0xe9ec, 0x81f4, 0xe9bb, 0x81fd, 0xe989, 0x8205,
    0xe958, 0x820e, 0xe926, 0x8217, 0xe8f5, 0x8220, 0xe8c4, 0x822a, 0xe892,
    0x8233, 0xe861, 0x823c, 0xe82f, 0x8246, 0xe7fe, 0x824f, 0xe7cd, 0x8259,
    0xe79b, 0x8262, 0xe76a, 0x826c, 0xe739, 0x8276, 0xe707, 0x827f, 0xe6d6,
    0x8289, 0xe6a5, 0x8293, 0xe673, 0x829d, 0xe642, 0x82a8, 0xe611, 0x82b2,
    0xe5e0, 0x82bc, 0xe5af, 0x82c6, 0xe57d, 0x82d1, 0xe54c, 0x82db, 0xe51b,
    0x82e6, 0xe4ea, 0x82f1, 0xe4b9, 0x82fb, 0xe488, 0x8306, 0xe457, 0x8311,
    0xe426, 0x831c, 0xe3f4, 0x8327, 0xe3c3, 0x8332, 0xe392, 0x833e, 0xe361,
    0x8349, 0xe330, 0x8354, 0xe2ff, 0x8360, 0xe2cf, 0x836b, 0xe29e, 0x8377,
    0xe26d, 0x8382, 0xe23c, 0x838e, 0xe20b, 0x839a, 0xe1da, 0x83a6, 0xe1a9,
    0x83b2, 0xe178, 0x83be, 0xe148, 0x83ca, 0xe117, 0x83d6, 0xe0e6, 0x83e2,
    0xe0b5, 0x83ef, 0xe085, 0x83fb, 0xe054, 0x8407, 0xe023, 0x8414, 0xdff2,
    0x8421, 0xdfc2, 0x842d, 0xdf91, 0x843a, 0xdf61, 0x8447, 0xdf30, 0x8454,
    0xdeff, 0x8461, 0xdecf, 0x846e, 0xde9e, 0x847b, 0xde6e, 0x8488, 0xde3d,
    0x8496, 0xde0d, 0x84a3, 0xdddc, 0x84b0, 0xddac, 0x84be, 0xdd7c, 0x84cc,
    0xdd4b, 0x84d9, 0xdd1b, 0x84e7, 0xdcea, 0x84f5, 0xdcba, 0x8503, 0xdc8a,
    0x8511, 0xdc59, 0x851f, 0xdc29, 0x852d, 0xdbf9, 0x853b, 0xdbc9, 0x8549,
    0xdb99, 0x8558, 0xdb68, 0x8566, 0xdb38, 0x8574, 0xdb08, 0x8583, 0xdad8,
    0x8592, 0xdaa8, 0x85a0, 0xda78, 0x85af, 0xda48, 0x85be, 0xda18, 0x85cd,
    0xd9e8, 0x85dc, 0xd9b8, 0x85eb, 0xd988, 0x85fa, 0xd958, 0x8609, 0xd928,
    0x8619, 0xd8f8, 0x8628, 0xd8c8, 0x8637, 0xd898, 0x8647, 0xd869, 0x8656,
    0xd839, 0x8666, 0xd809, 0x8676, 0xd7d9, 0x8686, 0xd7aa, 0x8696, 0xd77a,
    0x86a5, 0xd74a, 0x86b6, 0xd71b, 0x86c6, 0xd6eb, 0x86d6, 0xd6bb, 0x86e6,
    0xd68c, 0x86f6, 0xd65c, 0x8707, 0xd62d, 0x8717, 0xd5fd, 0x8728, 0xd5ce,
    0x8738, 0xd59e, 0x8749, 0xd56f, 0x875a, 0xd53f, 0x876b, 0xd510, 0x877b,
    0xd4e1, 0x878c, 0xd4b1, 0x879d, 0xd482, 0x87af, 0xd453, 0x87c0, 0xd424,
    0x87d1, 0xd3f4, 0x87e2, 0xd3c5, 0x87f4, 0xd396, 0x8805, 0xd367, 0x8817,
    0xd338, 0x8828, 0xd309, 0x883a, 0xd2da, 0x884c, 0xd2ab, 0x885e, 0xd27c,
    0x8870, 0xd24d, 0x8882, 0xd21e, 0x8894, 0xd1ef, 0x88a6, 0xd1c0, 0x88b8,
    0xd191, 0x88ca, 0xd162, 0x88dd, 0xd134, 0x88ef, 0xd105, 0x8902, 0xd0d6,
    0x8914, 0xd0a7, 0x8927, 0xd079, 0x8939, 0xd04a, 0x894c, 0xd01b, 0x895f,
    0xcfed, 0x8972, 0xcfbe, 0x8985, 0xcf90, 0x8998, 0xcf61, 0x89ab, 0xcf33,
    0x89be, 0xcf04, 0x89d2, 0xced6, 0x89e5, 0xcea7, 0x89f8, 0xce79, 0x8a0c,
    0xce4b, 0x8a1f, 0xce1c, 0x8a33, 0xcdee, 0x8a47, 0xcdc0, 0x8a5a, 0xcd92,
    0x8a6e, 0xcd63, 0x8a82, 0xcd35, 0x8a96, 0xcd07, 0x8aaa, 0xccd9, 0x8abe,
    0xccab, 0x8ad3, 0xcc7d, 0x8ae7, 0xcc4f, 0x8afb, 0xcc21, 0x8b10, 0xcbf3,
    0x8b24, 0xcbc5, 0x8b39, 0xcb97, 0x8b4d, 0xcb69, 0x8b62, 0xcb3c, 0x8b77,
    0xcb0e, 0x8b8b, 0xcae0, 0x8ba0, 0xcab2, 0x8bb5, 0xca85, 0x8bca, 0xca57,
    0x8bdf, 0xca29, 0x8bf5, 0xc9fc, 0x8c0a, 0xc9ce, 0x8c1f, 0xc9a1, 0x8c35,
    0xc973, 0x8c4a, 0xc946, 0x8c60, 0xc918, 0x8c75, 0xc8eb, 0x8c8b, 0xc8be,
    0x8ca1, 0xc890, 0x8cb6, 0xc863, 0x8ccc, 0xc836, 0x8ce2, 0xc809, 0x8cf8,
    0xc7db, 0x8d0e, 0xc7ae, 0x8d24, 0xc781, 0x8d3b, 0xc754, 0x8d51, 0xc727,
    0x8d67, 0xc6fa, 0x8d7e, 0xc6cd, 0x8d94, 0xc6a0, 0x8dab, 0xc673, 0x8dc1,
    0xc646, 0x8dd8, 0xc619, 0x8def, 0xc5ed, 0x8e06, 0xc5c0, 0x8e1d, 0xc593,
    0x8e34, 0xc566, 0x8e4b, 0xc53a, 0x8e62, 0xc50d, 0x8e79, 0xc4e0, 0x8e90,
    0xc4b4, 0x8ea8, 0xc487, 0x8ebf, 0xc45b, 0x8ed6, 0xc42e, 0x8eee, 0xc402,
    0x8f06, 0xc3d6, 0x8f1d, 0xc3a9, 0x8f35, 0xc37d, 0x8f4d, 0xc351, 0x8f65,
    0xc324, 0x8f7d, 0xc2f8, 0x8f95, 0xc2cc, 0x8fad, 0xc2a0, 0x8fc5, 0xc274,
    0x8fdd, 0xc248, 0x8ff5, 0xc21c, 0x900e, 0xc1f0, 0x9026, 0xc1c4, 0x903e,
    0xc198, 0x9057, 0xc16c, 0x9070, 0xc140, 0x9088, 0xc114, 0x90a1, 0xc0e9,
    0x90ba, 0xc0bd, 0x90d3, 0xc091, 0x90ec, 0xc066, 0x9105, 0xc03a, 0x911e,
    0xc00f, 0x9137, 0xbfe3, 0x9150, 0xbfb8, 0x9169, 0xbf8c, 0x9183, 0xbf61,
    0x919c, 0xbf35, 0x91b6, 0xbf0a, 0x91cf, 0xbedf, 0x91e9, 0xbeb3, 0x9202,
    0xbe88, 0x921c, 0xbe5d, 0x9236, 0xbe32, 0x9250, 0xbe07, 0x926a, 0xbddc,
    0x9284, 0xbdb1, 0x929e, 0xbd86, 0x92b8, 0xbd5b, 0x92d2, 0xbd30, 0x92ec,
    0xbd05, 0x9307, 0xbcda, 0x9321, 0xbcaf, 0x933c, 0xbc85, 0x9356, 0xbc5a,
    0x9371, 0xbc2f, 0x938b, 0xbc05, 0x93a6, 0xbbda, 0x93c1, 0xbbb0, 0x93dc,
    0xbb85, 0x93f7, 0xbb5b, 0x9412, 0xbb30, 0x942d, 0xbb06, 0x9448, 0xbadc,
    0x9463, 0xbab1, 0x947e, 0xba87, 0x949a, 0xba5d, 0x94b5, 0xba33, 0x94d0,
    0xba09, 0x94ec, 0xb9df, 0x9508, 0xb9b5, 0x9523, 0xb98b, 0x953f, 0xb961,
    0x955b, 0xb937, 0x9577, 0xb90d, 0x9592, 0xb8e3, 0x95ae, 0xb8b9, 0x95ca,
    0xb890, 0x95e6, 0xb866, 0x9603, 0xb83c, 0x961f, 0xb813, 0x963b, 0xb7e9,
    0x9657, 0xb7c0, 0x9674, 0xb796, 0x9690, 0xb76d, 0x96ad, 0xb743, 0x96c9,
    0xb71a, 0x96e6, 0xb6f1, 0x9703, 0xb6c7, 0x9720, 0xb69e, 0x973c, 0xb675,
    0x9759, 0xb64c, 0x9776, 0xb623, 0x9793, 0xb5fa, 0x97b0, 0xb5d1, 0x97ce,
    0xb5a8, 0x97eb, 0xb57f, 0x9808, 0xb556, 0x9826, 0xb52d, 0x9843, 0xb505,
    0x9860, 0xb4dc, 0x987e, 0xb4b3, 0x989c, 0xb48b, 0x98b9, 0xb462, 0x98d7,
    0xb439, 0x98f5, 0xb411, 0x9913, 0xb3e9, 0x9930, 0xb3c0, 0x994e, 0xb398,
    0x996d, 0xb36f, 0x998b, 0xb347, 0x99a9, 0xb31f, 0x99c7, 0xb2f7, 0x99e5,
    0xb2cf, 0x9a04, 0xb2a7, 0x9a22, 0xb27f, 0x9a40, 0xb257, 0x9a5f, 0xb22f,
    0x9a7e, 0xb207, 0x9a9c, 0xb1df, 0x9abb, 0xb1b7, 0x9ada, 0xb18f, 0x9af9,
    0xb168, 0x9b17, 0xb140, 0x9b36, 0xb118, 0x9b55, 0xb0f1, 0x9b75, 0xb0c9,
    0x9b94, 0xb0a2, 0x9bb3, 0xb07b, 0x9bd2, 0xb053, 0x9bf1, 0xb02c, 0x9c11,
    0xb005, 0x9c30, 0xafdd, 0x9c50, 0xafb6, 0x9c6f, 0xaf8f, 0x9c8f, 0xaf68,
    0x9caf, 0xaf41, 0x9cce, 0xaf1a, 0x9cee, 0xaef3, 0x9d0e, 0xaecc, 0x9d2e,
    0xaea5, 0x9d4e, 0xae7f, 0x9d6e, 0xae58, 0x9d8e, 0xae31, 0x9dae, 0xae0b,
    0x9dce, 0xade4, 0x9def, 0xadbd, 0x9e0f, 0xad97, 0x9e2f, 0xad70, 0x9e50,
    0xad4a, 0x9e70, 0xad24, 0x9e91, 0xacfd, 0x9eb2, 0xacd7, 0x9ed2, 0xacb1,
    0x9ef3, 0xac8b, 0x9f14, 0xac65, 0x9f35, 0xac3f, 0x9f56, 0xac19, 0x9f77,
    0xabf3, 0x9f98, 0xabcd, 0x9fb9, 0xaba7, 0x9fda, 0xab81, 0x9ffb, 0xab5c,
    0xa01c, 0xab36, 0xa03e, 0xab10, 0xa05f, 0xaaeb, 0xa080, 0xaac5, 0xa0a2,
    0xaaa0, 0xa0c4, 0xaa7a, 0xa0e5, 0xaa55, 0xa107, 0xaa30, 0xa129, 0xaa0a,
    0xa14a, 0xa9e5, 0xa16c, 0xa9c0, 0xa18e, 0xa99b, 0xa1b0, 0xa976, 0xa1d2,
    0xa951, 0xa1f4, 0xa92c, 0xa216, 0xa907, 0xa238, 0xa8e2, 0xa25b, 0xa8bd,
    0xa27d, 0xa899, 0xa29f, 0xa874, 0xa2c2, 0xa84f, 0xa2e4, 0xa82b, 0xa307,
    0xa806, 0xa329, 0xa7e2, 0xa34c, 0xa7bd, 0xa36f, 0xa799, 0xa391, 0xa774,
    0xa3b4, 0xa750, 0xa3d7, 0xa72c, 0xa3fa, 0xa708, 0xa41d, 0xa6e4, 0xa440,
    0xa6c0, 0xa463, 0xa69c, 0xa486, 0xa678, 0xa4a9, 0xa654, 0xa4cc, 0xa630,
    0xa4f0, 0xa60c, 0xa513, 0xa5e8, 0xa537, 0xa5c5, 0xa55a, 0xa5a1, 0xa57e,
    0xa57e, 0xa5a1, 0xa55a, 0xa5c5, 0xa537, 0xa5e8, 0xa513, 0xa60c, 0xa4f0,
    0xa630, 0xa4cc, 0xa654, 0xa4a9, 0xa678, 0xa486, 0xa69c, 0xa463, 0xa6c0,
    0xa440, 0xa6e4, 0xa41d, 0xa708, 0xa3fa, 0xa72c, 0xa3d7, 0xa750, 0xa3b4,
    0xa774, 0xa391, 0xa799, 0xa36f, 0xa7bd, 0xa34c, 0xa7e2, 0xa329, 0xa806,
    0xa307, 0xa82b, 0xa2e4, 0xa84f, 0xa2c2, 0xa874, 0xa29f, 0xa899, 0xa27d,
    0xa8bd, 0xa25b, 0xa8e2, 0xa238, 0xa907, 0xa216, 0xa92c, 0xa1f4, 0xa951,
    0xa1d2, 0xa976, 0xa1b0, 0xa99b, 0xa18e, 0xa9c0, 0xa16c, 0xa9e5, 0xa14a,
    0xaa0a, 0xa129, 0xaa30, 0xa107, 0xaa55, 0xa0e5, 0xaa7a, 0xa0c4, 0xaaa0,
    0xa0a2, 0xaac5, 0xa080, 0xaaeb, 0xa05f, 0xab10, 0xa03e, 0xab36, 0xa01c,
    0xab5c, 0x9ffb, 0xab81, 0x9fda, 0xaba7, 0x9fb9, 0xabcd, 0x9f98, 0xabf3,
    0x9f77, 0xac19, 0x9f56, 0xac3f, 0x9f35, 0xac65, 0x9f14, 0xac8b, 0x9ef3,
    0xacb1, 0x9ed2, 0xacd7, 0x9eb2, 0xacfd, 0x9e91, 0xad24, 0x9e70, 0xad4a,
    0x9e50, 0xad70, 0x9e2f, 0xad97, 0x9e0f, 0xadbd, 0x9def, 0xade4, 0x9dce,
    0xae0b, 0x9dae, 0xae31, 0x9d8e, 0xae58, 0x9d6e, 0xae7f, 0x9d4e, 0xaea5,
    0x9d2e, 0xaecc, 0x9d0e, 0xaef3, 0x9cee, 0xaf1a, 0x9cce, 0xaf41, 0x9caf,
    0xaf68, 0x9c8f, 0xaf8f, 0x9c6f, 0xafb6, 0x9c50, 0xafdd, 0x9c30, 0xb005,
    0x9c11, 0xb02c, 0x9bf1, 0xb053, 0x9bd2, 0xb07b, 0x9bb3, 0xb0a2, 0x9b94,
    0xb0c9, 0x9b75, 0xb0f1, 0x9b55, 0xb118, 0x9b36, 0xb140, 0x9b17, 0xb168,
    0x9af9, 0xb18f, 0x9ada, 0xb1b7, 0x9abb, 0xb1df, 0x9a9c, 0xb207, 0x9a7e,
    0xb22f, 0x9a5f, 0xb257, 0x9a40, 0xb27f, 0x9a22, 0xb2a7, 0x9a04, 0xb2cf,
    0x99e5, 0xb2f7, 0x99c7, 0xb31f, 0x99a9, 0xb347, 0x998b, 0xb36f, 0x996d,
    0xb398, 0x994e, 0xb3c0, 0x9930, 0xb3e9, 0x9913, 0xb411, 0x98f5, 0xb439,
    0x98d7, 0xb462, 0x98b9, 0xb48b, 0x989c, 0xb4b3, 0x987e, 0xb4dc, 0x9860,
    0xb505, 0x9843, 0xb52d, 0x9826, 0xb556, 0x9808, 0xb57f, 0x97eb, 0xb5a8,
    0x97ce, 0xb5d1, 0x97b0, 0xb5fa, 0x9793, 0xb623, 0x9776, 0xb64c, 0x9759,
    0xb675, 0x973c, 0xb69e, 0x9720, 0xb6c7, 0x9703, 0xb6f1, 0x96e6, 0xb71a,
    0x96c9, 0xb743, 0x96ad, 0xb76d, 0x9690, 0xb796, 0x9674, 0xb7c0, 0x9657,
    0xb7e9, 0x963b, 0xb813, 0x961f, 0xb83c, 0x9603, 0xb866, 0x95e6, 0xb890,
    0x95ca, 0xb8b9, 0x95ae, 0xb8e3, 0x9592, 0xb90d, 0x9577, 0xb937, 0x955b,
    0xb961, 0x953f, 0xb98b, 0x9523, 0xb9b5, 0x9508, 0xb9df, 0x94ec, 0xba09,
    0x94d0, 0xba33, 0x94b5, 0xba5d, 0x949a, 0xba87, 0x947e, 0xbab1, 0x9463,
    0xbadc, 0x9448, 0xbb06, 0x942d, 0xbb30, 0x9412, 0xbb5b, 0x93f7, 0xbb85,
    0x93dc, 0xbbb0, 0x93c1, 0xbbda, 0x93a6, 0xbc05, 0x938b, 0xbc2f, 0x9371,
    0xbc5a, 0x9356, 0xbc85, 0x933c, 0xbcaf, 0x9321, 0xbcda, 0x9307, 0xbd05,
    0x92ec, 0xbd30, 0x92d2, 0xbd5b, 0x92b8, 0xbd86, 0x929e, 0xbdb1, 0x9284,
    0xbddc, 0x926a, 0xbe07, 0x9250, 0xbe32, 0x9236, 0xbe5d, 0x921c, 0xbe88,
    0x9202, 0xbeb3, 0x91e9, 0xbedf, 0x91cf, 0xbf0a, 0x91b6, 0xbf35, 0x919c,
    0xbf61, 0x9183, 0xbf8c, 0x9169, 0xbfb8, 0x9150, 0xbfe3, 0x9137, 0xc00f,
    0x911e, 0xc03a, 0x9105, 0xc066, 0x90ec, 0xc091, 0x90d3, 0xc0bd, 0x90ba,
    0xc0e9, 0x90a1, 0xc114, 0x9088, 0xc140, 0x9070, 0xc16c, 0x9057, 0xc198,
    0x903e, 0xc1c4, 0x9026, 0xc1f0, 0x900e, 0xc21c, 0x8ff5, 0xc248, 0x8fdd,
    0xc274, 0x8fc5, 0xc2a0, 0x8fad, 0xc2cc, 0x8f95, 0xc2f8, 0x8f7d, 0xc324,
    0x8f65, 0xc351, 0x8f4d, 0xc37d, 0x8f35, 0xc3a9, 0x8f1d, 0xc3d6, 0x8f06,
    0xc402, 0x8eee, 0xc42e, 0x8ed6, 0xc45b, 0x8ebf, 0xc487, 0x8ea8, 0xc4b4,
    0x8e90, 0xc4e0, 0x8e79, 0xc50d, 0x8e62, 0xc53a, 0x8e4b, 0xc566, 0x8e34,
    0xc593, 0x8e1d, 0xc5c0, 0x8e06, 0xc5ed, 0x8def, 0xc619, 0x8dd8, 0xc646,
    0x8dc1, 0xc673, 0x8dab, 0xc6a0, 0x8d94, 0xc6cd, 0x8d7e, 0xc6fa, 0x8d67,
    0xc727, 0x8d51, 0xc754, 0x8d3b, 0xc781, 0x8d24, 0xc7ae, 0x8d0e, 0xc7db,
    0x8cf8, 0xc809, 0x8ce2, 0xc836, 0x8ccc, 0xc863, 0x8cb6, 0xc890, 0x8ca1,
    0xc8be, 0x8c8b, 0xc8eb, 0x8c75, 0xc918, 0x8c60, 0xc946, 0x8c4a, 0xc973,
    0x8c35, 0xc9a1, 0x8c1f, 0xc9ce, 0x8c0a, 0xc9fc, 0x8bf5, 0xca29, 0x8bdf,
    0xca57, 0x8bca, 0xca85, 0x8bb5, 0xcab2, 0x8ba0, 0xcae0, 0x8b8b, 0xcb0e,
    0x8b77, 0xcb3c, 0x8b62, 0xcb69, 0x8b4d, 0xcb97, 0x8b39, 0xcbc5, 0x8b24,
    0xcbf3, 0x8b10, 0xcc21, 0x8afb, 0xcc4f, 0x8ae7, 0xcc7d, 0x8ad3, 0xccab,
    0x8abe, 0xccd9, 0x8aaa, 0xcd07, 0x8a96, 0xcd35, 0x8a82, 0xcd63, 0x8a6e,
    0xcd92, 0x8a5a, 0xcdc0, 0x8a47, 0xcdee, 0x8a33, 0xce1c, 0x8a1f, 0xce4b,
    0x8a0c, 0xce79, 0x89f8, 0xcea7, 0x89e5, 0xced6, 0x89d2, 0xcf04, 0x89be,
    0xcf33, 0x89ab, 0xcf61, 0x8998, 0xcf90, 0x8985, 0xcfbe, 0x8972, 0xcfed,
    0x895f, 0xd01b, 0x894c, 0xd04a, 0x8939, 0xd079, 0x8927, 0xd0a7, 0x8914,
    0xd0d6, 0x8902, 0xd105, 0x88ef, 0xd134, 0x88dd, 0xd162, 0x88ca, 0xd191,
    0x88b8, 0xd1c0, 0x88a6, 0xd1ef, 0x8894, 0xd21e, 0x8882, 0xd24d, 0x8870,
    0xd27c, 0x885e, 0xd2ab, 0x884c, 0xd2da, 0x883a, 0xd309, 0x8828, 0xd338,
    0x8817, 0xd367, 0x8805, 0xd396, 0x87f4, 0xd3c5, 0x87e2, 0xd3f4, 0x87d1,
    0xd424, 0x87c0, 0xd453, 0x87af, 0xd482, 0x879d, 0xd4b1, 0x878c, 0xd4e1,
    0x877b, 0xd510, 0x876b, 0xd53f, 0x875a, 0xd56f, 0x8749, 0xd59e, 0x8738,
    0xd5ce, 0x8728, 0xd5fd, 0x8717, 0xd62d, 0x8707, 0xd65c, 0x86f6, 0xd68c,
    0x86e6, 0xd6bb, 0x86d6, 0xd6eb, 0x86c6, 0xd71b, 0x86b6, 0xd74a, 0x86a5,
    0xd77a, 0x8696, 0xd7aa, 0x8686, 0xd7d9, 0x8676, 0xd809, 0x8666, 0xd839,
    0x8656, 0xd869, 0x8647, 0xd898, 0x8637, 0xd8c8, 0x8628, 0xd8f8, 0x8619,
    0xd928, 0x8609, 0xd958, 0x85fa, 0xd988, 0x85eb, 0xd9b8, 0x85dc, 0xd9e8,
    0x85cd, 0xda18, 0x85be, 0xda48, 0x85af, 0xda78, 0x85a0, 0xdaa8, 0x8592,
    0xdad8, 0x8583, 0xdb08, 0x8574, 0xdb38, 0x8566, 0xdb68, 0x8558, 0xdb99,
    0x8549, 0xdbc9, 0x853b, 0xdbf9, 0x852d, 0xdc29, 0x851f, 0xdc59, 0x8511,
    0xdc8a, 0x8503, 0xdcba, 0x84f5, 0xdcea, 0x84e7, 0xdd1b, 0x84d9, 0xdd4b,
    0x84cc, 0xdd7c, 0x84be, 0xddac, 0x84b0, 0xdddc, 0x84a3, 0xde0d, 0x8496,
    0xde3d, 0x8488, 0xde6e, 0x847b, 0xde9e, 0x846e, 0xdecf, 0x8461, 0xdeff,
    0x8454, 0xdf30, 0x8447, 0xdf61, 0x843a, 0xdf91, 0x842d, 0xdfc2, 0x8421,
    0xdff2, 0x8414, 0xe023, 0x8407, 0xe054, 0x83fb, 0xe085, 0x83ef, 0xe0b5,
    0x83e2, 0xe0e6, 0x83d6, 0xe117, 0x83ca, 0xe148, 0x83be, 0xe178, 0x83b2,
    0xe1a9, 0x83a6, 0xe1da, 0x839a, 0xe20b, 0x838e, 0xe23c, 0x8382, 0xe26d,
    0x8377, 0xe29e, 0x836b, 0xe2cf, 0x8360, 0xe2ff, 0x8354, 0xe330, 0x8349,
    0xe361, 0x833e, 0xe392, 0x8332, 0xe3c3, 0x8327, 0xe3f4, 0x831c, 0xe426,
    0x8311, 0xe457, 0x8306, 0xe488, 0x82fb, 0xe4b9, 0x82f1, 0xe4ea, 0x82e6,
    0xe51b, 0x82db, 0xe54c, 0x82d1, 0xe57d, 0x82c6, 0xe5af, 0x82bc, 0xe5e0,
    0x82b2, 0xe611, 0x82a8, 0xe642, 0x829d, 0xe673, 0x8293, 0xe6a5, 0x8289,
    0xe6d6, 0x827f, 0xe707, 0x8276, 0xe739, 0x826c, 0xe76a, 0x8262, 0xe79b,
    0x8259, 0xe7cd, 0x824f, 0xe7fe, 0x8246, 0xe82f, 0x823c, 0xe861, 0x8233,
    0xe892, 0x822a, 0xe8c4, 0x8220, 0xe8f5, 0x8217, 0xe926, 0x820e, 0xe958,
    0x8205, 0xe989, 0x81fd, 0xe9bb, 0x81f4, 0xe9ec, 0x81eb, 0xea1e, 0x81e2,
    0xea4f, 0x81da, 0xea81, 0x81d1, 0xeab3, 0x81c9, 0xeae4, 0x81c1, 0xeb16,
    0x81b8, 0xeb47, 0x81b0, 0xeb79, 0x81a8, 0xebab, 0x81a0, 0xebdc, 0x8198,
    0xec0e, 0x8190, 0xec3f, 0x8188, 0xec71, 0x8181, 0xeca3, 0x8179, 0xecd5,
    0x8172, 0xed06, 0x816a, 0xed38, 0x8163, 0xed6a, 0x815b, 0xed9b, 0x8154,
    0xedcd, 0x814d, 0xedff, 0x8146, 0xee31, 0x813f, 0xee62, 0x8138, 0xee94,
    0x8131, 0xeec6, 0x812a, 0xeef8, 0x8123, 0xef2a, 0x811d, 0xef5c, 0x8116,
    0xef8d, 0x8110, 0xefbf, 0x8109, 0xeff1, 0x8103, 0xf023, 0x80fd, 0xf055,
    0x80f6, 0xf087, 0x80f0, 0xf0b9, 0x80ea, 0xf0eb, 0x80e4, 0xf11c, 0x80de,
    0xf14e, 0x80d9, 0xf180, 0x80d3, 0xf1b2, 0x80cd, 0xf1e4, 0x80c8, 0xf216,
    0x80c2, 0xf248, 0x80bd, 0xf27a, 0x80b7, 0xf2ac, 0x80b2, 0xf2de, 0x80ad,
    0xf310, 0x80a8, 0xf342, 0x80a3, 0xf374, 0x809e, 0xf3a6, 0x8099, 0xf3d8,
    0x8094, 0xf40a, 0x808f, 0xf43c, 0x808b, 0xf46e, 0x8086, 0xf4a0, 0x8082,
    0xf4d3, 0x807d, 0xf505, 0x8079, 0xf537, 0x8075, 0xf569, 0x8070, 0xf59b,
    0x806c, 0xf5cd, 0x8068, 0xf5ff, 0x8064, 0xf631, 0x8060, 0xf663, 0x805d,
    0xf695, 0x8059, 0xf6c8, 0x8055, 0xf6fa, 0x8052, 0xf72c, 0x804e, 0xf75e,
    0x804b, 0xf790, 0x8047, 0xf7c2, 0x8044, 0xf7f4, 0x8041, 0xf827, 0x803e,
    0xf859, 0x803b, 0xf88b, 0x8038, 0xf8bd, 0x8035, 0xf8ef, 0x8032, 0xf922,
    0x802f, 0xf954, 0x802d, 0xf986, 0x802a, 0xf9b8, 0x8027, 0xf9ea, 0x8025,
    0xfa1d, 0x8023, 0xfa4f, 0x8020, 0xfa81, 0x801e, 0xfab3, 0x801c, 0xfae5,
    0x801a, 0xfb18, 0x8018, 0xfb4a, 0x8016, 0xfb7c, 0x8014, 0xfbae, 0x8013,
    0xfbe1, 0x8011, 0xfc13, 0x800f, 0xfc45, 0x800e, 0xfc77, 0x800c, 0xfcaa,
    0x800b, 0xfcdc, 0x800a, 0xfd0e, 0x8009, 0xfd40, 0x8008, 0xfd73, 0x8007,
    0xfda5, 0x8006, 0xfdd7, 0x8005, 0xfe09, 0x8004, 0xfe3c, 0x8003, 0xfe6e,
    0x8002, 0xfea0, 0x8002, 0xfed2, 0x8001, 0xff05, 0x8001, 0xff37, 0x8001,
    0xff69, 0x8000, 0xff9b, 0x8000, 0xffce, 0x8000,
};

/**
 * @} end of CFFT_CIFFT group
 */

/*
 * @brief  Q15 table for reciprocal
 */
const q15_t ALIGN4 armRecipTableQ15[64] = {
    0x7F03, 0x7D13, 0x7B31, 0x795E, 0x7798, 0x75E0, 0x7434, 0x7294,
    0x70FF, 0x6F76, 0x6DF6, 0x6C82, 0x6B16, 0x69B5, 0x685C, 0x670C,
    0x65C4, 0x6484, 0x634C, 0x621C, 0x60F3, 0x5FD0, 0x5EB5, 0x5DA0,
    0x5C91, 0x5B88, 0x5A85, 0x5988, 0x5890, 0x579E, 0x56B0, 0x55C8,
    0x54E4, 0x5405, 0x532B, 0x5255, 0x5183, 0x50B6, 0x4FEC, 0x4F26,
    0x4E64, 0x4DA6, 0x4CEC, 0x4C34, 0x4B81, 0x4AD0, 0x4A23, 0x4978,
    0x48D1, 0x482D, 0x478C, 0x46ED, 0x4651, 0x45B8, 0x4521, 0x448D,
    0x43FC, 0x436C, 0x42DF, 0x4255, 0x41CC, 0x4146, 0x40C2, 0x4040};

/*
 * @brief  Q31 table for reciprocal
 */
const q31_t armRecipTableQ31[64] = {
    0x7F03F03F, 0x7D137420, 0x7B31E739, 0x795E9F94, 0x7798FD29, 0x75E06928,
    0x7434554D, 0x72943B4B, 0x70FF9C40, 0x6F760031, 0x6DF6F593, 0x6C8210E3,
    0x6B16EC3A, 0x69B526F6, 0x685C655F, 0x670C505D, 0x65C4952D, 0x6484E519,
    0x634CF53E, 0x621C7E4F, 0x60F33C61, 0x5FD0EEB3, 0x5EB55785, 0x5DA03BEB,
    0x5C9163A1, 0x5B8898E6, 0x5A85A85A, 0x598860DF, 0x58909373, 0x579E1318,
    0x56B0B4B8, 0x55C84F0B, 0x54E4BA80, 0x5405D124, 0x532B6E8F, 0x52556FD0,
    0x5183B35A, 0x50B618F3, 0x4FEC81A2, 0x4F26CFA2, 0x4E64E64E, 0x4DA6AA1D,
    0x4CEC008B, 0x4C34D010, 0x4B810016, 0x4AD078EF, 0x4A2323C4, 0x4978EA96,
    0x48D1B827, 0x482D77FE, 0x478C1657, 0x46ED801D, 0x4651A2E5, 0x45B86CE2,
    0x4521CCE1, 0x448DB244, 0x43FC0CFA, 0x436CCD78, 0x42DFE4B4, 0x42554426,
    0x41CCDDB6, 0x4146A3C6, 0x40C28923, 0x40408102};
