//=============================================================================================
// Fledermausdetektor Surasto.de -- 2022
// 
// Hardware: Feather M0 Proto Board with ARM M0 und 350ksps ADC
//           MEMS Mikrofon (ELV MEMS-1)  Knowles SPU0410LR5H 
//           60dB Amplification with TS912 Rail-to-Rail OpAmp
//           
// Software: Adafruit ADC-DMA Functions (see below)
//           Adafruit FFT Library
//              
// Nach http://www.atmel.com/Images/Atmel-42258-ASF-Manual-SAM-D21_AP-Note_AT07627.pdf pg 73
// 
// Sampling Rate = ~196ksps
// Bins: Bin[1024] = 93.77kHz;
// Conversion from binNr to Frequency: f(binNr) = 250 / HWORDS * binNr    [in kHz]
//==============================================================================================

#define ADC_PIN A4

DmacDescriptor *dmac_descriptor_1;
DmacDescriptor *dmac_descriptor_2;

uint16_t adc_buffer[SAMPLE_BLOCK_LENGTH * 2];
int16_t adc_sample_block[SAMPLE_BLOCK_LENGTH];     // was uint16_t in original code
volatile bool filling_first_half = true;
volatile uint16_t *active_adc_buffer;
volatile bool adc_buffer_filled = false;

void dma_callback(Adafruit_ZeroDMA *dma) {
  (void) dma;
  if (filling_first_half) {
    // DMA is filling the first half of the buffer, use data from the second half
    active_adc_buffer = &adc_buffer[SAMPLE_BLOCK_LENGTH];
    filling_first_half = false;
  } else {
    // DMA is filling the second half of the buffer, use data from the first half
    active_adc_buffer = &adc_buffer[0];
    filling_first_half = true;
  }
  adc_buffer_filled = true;
}


static void ADCsync() {
  while (ADC->STATUS.bit.SYNCBUSY == 1);
}


void adc_init() {
  analogRead(ADC_PIN);
  ADC->CTRLA.bit.ENABLE = 0;
  ADCsync();
//  ADC->INPUTCTRL.bit.GAIN = ADC_INPUTCTRL_GAIN_DIV2;
  ADC->INPUTCTRL.bit.GAIN = ADC_INPUTCTRL_GAIN_DIV2_Val;
  ADC->REFCTRL.bit.REFSEL = ADC_REFCTRL_REFSEL_INTVCC1;
  ADCsync();
  ADC->INPUTCTRL.bit.MUXPOS = g_APinDescription[ADC_PIN].ulADCChannelNumber;
  ADCsync();
  ADC->AVGCTRL.reg = 0;
  ADC->SAMPCTRL.reg = 2;
  ADCsync();
  ADC->CTRLB.reg = ADC_CTRLB_PRESCALER_DIV32 | ADC_CTRLB_FREERUN | ADC_CTRLB_RESSEL_12BIT;
//  ADC->CTRLB.reg = ADC_CTRLB_PRESCALER_DIV16 | ADC_CTRLB_FREERUN | ADC_CTRLB_RESSEL_12BIT;
  ADCsync();
  ADC->CTRLA.bit.ENABLE = 1;
  ADCsync();
}

void dma_init(){
  ADC_DMA.allocate();
  ADC_DMA.setTrigger(ADC_DMAC_ID_RESRDY);
  ADC_DMA.setAction(DMA_TRIGGER_ACTON_BEAT);
  dmac_descriptor_1 = ADC_DMA.addDescriptor(
           (void *)(&ADC->RESULT.reg),
           adc_buffer,
           SAMPLE_BLOCK_LENGTH,
           DMA_BEAT_SIZE_HWORD,
           false,
           true);
  dmac_descriptor_1->BTCTRL.bit.BLOCKACT = DMA_BLOCK_ACTION_INT;

  dmac_descriptor_2 = ADC_DMA.addDescriptor(
           (void *)(&ADC->RESULT.reg),
           adc_buffer + SAMPLE_BLOCK_LENGTH,
           SAMPLE_BLOCK_LENGTH,
           DMA_BEAT_SIZE_HWORD,
           false,
           true);
  dmac_descriptor_2->BTCTRL.bit.BLOCKACT = DMA_BLOCK_ACTION_INT;

  ADC_DMA.loop(true);
  ADC_DMA.setCallback(dma_callback);
}

//================================================================================
// This function has to be called by loop() 
// Once the DMA Buffer is full, its content gets copied to an array
// Further the FFT is performed and added to the global ffft_signal_averaged array
//================================================================================
void dma_polling() {
  long int signal_total;
  int average;
  
  if(adc_buffer_filled){
    adc_buffer_filled=false;
    memcpy(adc_sample_block, (const void*)active_adc_buffer, SAMPLE_BLOCK_LENGTH*sizeof(uint16_t));

    //=============== Eliminate the DC Offset =======================
    signal_total = 0;
    for (int i=0; i<SAMPLE_BLOCK_LENGTH; i++) signal_total+=adc_sample_block[i];
    average = signal_total / SAMPLE_BLOCK_LENGTH;  
    for (int i=0; i<SAMPLE_BLOCK_LENGTH; i++) adc_sample_block[i] = adc_sample_block[i]-average;

    //================ FFT ========================================
    ZeroFFT(adc_sample_block, SAMPLE_BLOCK_LENGTH);

    //====== Adding to the FFT array (max signal amplitude) =======
    for (int i=0; i<SAMPLE_BLOCK_LENGTH/2; i++) {
       if (adc_sample_block[i] > fft_signal_averaged[i]) {     
          fft_signal_averaged[i] = adc_sample_block[i];
       }
    }
  }
}


//============================================================================
// This function returns the bins in a sorted order 
//============================================================================
void sort_bins(int16_t bin[], int16_t index[]) {
   int16_t temp;
   
   for (int i=0; i<SAMPLE_BLOCK_LENGTH/2; i++) index[i]=i;   // index with sorted bin numbers 0...SAMPLE_BLOCK_LENGTH/2

   //============== bubble sort ====================
   for (int j=0; j<SAMPLE_BLOCK_LENGTH/2; j++) {
     for (int i=0; i<SAMPLE_BLOCK_LENGTH/2-1-j; i++) {
        if (bin[i+1] > bin[i]) {
          temp = bin[i];         // swap bins
          bin[i] = bin[i+1];
          bin[i+1] = temp;   
          temp = index[i];       // swap bin number as well
          index[i] = index[i+1];
          index[i+1] = temp;
        }
     }
   }  
}

//==============================================================================
// sets all bins of the averaged FFT to zero
//==============================================================================
void clear_fft_average() {

   for (int i=0; i<SAMPLE_BLOCK_LENGTH/2; i++) fft_signal_averaged[i] = 0;

}

//=============================================================================
// Copy the 12 highest bins into the lora send array
//
// Format: [index_high index_low bin_high bin_low] .... [index_high ....
//=============================================================================
void create_ttn_data(uint8_t ttn_data[], int16_t bin[], int16_t index[]) {
  uint8_t bin_low, bin_high;
  uint8_t idx_low, idx_high;
  
  for (int i=0; i<12; i++) {
    bin_low = bin[i] & 0xFF;
    bin_high = bin[i]>> 8;
    idx_low = idx[i] & 0xFF;
    idx_high = idx[i] >> 8;
    ttn_data[i*4] = idx_high;
    ttn_data[i*4+1] = idx_low;
    ttn_data[i*4+2] = bin_high;
    ttn_data[i*4+3] = bin_low;
  }
}
